package sneps3.classes;

import sneps3.exception.*;

/**
 * Public access to a network
 * Gives a bit more power than the other public interfaces.
 * Meant to give most of the powers of SNePSUL.
 *
 * @author SNePS 3 Java Programming Team
 * @version 3.0
 */
public interface Network{
  /**
   * Initializes the network.  Won't work if network is not empty.
   *
   * @return <code>true</code> if the network was initialized,
   *  <code>false</code> otherwise
   */
  public boolean init();
  

  /**
   * Define a semantic class.
   *
   * @param sc the semantic class to add to the network
   * @return <code>true</code> if the class was defined, <code>false</code>
   *  otherwise
   */
  public SemanticClass defineSemanticClass(final SemanticClass sc)
    throws UniqueException, UndefinedException, StructureException;


  /**
   * Create and define a semantic class.
   *
   * @param name the name for the new semantic class
   * @param parent the parent semantic class for the new semantic class
   * @return the newly defined semantic class
   */
  public SemanticClass defineSemanticClass(final String name,
      final String parentName) throws UniqueException, UndefinedException,
      StructureException;


  /**
   * Remove a semantic class.  Only works if nothing refrences it.
   *
   * @param sc the semantic class to be removed
   * @return <code>true</code> if the semantic class was removed,
   *  <code>false</code> otherwise
   */
  public boolean undefineSemanticClass(final String name)
    throws InUseException, StructureException;

  /**
   * Sets associated color for a semantic class.
   *
   * @param name the name of the semantic class to be altered
   * @param color the new color for the semantic class
   */
  public boolean setSemanticClassColor(final String name,
      final java.awt.Color color);

  /**
   * Retrieves a list of references to all semantic classes defined in the
   *  network
   * @return A SemClassSet object of references to all existing semantic
   *  classes
   */
  public SemanticClassSet getSemanticClasses();


  /**
   * Define a relation
   *
   * @param relation The relation to add to the network
   * @return <code>true</code> if ?, <code>false</code>otherwise
   */
  public Relation defineRelation(final Relation relation)
    throws UniqueException, UndefinedException, OutOfBoundsException;

  /**
   * Define a relation.
   */
  public Relation defineRelation(final String name, final String scname,
      final int adjust, final int limit) throws UndefinedException,
      UniqueException, OutOfBoundsException;

  /**
   * Define a relation.
   */
  public Relation defineRelation(final String name, final String scname,
      final String adjust, final int limit) throws UndefinedException,
      UniqueException, OutOfBoundsException;

  /**
   * Define a relation.
   */
  public Relation defineRelation(final String name, final String scname)
    throws UndefinedException, UniqueException;

  /**
   * Undefine a relation
   *
   * @param relationName The name of the relation to undefine
   * @return <code>true</code> if ?, <code>false</code>otherwise
   */
  public boolean undefineRelation(String relationName)
    throws InUseException;


  /**
   * Retrieves a list of references to all relations defined in the network
   *
   * @return A RelationSet object of references to all existing relations
   */
  public RelationSet getRelations();


  public CaseFrame defineCaseFrame(CaseFrame caseFrameToDefine)
    throws UniqueException, UndefinedException;

  public CaseFrame defineCaseFrame(String semClass, String[] rels)
    throws UniqueException, UndefinedException;

  public boolean undefineCaseFrame(String[] rels)
    throws InUseException, UndefinedException;

  public CaseFrameSet getCaseFrames();
  
  /**
   * Removes all nodes from the network 
   */
  public void clearNodes();

  /**
   * Removes all nodes from the network and removes all but the default
   *  context
   */
  public void clearNodesAndContexts();

  /**
   * Removes all nodes from the network, removes all contexts except the
   *  default, and removes all defined semantic classes, relations and
   *  case frames.
   */
  public void clearAll();


  /**
   *  Adds a node to the network
   *  The Node should already be set up to have a CaseFrame,
   *  a cableset pointing to valid nodes, etc.   
   *
   * @param node The node to be added
   * @return <code>true</code> if ?,<code>false</code> otherwise 
   */
  public boolean addNode(final Node node);
  

  /**
   * Pass a reference to a retrieved node, the network deletes the node
   *
   * @param node A reference (retrieved earlier) to the node to delete
   * @return <code>true</code> if ?, <code>false</code> otherwise
   */
  public boolean deleteNode(Node node);


  /**
   * Find a node that fits the description of the given node exactly, i.e.
   * points to same CaseFrame, has same set of Cables where each cable has
   * the same case frame and has cablesets containing the same set of nodes
   * This will be used to assert existing nodes, avoid rebuilding nodes, etc
   *
   * @param node The node to find
   * @return The node if it was found, otherwise 'null'
   */
  public Node findWholeNode(final Node node);


  /**
   * Returns a list of all nodes in the network
   * @return A Nodeset containing all nodes in the network
   */
  public NodeSet getNodes();


  /**
   * Returns a list of all nodes currently believed in the specified context
   * @param A string containing the name of the context of interest
   * @return A Nodeset containing all nodes believed in the specified context
   */
  public NodeSet getNodes(final String contextName);


  /**
   * Retrieve a ContextSet of references to all existing contexts
   * @return A ContextSet object containing references to all existing contexts
   */
  public ContextSet getContexts();


  /**
   * Add a context to the current network
   *
   * @param context The context to add
   * @return <code>true</code> if ?, <code>false</code>otherwise
   */
  public Context addContext(final String name) throws UniqueException;


  /**
   * Remove a context from the network
   *
   * @param contextName The name of the context to remove
   * @return <code>true</code> if ?, <code>false</code>otherwise
   */
  public Context removeContext(final String contextName);


  /**
   * Set the default context to be 'contextName'
   *
   * @param contextName the name of the context to set as default
   * @return <code>true</code> if ?, <code>false</code>otherwise
   */
  public boolean setDefaultContext(String contextName);
  

  /**
   * Get the name of the default context
   *
   * @return A string containing the name of the current default context
   */
  public String getDefaultContext();


  /**
   * Add the node 'node' to the context 'contextName'
   *
   * @param node The node to add to the context
   * @param contextName The name of the context to which to add the node
   * @return <code>true</code> if ?, <code>false</code>otherwise
   */
  public boolean believeNode(Node node, String contextName);


  /**
   * Remove the node 'node' from the context 'contextName'
   *
   * @param node The node to remove from the context
   * @param contextName The name of the context from which to remove the node
   * @return <code>true</code> if ?, <code>false</code>otherwise
   */
  public boolean disbelieveNode(Node node, String contextName);


  /**
   * Clears out (removes) all defined relations
   */
  public void clearRelations();
  
  public void addNetworkListener(NetworkListener l);
  
  public void removeNetworkListener(NetworkListener l);

  /**
   * This method returns a printable string representation of the network
   */
  public String toString();   

  public Context findContextByName(String contextName);

  public Relation findRelationByName(String relationName);

  public SemanticClass findSemanticClassByName(String semClassName);

}
