package sneps3.corecode;

import java.util.Iterator;
import java.util.Vector;
import sneps3.exception.*;
import sneps3.classes.*;
import sneps3.event.*;

class S3_Network implements sneps3.classes.Network
{

  NodeSet _nodes;
  ContextSet _contexts;
  CaseFrameSet _caseFrames;
  RelationSet _relations;
  SemanticClassSet _semClasses;
  Context _defaultContext;
  javax.swing.event.EventListenerList _listeners;

  /**
   * Class constructor.
   *
   * @return a new instance of this class
   */
  public S3_Network()
  {
    _semClasses = null;
    _relations = null;
    _caseFrames = null;
    _contexts = null;
    _nodes = null;
    _defaultContext = null;
    _listeners = new javax.swing.event.EventListenerList();
  }

	public void addNetworkListener(NetworkListener l){
		_listeners.add(NetworkListener.class, l);
	}

	public void removeNetworkListener(NetworkListener l){
		_listeners.remove(NetworkListener.class, l);
	}

  public boolean init(){
    if (_semClasses!=null)
      return (false);

    _semClasses = Defaults.FACTORY.createSemanticClassSet();
    Iterator sci = Defaults.SEMANTIC_CLASSES.iterator();
    while (sci.hasNext()){
      SemanticClass node = (SemanticClass)sci.next();
      _semClasses.add(node);
      if (node.getParent()!=null)
        ((SemanticClassSet)node.getParent().getChildren()).add(node);
    }

    _relations = Defaults.FACTORY.createRelationSet();
    Iterator ri = Defaults.RELATIONS.iterator();
    while (ri.hasNext()){
      Relation r = (Relation)ri.next();
      _relations.add(r);
      ((RelationSet)r.getType().getRelations()).add(r);
    }

    _caseFrames = Defaults.FACTORY.createCaseFrameSet();
    Iterator cfi = Defaults.CASE_FRAMES.iterator();
    while (cfi.hasNext()){
      CaseFrame cf = (CaseFrame)cfi.next();
      _caseFrames.add(cf);
      Iterator ri2 = cf.getRelations().iterator();
      while (ri2.hasNext())
        ((CaseFrameSet)((Relation)ri2.next()).getCaseFrames()).add(cf);
    }

    _contexts = Defaults.FACTORY.createContextSet();
    Iterator cti = Defaults.CONTEXTS.iterator();
    while (cti.hasNext()){
      Context ct = (Context)cti.next();
      _contexts.add(ct);
    }
    _defaultContext = Defaults.DEFAULT_CT;

    return (true);
  }

  public sneps3.classes.SemanticClass defineSemanticClass(
      sneps3.classes.SemanticClass sc)
    throws UniqueException, UndefinedException, StructureException
  {
    SemanticClass sc2 = (SemanticClass)sc;
    if (sc2.getParent()==null){
      if (!_semClasses.isEmpty())
        throw new StructureException("Only the first semantic class may " +
            "be declared as the root class");
      _semClasses.add(sc2);
      return (sc2);
    }
    if (!_semClasses.contains(sc2.getParent().getName()))
      throw new UndefinedException(sc2);
    if (!_semClasses.add(sc2))
      throw new UniqueException(sc2,_semClasses);
    ((SemanticClassSet)sc2.getParent().getChildren()).add(sc2);
    sc2.setColor(sc2.getParent().getColor());
    return (sc2);
  }

  public sneps3.classes.SemanticClass defineSemanticClass(final String name,
      final String parentName) throws UniqueException, UndefinedException,
      StructureException
  {
    sneps3.classes.SemanticClass parent;

    if (_semClasses.isEmpty())
      if (parentName!=null)
        throw new StructureException("Top-level semantic class must be " +
            "declared with its parent string equal to null.");
      else
        parent = null;
    else
      if (parentName==null || parentName.length()==0)
        throw new StructureException("Only the top-level semantic class may " +
            "be declared with its parent string equal to null.");
      else{
        parent = _semClasses.get(parentName);
        if (parent==null)
          throw new UndefinedException(Defaults.FACTORY.createSemanticClass(
            parentName,null));
      }
    sneps3.classes.SemanticClass sc =
      Defaults.FACTORY.createSemanticClass(name,parent);
    defineSemanticClass(sc);
    return (sc);
  }

  public boolean undefineSemanticClass(final String name)
    throws InUseException, StructureException{

    sneps3.classes.SemanticClass sc = _semClasses.get(name);

    if (sc==null)
      return (false);

    if (!sc.getChildren().isEmpty())
      throw new StructureException("Cannot undefine a semantic class that " +
          "has descendent semantic classes.");

    if (!sc.getRelations().isEmpty())
      throw new InUseException(sc, sc.getRelations());

    SemanticClass parent = (SemanticClass)sc.getParent();
    if (parent!=null)
      ((SemanticClassSet)parent.getChildren()).remove(name);

    _semClasses.remove(name);
    return (true);
  }

  public boolean setSemanticClassColor(final String name,
      final java.awt.Color color){
    SemanticClass sc = (SemanticClass)_semClasses.get(name);
    if (sc == null)
      return (false);
    sc.setColor(color);
    return (true);
  }

  public sneps3.classes.SemanticClassSet getSemanticClasses()
  {
    return (_semClasses);
  }

  public sneps3.classes.Relation defineRelation(sneps3.classes.Relation
      relation)
    throws UniqueException, UndefinedException, OutOfBoundsException
  {
    Relation r = (Relation)relation;
    if (_relations.contains(r.getName()))
      throw new UniqueException(r, _relations);
    if (!_semClasses.contains(r.getType().getName()))
      throw new UndefinedException(r.getType());
    if (r.getAdjust()<-1 ||r.getAdjust()>1)
      throw new OutOfBoundsException(relation);
    ((RelationSet)r.getType().getRelations()).add(r);
    _relations.add(r);
    return (r);
  }

  public sneps3.classes.Relation defineRelation(final String name,
      final String scname, final int adjust, final int limit)
    throws UndefinedException, UniqueException, OutOfBoundsException {
    sneps3.classes.SemanticClass sc = _semClasses.get(scname);
    if (sc==null)
      throw new UndefinedException(
        Defaults.FACTORY.createSemanticClass(scname,null));
    
    return(defineRelation(
      Defaults.FACTORY.createRelation(name,sc,adjust,limit)));  
  }

  public sneps3.classes.Relation defineRelation(final String name,
      final String scname, final String adjust, final int limit)
    throws OutOfBoundsException, UndefinedException, UniqueException{
    int a;
    if (adjust.compareToIgnoreCase("reduce")==0)
      a=-1;
    else if (adjust.compareToIgnoreCase("expand")==0)
      a=1;
    else if (adjust.compareToIgnoreCase("none")==0)
      a=0;
    else
      throw new OutOfBoundsException(adjust);

    return (defineRelation(name, scname, a, limit));
  }

  public sneps3.classes.Relation defineRelation(final String name,
      final String scname)
    throws UndefinedException, UniqueException{
    try{
      return (defineRelation(name,scname,Defaults.ADJUST_MODE,1));
    }
    catch (OutOfBoundsException e){
      // should not happen if ADJUST_MODE is a valid value;
    }
    return (null);
  }

  public boolean undefineRelation(final String name)
    throws InUseException
  {
    sneps3.classes.Relation r = _relations.get(name);
    if (r==null)
      return (false);

    if (!r.getCaseFrames().isEmpty())
      throw new InUseException(r, r.getCaseFrames());
    if (!r.getPaths().isEmpty())
      throw new InUseException(r, r.getPath());

    if (!Defaults.CASE_SENSITIVE)
      _relations.remove(name.toUpperCase());
    else
      _relations.remove(name);

    ((RelationSet)r.getType().getRelations()).remove(name);
      
    return (r!=null);
  }

  public sneps3.classes.RelationSet getRelations()
  {
    return (_relations);
  }

  public sneps3.classes.CaseFrame defineCaseFrame(sneps3.classes.CaseFrame cf)
    throws UniqueException, UndefinedException {
    if (_caseFrames.contains(cf))
      throw new UniqueException(cf, _caseFrames);
    Iterator ri = cf.getRelations().iterator();
    while (ri.hasNext()){
      Relation r = (Relation)ri.next();
      if (!_relations.contains(r.getName()))
        throw new UndefinedException(r);
    }
    if(_caseFrames.add(cf)){
      ri = cf.getRelations().iterator();
      while(ri.hasNext())
        ((CaseFrameSet)((Relation)ri.next()).getCaseFrames()).add(cf);
      return (cf);
    }
    else
      return (null);
  }

  public sneps3.classes.CaseFrame defineCaseFrame(final String semClass,
      final String[] rels) throws UniqueException, UndefinedException{
    sneps3.classes.SemanticClass sc = _semClasses.get(semClass);
    if (sc==null)
      throw new UndefinedException(Defaults.FACTORY.createSemanticClass(
        semClass, null));
    RelationSet rs = Defaults.FACTORY.createRelationSet();
    for(int i=0; i<java.lang.reflect.Array.getLength(rels); i++){
      sneps3.classes.Relation temp = _relations.get(rels[i]);
      if (temp==null)
        throw new UndefinedException(Defaults.FACTORY.createRelation(rels[i],
            null, 0, -1));
      rs.add((Relation)temp);
    }
    CaseFrame cf = Defaults.FACTORY.createCaseFrame(sc, rs);
    return (defineCaseFrame(cf));
  }

  public boolean undefineCaseFrame(String[] rels)
    throws InUseException, UndefinedException{
    RelationSet rs = Defaults.FACTORY.createRelationSet();
    for (int i=0; i<java.lang.reflect.Array.getLength(rels); i++)
      rs.add((Relation)_relations.get(rels[i]));
    CaseFrame cf = _caseFrames.remove(rs);
    if (cf == null)
      throw new UndefinedException();
    else if (!cf.getNodes().isEmpty())
      throw new InUseException(cf,_nodes);
    else{
      Iterator ri = cf.getRelations().iterator();
      while (ri.hasNext())
        ((CaseFrameSet)
            ((Relation)ri.next()).getCaseFrames()).remove(cf.getRelations());
      return (true);
    }
  }

  public sneps3.classes.CaseFrameSet getCaseFrames(){
    return (_caseFrames);
  }







  public void clearNodes()
  {
    _nodes.clear();
  }

  public void clearNodesAndContexts()
  {
    _nodes.clear();
    _contexts.clear();
  }

  public void clearAll()
  {
    this.clearNodesAndContexts();
    _caseFrames.clear();
    _relations.clear();
    _semClasses.clear();
  }

  public boolean addNode(sneps3.classes.Node node)
  {
    return _nodes.add(node);
  }
  
  public boolean deleteNode(sneps3.classes.Node node)
  {
    if (!_nodes.remove(node))
      return false;
    Iterator ci = _contexts.iterator();
    while (ci.hasNext())
      ((Context)ci.next()).removeNode(node);
    return true;
  }

  public sneps3.classes.Node findWholeNode(sneps3.classes.Node node)
  {
    Node n;
    java.util.Iterator ni;

    // check if any node in the network match <node> by WIRE-BASED INFERENCE.
    // if yes, return the first one matching.
    ni = getNodes().iterator();
    while( ni.hasNext()){
      	n = (Node)ni.next();
      	if( n instanceof S3_BaseNode && node instanceof S3_BaseNode){
      	    if( n.deepEquals(node) ) return n;
     	}
      	else { 
	    // Notice that implementation below is very ineffecient now. 
	    // Also notice that we can not use <deepEquals()> function here
	    //  because we can't just check name or reference for equality.
          if(n instanceof S3_MolecularNode && node instanceof S3_MolecularNode){
            if( ((MolecularNode)n).isAdjustableTo((MolecularNode)node) &&
                ((MolecularNode)node).isAdjustableTo((MolecularNode)n) ) 
              return n;
      	    }
      	}
    }
      
    // if nothing is found matching <node>, return <null>.
    return null;
  }

  public sneps3.classes.NodeSet getNodes()
  {
    return (_nodes);
  }

  public sneps3.classes.NodeSet getNodes(String contextName)
  {
    Context c = (Context)findContextByName(contextName);
     if (c == null)
       return null;
     else
       return c.getNodes();
  }

  public sneps3.classes.ContextSet getContexts()
  {
    return (_contexts);
  }

  public sneps3.classes.Context addContext(final String name)
    throws UniqueException{
    Context ret = Defaults.FACTORY.createContext(name);
    if (!_contexts.add(ret))
      throw new UniqueException(ret,_contexts);
    return (ret);
  }
  
  public sneps3.classes.Context findContextByName(String contextName)
  {
    Context c = null;
    Iterator i = _contexts.iterator();
    if (i.hasNext())
      {
        for (c = (Context)i.next();
             (!c.getName().equals(contextName)) && i.hasNext();
             c = (Context)i.next());
        if (!c.getName().equals(contextName))
	   c = null;
      }
    return c;
  }

  public sneps3.classes.Context removeContext(String contextName)
  {
    return(_contexts.remove(contextName));
  }

  public boolean setDefaultContext(String contextName)
  {
    Context c = (Context)findContextByName(contextName);
    if (c != null)
      {
	_defaultContext = c;
	return true;
      }
    return false;
  }
  
  public String getDefaultContext()
  {
    return _defaultContext.getName();    
  }

  public boolean believeNode(sneps3.classes.Node node, String contextName)
  {
    Context c = (Context)findContextByName(contextName);
    if (c != null)
      {
	c.addNode(node);
	return true;
      }
    return false;
  }

  public boolean disbelieveNode(sneps3.classes.Node node, String contextName)
  {
    Context c = (Context)findContextByName(contextName);
    if (c != null)
      return c.removeNode(node);
    else
      return false;
  }

  public sneps3.classes.Relation findRelationByName(String relationName)
  {
    Relation r = null;
    java.util.Iterator i = _relations.iterator();
    if (i.hasNext())
      {
	 for (r = (Relation)i.next();
              (!r.getName().equals(relationName)) && i.hasNext();
	      r = (Relation)i.next());
	 if (!r.getName().equals(relationName))
	    r = null;
      }
    return r;
  }

  public void clearRelations()
  {
    _relations.clear();
  }

  public sneps3.classes.SemanticClass findSemanticClassByName(String
      semClassName)
  {
    SemanticClass s = null;
    java.util.Iterator i = _semClasses.iterator();
    if (i.hasNext())
      {
	 for (s = (SemanticClass)i.next();
              (!s.getName().equals(semClassName)) && i.hasNext();
	      s = (SemanticClass)i.next());
	 if (!s.getName().equals(semClassName))
	    s = null;
      }
    return s;

  }

  public String toString()
  {
     String retVal = "<<Network>>\n";

     retVal += "---------\n[Semantic Classes]\n";     
     java.util.Iterator si = _semClasses.iterator();
     while (si.hasNext())
	retVal += ((SemanticClass)si.next()).toString() + "\n";

     retVal += "---------\n[Relations]\n";
     java.util.Iterator ri = _relations.iterator();
     while (ri.hasNext())
	retVal += ((Relation)ri.next()).toString();     

     retVal += "---------\n[Case Frames]\n";     
     java.util.Iterator cfi = _caseFrames.iterator();
     while (cfi.hasNext())
	retVal += ((CaseFrame)cfi.next()).toString();     

     retVal += "---------\n[Nodes]\n";     
     java.util.Iterator ni = _nodes.iterator();
     while (ni.hasNext())
	retVal += ((Node)ni.next()).toString();     

     retVal += "---------\n[Contexts]\n";     
     Iterator ci = _contexts.iterator();
     while (ci.hasNext()) 
			{
				Context thisOne = (Context)ci.next();
	 			retVal += thisOne.toString();
	 			if (_defaultContext.getName().equals(thisOne.getName()))
	   			retVal += "^^ DEFAULT ^^\n";
			}
    
    return retVal;
  }
  
  
  void fireNetworkEvent(NetworkEvent e){
  	Object[] listeners = _listeners.getListenerList();
		for (int i = listeners.length-2; i>=0; i-=2){
			if (listeners[i]==NetworkListener.class)
				switch (e.getType()){
					case NetworkEvent.ADD:
						((NetworkListener)listeners[i+1]).addEvent(e);
					break;
					case NetworkEvent.DELETE:
						((NetworkListener)listeners[i+1]).removeEvent(e);
					break;
					case NetworkEvent.CHANGE:
						((NetworkListener)listeners[i+1]).changeEvent(e);
					break;
					default:
					break;
				}
		}
  }

}
