package sneps3.corecode;
import java.io.*;

/**
 * A simple implementation for nodes.  Specific syntatic types should each
 * have their own class, which should extend this class.
 *
 * @author SNePS 3 Java Programming Team
 * @version 3.0
 */
abstract class S3_Node implements Node, Serializable, Cloneable, Comparable{

  /**
   * A count of the number of nodes instantiated.
   *
   * @serial
   */
  protected static int _count = 0;

  /**
   * The name of the node.
   *
   * @serial
   */
  protected String _name;

  /**
   * The semantic class the node belongs to.
   *
   * @serial
   */
  protected SemanticClass _semClass;

  /**
   * Unique identifier for each instance.
   *
   * @serial
   */
  protected int _id;

  /**
   * Contexts node is asserted in.
   *
   * @serial
   */
  protected ContextSet _assertedIn;

  /**
   * The set of cables pointing at this node.
   *
   * @serial
   */
  protected CableSet _upCables;

  /**
   * @serial
   */
  protected Object _slot1;
  /**
   * @serial
   */
  protected Object _slot2;
  /**
   * @serial
   */
  protected Object _slot3;

  /**
   * A lock for mutators.
   *
   * @serial
   */
  protected boolean _lock;


  /**
   * This method modifies the class counter if nodes are read in from
   * the serializable interface.  It tries to make sure loading new
   * nodes in with old ones won't violate the unique
   * identifiers.  ID values larger than the current count cause the count to
   * be set one larger than that ID, and ID values smaller than the current
   * count are reset as though it were a new node were being created.
   *
   * @param in the object input stream the node is being read from
   */
  private void readObject(ObjectInputStream in)
    throws IOException, ClassNotFoundException {
    in.defaultReadObject();
    if (_id>=_count)
      _count=_id+1;
    else
      _id=_count++;
  }


  /**
   * Compares nodes based on their name.  Makes this class comparable.
   *
   * @param o the node to be compared with this instance
   * @return the value of <code>compareToIgnoreCase</code> on the two
   *  nodes' name strings
   */
  public int compareTo(Object o) throws ClassCastException{
    return(getName().compareToIgnoreCase(((Node)o).getName()));
  }


  /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
  /*+++                                                                  +++*/
  /*+++ Remaining methods are prototyped in Node.java, see that          +++*/
  /*+++ file for documentation describing these methods.                 +++*/
  /*+++                                                                  +++*/
  /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/

  public boolean setSlots(Object slot1, Object slot2, Object slot3){
    _slot1 = slot1;
    _slot2 = slot2;
    _slot3 = slot3;
    return(true);
  }

  public sneps3.classes.ContextSet isAssertedIn(){
    return _assertedIn;
  }

  public boolean equals(final sneps3.classes.Node rhs)
  {
    return rhs.getName().equals(_name);
  }

  public boolean deepEquals(final sneps3.classes.Node rhs)
  {
    return this.equals(rhs) &&
       rhs.getSemanticClass() == (_semClass);
  }

  public boolean setName(final String name)
  {
     if (_lock)
       return (false);
     _name = name;	
     return (true);
  }

  public String getName()
  {
    return _name;
  }

  public boolean setSemanticClass(final SemanticClass semClass)
  {
     if (_lock)
       return (false);
     _semClass = semClass;
     return (true);
  }

  public sneps3.classes.SemanticClass getSemanticClass()
  {
    return _semClass;
  }

  public boolean lock()
    {
      boolean ret=_lock;
      _lock = true;
      return ret;
    }

  public String toString(final int level)
    {
      String ind="";
      for (int i=0; i<level; i++)
        ind+=Defaults.INDENT;
      return
        ind + "Name: " + _name + "   " +
        "Semantic Class: " + _semClass.getName() + '\n';
    }

  public String toString(){
    return (toString(0));
  }

  public String defaultName(){
    return ((getSemanticClass().getName()).charAt(0) + Integer.toString(_id));
  }

  public boolean lt(final sneps3.classes.Node rhs){
    return (_id < ((S3_Node)rhs)._id);
  }

	public S3_Node()
	{
		//no superclass constructor
		
		//assign _id and increment the count
		_id=_count++;
	}

}
