package sneps3.corecode;
import java.io.*;


/**
 * Relations connect nodes in the network.
 *
 * @author SNePS 3 Java Programming Team
 * @version 3.0
 */
class S3_Relation implements Relation, Serializable, Cloneable, Comparable{


  /**
   * Number of instances of this class that have been created.
   *
   * @serial
   */
  protected static int _count = 0;


  /**
   * The name of the relation.
   * Relations in a system should have a unique name.
   *
   * @serial
   */
  protected String _name;


  /**
   * The type of the relation.
   * Type indicates which semantic classes of nodes it may point at.
   *
   * @serial
   */
  protected SemanticClass _type;


  /**
   * Indicates how wire-based inference works with this relation.<BR>
   * <code>-1==Relation.REDUCE 0==Relation.NONE 1==Relation.EXPAND</code><BR>
   * Build all constructors and methods to take either int or String args
   * for this field, and when taking string arguments use the constants
   * provided to set the adjust field.
   *
   * @serial
   */
  protected int _adjust;


  /**
   * Minimal limit on wire-based inference.
   * If adjust is reduce, then this is the minimal allowed reduction.
   * If adjust is expand, then this is the minimal number of cables
   * to allow the addition of more.
   *
   * @serial 
   */
  protected int _limit;


  /**
   * The path associated with this relation.
   *
   * @serial
   */
  protected Path _path;


  /**
   * Set of nodes with cables of this relation pointing to them.
   *
   * @serial
   */
  protected NodeSet _nodes;


  /**
   * Set of case frames that contain this relation.
   *
   * @serial
   */
  protected CaseFrameSet _caseFrames;

  /**
   * @serial
   */
  protected PathSet _paths;

  /**
   * Unique identifier for each instance.
   *
   * @serial
   */
  protected int _id;


  /**
   * Access lock for the mutators.  Mutators may only operate when this field
   * is set to <code>false</code>.
   *
   * @serial
   */
  protected boolean _lock;


  /**
   * The class constructor.  Fields will be set with mutators after
   * construction.
   *
   * @return an empty instance
   */
  S3_Relation(){
    _name=null;
    _type=null;
    _paths=null;
    _adjust=-1;
    _limit=-1;
    _id=_count++;
    _lock=false;
  }


  /**
   * This method modifies the class counter if nodes are read in from
   * the serializable interface.  It tries to make sure loading new
   * relations in with old ones won't violate the unique
   * identifiers.
   */
  private void readObject(ObjectInputStream in)
    throws IOException, ClassNotFoundException {
    in.defaultReadObject();
    if (_id>=_count)
      _count=_id+1;
    else
      _id=_count++;
  }


  /**
   * Compares relations s based on their name.  Makes this class comparable.
   *
   * @param o the relation to be compared with this instance
   * @return the value of <code>compareToIgnoreCase</code> on the two
   *  relations' name strings
   */
  public int compareTo(Object o) throws ClassCastException{
    return(getName().compareToIgnoreCase(((Relation)o).getName()));
  }


  /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
  /*+++                                                                  +++*/
  /*+++ Remaining methods are prototyped in Relation.java, see that      +++*/
  /*+++ file for documentation describing these methods.                 +++*/
  /*+++                                                                  +++*/
  /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/

  public boolean setPaths(final PathSet ps){
    _paths=ps;
    return true;
  }

  public sneps3.classes.PathSet getPaths(){
    return (_paths);
  }

  public boolean equals(final sneps3.classes.Relation rhs){
    return(_name.compareToIgnoreCase(rhs.getName())==0);
  }  


  public String getName(){
    return (_name);
  }

  public boolean setName(final String name){
    if (_lock)
      return (false);
    if (name.indexOf(Defaults.SEPERATION_CHARACTER)>=0)
      return (false);
    _name=name;
    return (true);
  }

  public sneps3.classes.SemanticClass getType(){
    return (_type);
  }

  public boolean setType(final SemanticClass type){
    if (_lock)
      return (false);
    _type=type;
    return (true);
  }

  public int getAdjust(){
    return (_adjust);
  }

  public boolean setAdjust(final int value){
    if (_lock)
      return (false);
    _adjust=value;
    return (true);
  }

  public int getLimit(){
    return (_limit);
  }

  public boolean setLimit(final int limit){
    if (_lock)
      return (false);
    _limit=limit;
    return (true);
  }

  public sneps3.classes.Path getPath(){
    return (_path);
  }

  public boolean setPath(final Path path){
    if (_lock)
      return (false);
    _path=path;
    return (true);
  }

  public sneps3.classes.NodeSet getNodes(){
    return (_nodes);
  }

  public boolean setNodes(final NodeSet ns){
    _nodes=ns;
    return (true);
  }

  public boolean setCaseFrames(final CaseFrameSet cfs){
    _caseFrames=cfs;
    return (true);
  }

  public sneps3.classes.CaseFrameSet getCaseFrames(){
    return (_caseFrames);
  }

  public String toString(final int level)
  {
    String ret="";
    String adjustType = "";
    switch (_adjust)
	{
	case REDUCE:
	    adjustType = "Reduce";
	    break;
	case EXPAND:
	    adjustType = "Expand";
	    break;
	case NONE:
	    adjustType = "None";
	    break;
	default:
	    adjustType = "??? invalid adjust type ??? ";
	}
    for (int i=0; i<level; i++)
      ret+=Defaults.INDENT;
    return (ret +
       "Name: " + _name +
       "   Type: " + _type.getName() +
       "   Adjust: " + adjustType +
       "   Limit: " + _limit + '\n');
  }

  /* using a simpler model for default
  public String toString(){
    return (toString(0));
  }
  */

  public String toString(){
    return (_name);
  }

  public boolean lock(){
    boolean ret=_lock;
    _lock=true;
    return (ret);
  }

}
