package sneps3.world.gui;

/*
 * This code is based on an example provided by Richard Stanford, 
 * a tutorial reader.
 */

import sneps3.exception.*;
import sneps3.spine.*;
import sneps3.classes.*;

import java.awt.*;
import javax.swing.*;
import javax.swing.tree.*;
import javax.swing.event.*;
import javax.swing.plaf.basic.BasicTreeUI;


class SemanticClassTree extends JPanel {

  /**
   * @serial
   */
  protected DefaultMutableTreeNode _rootNode;
  /**
   * @serial
   */
  protected DefaultMutableTreeNode _last;
  /**
   * @serial
   */
  protected DefaultTreeModel _treeModel;
  /**
   * @serial
   */
  protected JTree _tree;
  /**
   * @serial
   */
  protected java.util.Hashtable _nodes;
  /**
   * @serial
   */
  protected TelePort _port;
  /**
   * @serial
   */
  protected Toolkit _toolkit;
  /**
   * @serial
   */
  protected SemanticClassFrame _parent;
  
  SemanticClassTree(SemanticClassFrame parent, TelePort port) {

    _parent = parent;
    DefaultTreeCellRenderer myRenderer = new SemanticClassRenderer();

    _port=port;
    _toolkit = Toolkit.getDefaultToolkit();

    _rootNode = new DefaultMutableTreeNode("");
    _treeModel = new DefaultTreeModel(_rootNode);
    _tree = new JTree(_treeModel);
    _tree.setEditable(false);
    _tree.getSelectionModel().setSelectionMode
      (TreeSelectionModel.SINGLE_TREE_SELECTION);
    _tree.setShowsRootHandles(false);
    _tree.putClientProperty("JTree.lineStyle", "Angled");
    _tree.setRootVisible(true);
    _tree.setCellRenderer(myRenderer);
    _tree.setUI(new BasicTreeUI());
    DefaultTreeCellEditor editor = new DefaultTreeCellEditor(_tree,myRenderer);
    _tree.setCellEditor(editor);
    editor.addCellEditorListener(new MyCellEditorListener());
    _tree.addTreeSelectionListener(new MyTreeSelectionListener());
    JScrollPane scrollPane = new JScrollPane(_tree);
    setLayout(new GridLayout(1,0));
    add(scrollPane);
    refresh();
  }

  /** rebuild tree based on the semantic classes now present */
  public void refresh(){

    _rootNode.removeAllChildren();
    _treeModel.reload();

    _nodes = new java.util.Hashtable();

    SemanticClass[] sca =
      (SemanticClass[])_port.getSemanticClasses().toArray();

    java.util.Arrays.sort(sca);

    if (java.lang.reflect.Array.getLength(sca)==0)
      _tree.setRootVisible(false);
    else{
      _tree.setRootVisible(true);
      _rootNode.setUserObject(sca[0]);
      _nodes.put(sca[0].getName(), _rootNode);
    }

    for(int i=1; i<java.lang.reflect.Array.getLength(sca); i++){
      SemanticClass next = sca[i];
      DefaultMutableTreeNode child =
        new DefaultMutableTreeNode(next);
      MutableTreeNode parent=
        (MutableTreeNode)_nodes.get(next.getParent().getName());
      _treeModel.insertNodeInto(child, parent, parent.getChildCount());
      _tree.scrollPathToVisible(new TreePath(child.getPath()));
      _nodes.put(next.getName(), child);      
    }
  }

  public void scUpdate(final int change, final SemanticClass sc){
    switch (change){
    case -1:
      MutableTreeNode removed = (MutableTreeNode)_nodes.get(sc.getName());
      if (removed==null)
        break;
      _treeModel.removeNodeFromParent(removed);
      _nodes.remove(sc.getName());
      break;
    case 0:
      _treeModel.nodeChanged((MutableTreeNode)_nodes.get(sc.getName()));
      break;
    case 1:
      DefaultMutableTreeNode child = new DefaultMutableTreeNode(sc);
      MutableTreeNode parent = (MutableTreeNode)
        _nodes.get(sc.getParent().getName());
      if (parent==null)
        break;
      int i=0;
      while (i<parent.getChildCount()&&((Comparable)((DefaultMutableTreeNode)
          parent.getChildAt(i)).getUserObject()).compareTo(sc)<0)
        i++;
      _treeModel.insertNodeInto(child, parent, i);
      _tree.scrollPathToVisible(new TreePath(child.getPath()));
      _nodes.put(sc.getName(), child);      
      break;
    default:
      refresh();
      break;
    }
  }

  public void setNodeColor(){
    TreePath currentSelection = _tree.getSelectionPath();
    if (currentSelection != null) {
      DefaultMutableTreeNode currentNode = (DefaultMutableTreeNode)
        (currentSelection.getLastPathComponent());
      SemanticClass sc = (SemanticClass)currentNode.getUserObject();
      java.awt.Color color = JColorChooser.showDialog(_tree,
          "Choose a Color", sc.getColor());
      if (color != null){
        try{
          _port.setSemanticClassColor(sc.getName(),color);
        }catch (Exception e){
          _toolkit.beep();
          JOptionPane.showMessageDialog(_parent, e, "Error",
              JOptionPane.WARNING_MESSAGE);
        }
      }
      _parent._mainFrame.scUpdate(0,sc);
      return;
    }
    // There was no selection error dialoge
    _toolkit.beep();
    JOptionPane.showMessageDialog(_parent, "No semantic class selected.",
        "Error", JOptionPane.WARNING_MESSAGE);
  }

  /** Remove the currently selected node. */
  public void removeCurrentNode() {
    TreePath currentSelection = _tree.getSelectionPath();
    if (currentSelection != null) {
      DefaultMutableTreeNode currentNode = (DefaultMutableTreeNode)
        (currentSelection.getLastPathComponent());
      SemanticClass sc=null;
      try{
        sc = (SemanticClass)currentNode.getUserObject();
        _port.undefineSemanticClass(sc.getName());
      }
      catch (Exception e){
        _toolkit.beep();
        JOptionPane.showMessageDialog(_parent, e, "Error",
            JOptionPane.WARNING_MESSAGE);
        return;
      }
      _parent._mainFrame.scUpdate(-1,sc);
      return;
    } 
    
    // There was no selection error dialoge
    _toolkit.beep();
    JOptionPane.showMessageDialog(_parent, "No semantic class selected.",
        "Error", JOptionPane.WARNING_MESSAGE);
  }

  /** Add child to the currently selected node, or a root if empty. */
  public void addObject() {
    DefaultMutableTreeNode parentNode = null;
    TreePath parentPath = _tree.getSelectionPath();

    if (!_tree.isRootVisible()){
      TreePath path = new TreePath(_rootNode.getPath());
      _rootNode.setUserObject(sneps3.corecode.Defaults.FACTORY.createSemanticClass("",null));
      _tree.setRootVisible(true);
      _tree.scrollPathToVisible(path);
      _tree.setEditable(true);
      _last = _rootNode;
      _tree.startEditingAtPath(path);
    }
    else{
      if (parentPath == null) {
        // error dialoge, must select a parent class
        _toolkit.beep();
        JOptionPane.showMessageDialog(_parent,
            "Select a parent semantic class first.",
            "Error", JOptionPane.WARNING_MESSAGE);
        return;
      }
      parentNode = (DefaultMutableTreeNode)(parentPath.getLastPathComponent());
      DefaultMutableTreeNode childNode = new DefaultMutableTreeNode(sneps3.corecode.Defaults.FACTORY.createSemanticClass("",null));
      _treeModel.insertNodeInto(childNode,
          parentNode, 0);
      TreePath childPath = new TreePath(childNode.getPath());
      _tree.scrollPathToVisible(childPath);
      _tree.setEditable(true);
      _last = childNode;
      _tree.startEditingAtPath(childPath);
    }
    
  }


  class MyCellEditorListener implements CellEditorListener {

    public void editingCanceled(ChangeEvent e){
      _tree.setEditable(false);

      if (_last==null) return;

      if (_last.isRoot()){
        _tree.setRootVisible(false);
        _rootNode.setUserObject(null);
      }
      else
        _treeModel.removeNodeFromParent(_last);      

      _last = null;
    }

    public void editingStopped(ChangeEvent e){
      _tree.setEditable(false);

      if (_last==null) return;

      String field =
        ((String)((TreeCellEditor)e.getSource()).getCellEditorValue()).trim();

      if (field.length()==0){
        //error dialoge Cannot have blank name
        _toolkit.beep();
        JOptionPane.showMessageDialog(_parent,
            "Invalid name for semantic class.",
            "Error", JOptionPane.WARNING_MESSAGE);
        editingCanceled(e);
        return;
      }

      String parentName;

      if (_last.isRoot())
        parentName = null;
      else{
        DefaultMutableTreeNode parent =
          (DefaultMutableTreeNode)_last.getParent();
        parentName = ((SemanticClass)parent.getUserObject()).getName();
      }

      try{
        SemanticClass sc=_port.defineSemanticClass(field,parentName);
        if (_last==null) return;

        if (_last.isRoot()){
          _tree.setRootVisible(false);
          _rootNode.setUserObject(null);
        }
        else
          _treeModel.removeNodeFromParent(_last);      
        
        _last = null;
        _parent._mainFrame.scUpdate(1,sc);
      }
      catch (Exception exception){
        _toolkit.beep();
        JOptionPane.showMessageDialog(_parent, exception,
            "Error", JOptionPane.WARNING_MESSAGE);
        editingCanceled(e);
      }
    }
  }

  class MyTreeSelectionListener implements TreeSelectionListener {
    public void valueChanged(TreeSelectionEvent e){
      SemanticClass current=null;
      if (_tree.getSelectionPath()!=null)
        current = (SemanticClass)((DefaultMutableTreeNode)
            (_tree.getSelectionPath().getLastPathComponent())).getUserObject();
      if (current!=null)
        _parent._mainFrame.scSelect(1,current);
    }
  }
}
