/*
 * util.c
 *
 */

#include <time.h>
#include <stdlib.h>
#include <gsl/gsl_math.h>
#include <gsl/gsl_vector.h>
#include <gsl/gsl_matrix.h>
#include <gsl/gsl_sf.h>
#include <gsl/gsl_randist.h>
#include <gsl/gsl_cdf.h>
#include <gsl/gsl_blas.h>
#include <gsl/gsl_linalg.h>
#include <gsl/gsl_sf_gamma.h>
#include <math.h>
#include <assert.h>

#include "util.h"

#define min(x1, x2) ( (x1) > (x2) ? (x2) : (x1) )

// global variable
gsl_rng* glob_r;

double maxx(double a, double b)
{
	if(a > b){
		return a;
	}else{
		return b;
	}
}

/*
 * wrapped vector operations
 *
 */
double vget(const gsl_vector* v, int i)
{
	return (gsl_vector_get(v, i));
}

void vset(gsl_vector* v, int i, double x)
{
	gsl_vector_set(v, i, x);
}

void vinc(gsl_vector* v, int i, double x)
{
	vset(v, i, vget(v, i) + x);
}

void vprint(gsl_vector * v)
{
	int i;
	for (i = 0; i < (int)v->size; i++)
		printf("%0.9lf ", vget(v, i));
	printf(";\n");
}

void scanf_vector(char* filename, gsl_vector* v)
{
	FILE* fileptr;
	fileptr = fopen(filename, "r");
	gsl_vector_fscanf(fileptr, v);
	fclose(fileptr);
}

void save_vector(char* filename, gsl_vector* v)
{
	FILE* fileptr;
	fileptr = fopen(filename, "w");
	gsl_vector_fprintf(fileptr, v, "%g");
	fclose(fileptr);
}

void normalize(gsl_vector* v)
{
	int size = (int)v->size;
	double sum_v = vsum(v);
	int i;
	for (i = 0; i < (int)size; i++)
		vset(v, i, vget(v, i) / sum_v);
}

double vsum(gsl_vector* v)
{
	double *data = v->data, val = 0;
	int size = v->size;
	int i;
	for (i = 0; i < (int)size; i++)
		val += data[i];
	return (val);
}

int veq(const gsl_vector* v1, const gsl_vector* v2)
{
	int i;
	if(v1->size != v2->size){
		fprintf(stderr, "V_EQUAL: v1->size != v2->size !!!\n");
		exit(1);
	}
	double *data1 = v1->data;
	double *data2 = v2->data;
	for(i = 0; i < (int)v1->size; i++){
		if(data1[i] != data2[i]){
			return 0;
		}
	}
	return 1;
}

int v_contain(const gsl_vector* v, double k)
{
	int i;
	for (i = 0; i < (int)v->size; i++) {
		if (vget(v, i) == k) {
			return 1;
		}
	}
	return 0;
}

int v_non_zero(const gsl_vector* v)
{
	int i;
	int count = 0;
	for(i = 0; i < (int)v->size; i++){
		if(vget(v, i) != 0){
			count++;
		}
	}
	return count;
}

/*
 * wrapped matrix operations
 *
 */
double mget(const gsl_matrix* m, int i, int j)
{
	return (gsl_matrix_get(m, i, j));
}

void mset(gsl_matrix* m, int i, int j, double x)
{
	gsl_matrix_set(m, i, j, x);
}

void minc(gsl_matrix* m, int i, int j, double x)
{
	mset(m, i, j, mget(m, i, j) + x);
}

void col_sum(gsl_matrix* m, gsl_vector* v)
{
	int i, j;

	if(m->size2 != v->size){
		fprintf(stderr, "COL_SUM: v->size != m->size2 !!!\n");
		exit(1);
	}

	gsl_vector_set_all(v, 0);
	for (i = 0; i < (int)m->size1; i++)
		for (j = 0; j < (int)m->size2; j++)
			vinc(v, j, mget(m, i, j));
}

double col_sum1(gsl_matrix* m, int col)
{
	double s = 0;
	int i;
	for(i = 0; i < (int)m->size1; i++){
		s += mget(m, i, col);
	}
	return s;
}

void row_sum(gsl_matrix* m, gsl_vector* v)
{
	int i, j;
	if(m->size1 != v->size){
		fprintf(stderr, "ROW_SUM: v->size != m->size1 !!!\n");
		exit(1);
	}

	gsl_vector_set_zero(v);
	for (i = 0; i < (int)m->size1; i++)
	{
		for( j = 0; j < (int)m->size2; j++)
			vinc(v, i, mget(m, i, j));
	}
}

double msum(gsl_matrix* m)
{
	int i, j;
	double sum = 0;

	for(i = 0; i < (int)m->size1; i++)
	{
		for(j = 0; j < (int)m->size2; j++)
		{
			sum += mget(m, i, j);
		}
	}
	return sum;
}

void minverse(gsl_matrix *src, gsl_matrix *inv)
{
	int ss, n;
	assert(src->size1 == src->size2);
	n = src->size1;
	gsl_permutation * perm = gsl_permutation_alloc(n);
	gsl_linalg_LU_decomp(src, perm, &ss);
	gsl_linalg_LU_invert(src, perm, inv);
	gsl_permutation_free(perm);
}

void mm_mult(gsl_matrix *m1, gsl_matrix *m2, gsl_matrix *result)
{
	int i, j, k;
	double s;
	assert(m1->size1 > 0 && m1->size2 > 0 
			&& m2->size1 > 0 && m2->size2 > 0 
			&& m1->size2 == m2->size1 && m1->size1 == result->size1 && m2->size2 == result->size2);
	for(i = 0; i < result->size1; i++){
		for(j = 0; j < result->size2; j++){
			s = 0;
			for(k = 0; k < m1->size2; k++){
				s += mget(m1, i, k) * mget(m2, k, j);
			}
			mset(result, i, j, s);
		}
	}
}

void mv_mult(gsl_matrix *m1, gsl_vector *m2, gsl_vector *result)
{
	int i, j;
	double s;
	assert(m1->size1 > 0 && m1->size2 > 0 
			&& m1->size2 == m2->size && m1->size1 == result->size);
	for(i = 0; i < result->size; i++){
		s = 0;
		for(j = 0; j < m1->size2; j++){
			s += mget(m1, i, j) * vget(m2, j);
		}
		vset(result, i, s);
	}
}

void mprint(gsl_matrix * m)
{
	int i, j;
	for (i = 0; i < (int)m->size1; i++)
	{
		for (j = 0; j < (int)m->size2; j++)
			printf("%0.9lf ", mget(m, i, j));
		printf("\n");
	}
}

void scanf_matrix(char* filename, gsl_matrix * m)
{
	FILE* fileptr;
	fileptr = fopen(filename, "r");
	gsl_matrix_fscanf(fileptr, m);
	fclose(fileptr);
}

void save_matrix(char* filename, gsl_matrix* m)
{
	FILE* fileptr;
	fileptr = fopen(filename, "w");
	gsl_matrix_fprintf(fileptr, m, "%g");
	fclose(fileptr);
}

void printf_matrix(char* file, gsl_matrix * m)
{
	FILE* fileptr;
	int i, j;
	fileptr = fopen(file, "w");
	for(i = 0; i < (int)m->size1; i++)
	{
		for(j = 0; j < (int)m->size2; j++)
		{
			fprintf(fileptr, "%.9lf ", mget(m, i, j));
		}
		fprintf(fileptr, "\n\n");
	}
	fclose(fileptr);
}

/*
 * initial the random number generator
 *
 */
void initial_rng()
{
//	long seed = (long)1115574245;
	glob_r = gsl_rng_alloc(gsl_rng_taus);
	time_t seed;
	time(&seed);
	gsl_rng_set(glob_r, (long)seed);
	printf("The seed for glob_r: %ld\n", (long)seed);
}

/*
 * return a random double in the range 0 to 1, exclusive
 *
 */
double next_uniform()
{
	return (gsl_rng_uniform(glob_r));
}

double exp_rand(double mu)
{
	return gsl_ran_exponential (glob_r, mu);
}

double norm_rand(double mu, double sigma)
{
	return gsl_ran_gaussian (glob_r, sigma) + mu;
}

static double INV_Gau_CDF(double t, double mu, double lambda)
{
	return gsl_cdf_gaussian_P(sqrt(lambda / t) * (t / mu - 1), 1) + exp(2 * lambda / mu) * gsl_cdf_gaussian_P(-sqrt(lambda / t) * (t / mu + 1), 1);
}

static double an_fun(double x, double t, int n)
{
	assert(x > 0);
	if(x <= t){
		return M_PI * (n + 0.5) * pow(2 / M_PI / x, 1.5) * exp(-2 * (n + 0.5) * (n + 0.5) / x);
	}else{
		return M_PI * (n + 0.5) * exp(-(n + 0.5) * (n + 0.5) * M_PI * M_PI * x / 2);
	}
}

double PG_rand(double z)
{
	double t, K, p, q, U, V, X;
	if(z < 0){
		z /= -2;
	}else{
		z /= 2;
	}
	t = 0.64;
	K = M_PI * M_PI / 8 + z * z / 2;
	p = M_PI * exp(-K * t) / 2 / K;
	q = 2 * exp(-z) * INV_Gau_CDF(t, 1 / z, 1.0);
	while(1){
		U = next_uniform();
		V = next_uniform();
		if(U < p / (p + q)){
			X = t + exp_rand(1) / K;
		}else{
			double mu = 1 / z;
			if(mu > t){
				while(1){
					double E, E1;
					E = exp_rand(1);
					E1 = exp_rand(1);
					while(E * E > 2 * E1 / t){
						E = exp_rand(1);
						E1 = exp_rand(1);
					}
					X = t / (1 + t * E) / (1 + t * E);
					if(next_uniform() < exp(-z * z * X / 2)){
						//X = 1.0 / X;
						break;
					}
				}
			}else{
				while(1){
					double Y = norm_rand(0, 1);
					Y = Y * Y;
					X = mu + 0.5 * mu * mu * Y - 0.5 * mu * sqrt(4 * mu * Y + mu * mu * Y * Y);
					if(next_uniform() > mu / (mu + X)){
						X = mu * mu / X;
					}
					if(X <= t){
						break;
					}
				}
			}
		}
		double S = an_fun(X, t, 0);
		double Y = V * S;
		int n = 0;int a5 = 0;
		while(1){a5++; assert(a5 < 1000);
			n++;
			if(n % 2 == 1){
				S = S - an_fun(X, t, n);
				if(Y < S){
					return X / 4;
				}
			}else{
				S = S + an_fun(X, t, n);
				if(Y > S){
					break;
				}
			}
		}
	}
}

double PG_m(int b, double z)
{
	double s = 0;
	for(int i = 0; i < b; i++){
		s += PG_rand(z);
	}
	return s;
}

//////////////////////////////////////////////
static double gamma_rate(double shape, double rate)
{
	return gsl_ran_gamma (glob_r, shape, 1.0 / rate);
}
static double gamma_scale(double shape, double scale)
{
	return gsl_ran_gamma (glob_r, shape, scale);
}

// Truncatatoin at t = 1.
static double right_tgamma_reject(double shape, double rate)
{
	double x = 2.0;
	while (x > 1.0)
		x = gamma_rate(shape, rate);
	return x;
}

static double Gamma(double x, int use_log)
{
  double y = gsl_sf_lngamma(x);
  if (!use_log) y = exp(y);
  return y;
}

static double p_gamma_rate(double x, double shape, double rate, int use_log)
{
  double scale = 1.0 / rate;
  double y = gsl_cdf_gamma_P(x, shape, scale);
  if (use_log) y = log(y);
  return y;
}

static double omega_k(int k, double a, double b)
{
	double log_coef = -b + (a+k-1) * log(b) - Gamma(a+k, true) - p_gamma_rate(1.0, a, b, true);
	return exp(log_coef);
}

// Truncation at t = 1.
static double right_tgamma_beta(double shape, double rate)
{
	double a = shape;
	double b = rate;

	double u = next_uniform();

	int k = 1;
	double cdf = omega_k(1, a, b);
	while (u > cdf) {
		cdf += omega_k(++k, a, b);
	}

	return gsl_ran_beta (glob_r, a, k);
}

double rtgamma_rate(double shape, double rate, double right_t)
{
  // x \sim (a,b,t)
  // ty = x
  // y \sim (a, bt, 1);
	double a = shape;
	double b = rate * right_t;

	double p = p_gamma_rate(1, a, b, false);
	double y = 0.0;
	if (p > 0.95)
		y = right_tgamma_reject(a, b);
	else
		y = right_tgamma_beta(a,b);

	double x = right_t * y;
	return x;
}

/*
 * daw a single sample from (unnormalized) multinormial v, with normalizing factor m
 *
 */
int next_discrete_unnormalised(gsl_vector* v, double sum)
{
	int i;
	double b = 0;
	double r = next_uniform() * sum;
//	printf(">>>>>>>>>>>>>>>r = %lf\n", r);
	for (i = 0; i < (int)v->size; i++)
	{
		b += vget(v, i);
		if (b > r) {
			return (i);
		}
	}
	return (v->size - 1);
}
/*
 * daw a single sample from multinormial v
 *
 */
int next_discrete_normalised(double* v, int k)
{
	int i;
	double b = 0;
	double r = next_uniform();
	for (i = 0; i < (int)k; i++){
		b += v[i];
		if (b > r) {
			return (i);
		}
	}
	return (k - 1);
}

double max_val(double* vec, int k)
{
	int i;
	double max = vec[0];
	for(i = 1; i < k; ++i){
		if(vec[i] > max){
			max = vec[i];
		}
	}
	return max;
}

void free_rng()
{
	gsl_rng_free(glob_r);
}

double safe_log(double x)
{
	if (x == 0)
		return (-1000);
	else
		return (log(x));
}

double log_sum(double V, double lp)
{
	if (lp > V) {
		// swap so V is bigger
		double t = lp;
		lp = V;
		V = t;
	}
	return V + log(1.0 + exp(lp - V));
}

/*
 * (x|y)_n
 */
double poch_sym(double x, double y, int n)
{
	int i;
	double tmp = 1.0;
	for(i = 0; i < n; i++){
		tmp *= (x + i*y);
	}
	return tmp;
}
/*
 * (x|y)_n
 */
double log_poch_sym(double x, double y, int n)
{
	double log_poch = 0;

	if(y == 0){
		log_poch = n * gsl_sf_log(x);
	}else{
		double tmp = x/y;
		log_poch = gsl_sf_lngamma(tmp + n) - gsl_sf_lngamma(tmp) + n * gsl_sf_log(y);
	}
	return log_poch;
}

/*
 * (x|y)_n
 */
double log_poch_sym_1(double x, double y, int n)
{
	int i;
	double log_poch = 0;
	for(i = 0; i < n; i++)
	{
		log_poch += gsl_sf_log(x + i*y);
	}
	return log_poch;
}

double randgamma(double rr)
{
	double bb, cc, dd;
	double uu, vv, ww, xx, yy, zz;

	if ( rr <= 0.0 ) {
		/* Not well defined, set to zero and skip. */
		return 0.0;
	} else if ( rr == 1.0 ) {
		/* Exponential */
		return - log(drand48());
	} else if ( rr < 1.0 ) {
		/* Use Johnks generator */
		cc = 1.0 / rr;
		dd = 1.0 / (1.0-rr);
		while (1) {
			xx = pow(drand48(), cc);
			yy = xx + pow(drand48(), dd);
			if ( yy <= 1.0 ) {
				return -log(drand48()) * xx / yy;
			}
		}
	} else { /* rr > 1.0 */
		/* Use bests algorithm */
		bb = rr - 1.0;
		cc = 3.0 * rr - 0.75;
		while (1) {
			uu = drand48();
			vv = drand48();
			ww = uu * (1.0 - uu);
			yy = sqrt(cc / ww) * (uu - 0.5);
			xx = bb + yy;
			if (xx >= 0) {
				zz = 64.0 * ww * ww * ww * vv * vv;
				if ( ( zz <= (1.0 - 2.0 * yy * yy / xx) ) ||
						( log(zz) <= 2.0 * (bb * log(xx / bb) - yy) ) ) {
					return xx;
				}
			}
		}
	}
}

void randdir(double *pi, double *alpha, int veclength, int skip)
{
	double *pi2, *piend;
	double sum;

	sum = 0.0;
	piend = pi + veclength*skip;
	for ( pi2 = pi ; pi2 < piend ; pi2 += skip) {
		sum += *pi2 = randgamma(*alpha);
		alpha += skip;
	}
	for ( pi2 = pi ; pi2 < piend ; pi2 += skip) {
		*pi2 /= sum;
	}
}

struct gammaPost{
	double a, b;
	gammaPost(double a, double b)
	:a(a), b(b)
	{}
	double operator()(double x) const
	{
		return (a - 1) * log(x) - b * x;
	}
};
double trun_Gamma(gsl_rng *gen, double a, double b, double lower, double upper, double initx)
{
	glob_r = gen;
	/*if(a > 1.0){
		gammaPost gpos(a, b);
		if(upper < 1.0 / 0.0){
			initx = (lower + upper) / 2;
		}else{
			initx = lower + 1;
		}
		return slice_sampler1d(gpos, initx, drand48, lower, upper, 0.0, 10, 1000);*/
		/*double gam, x, g;
		g = (b * lower - a + sqrt((b * lower - a) * (b * lower - a) + 4 * b * lower)) / 2 / lower;
		if(g <= b){
			gam = g;
		}else{
			gam = b;
		}
		while(1){
			x = lower + gsl_ran_gamma (gen, a, 1.0 / gam);
			g = (a - 1) * log(x) - (b - gam) * x - (a - 1) * log((a - 1) / (b - gam)) + (a - 1);
			if(log(next_uniform()) < g){
				return x;
			}
		}
	}else{//only for one side
		double gam, x, g;
		if(1.0 / lower <= b){
			gam = 1.0 / lower;
		}else{
			gam = b;
		}
		while(1){
			x = lower + gsl_ran_gamma (gen, a, 1.0 / gam);
			g = (a - 1) * log(x) - (b - gam) * x - (a - 1) * log(lower) + (b - gam) * lower;
			if(log(next_uniform()) < g){
				return x;
			}
		}
	}*/

	// two-sided truncation
	double g, x, l, x0 = (a - 1) / b;
	l = upper - lower;
	while(1){
		x = lower + next_uniform() * l;
		if(a >= 1 && x0 >= lower && x0 <= upper){
			g = (a - 1) * log(x) - b * x - (a - 1) * log((a - 1) / b) + a + 1;
		}else if(a >= 1 && x0 > upper){
			g = (a - 1) * log(x) - b * x - (a - 1) * log(upper) + b * upper;
		}else{
			g = (a - 1) * log(x) - b * x - (a - 1) * log(lower) + b * lower;
		}
		if(log(next_uniform()) <= g){
			return x;
		}
	}
}

/***************************************************************************************
 *  Multivariate Normal density function and random number generator
 *  Multivariate Student t density function and random number generator
 *  Wishart random number generator
 *  Using GSL -> www.gnu.org/software/gsl
 *
 *  Copyright (C) 2006  Ralph dos Santos Silva
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  AUTHOR
 *     Ralph dos Santos Silva,  address@hidden
 *     March, 2006
***************************************************************************************/

/*****************************************************************************************************************/
/*****************************************************************************************************************/
int rmvnorm(const gsl_rng *r, const int n, const gsl_vector *mean, const gsl_matrix *var, gsl_vector *result)
{
	/* multivariate normal distribution random number generator */
	/*
	*	n	dimension of the random vetor
	*	mean	vector of means of size n
	*	var	variance matrix of dimension n x n
	*	result	output variable with a sigle random vector normal distribution generation
	*/
	int k;
	gsl_matrix *work = gsl_matrix_alloc(n,n);

	gsl_matrix_memcpy(work,var);
	gsl_linalg_cholesky_decomp(work);

	for(k=0; k<n; k++)
		gsl_vector_set( result, k, gsl_ran_ugaussian(r) );

	gsl_blas_dtrmv(CblasLower, CblasNoTrans, CblasNonUnit, work, result);
	gsl_vector_add(result,mean);

	gsl_matrix_free(work);

	return 0;
}

int rmvnorm_1(const gsl_rng *r, const int n, scythe::Matrix<> &mean, scythe::Matrix<> &var,
		gsl_vector * res1, gsl_matrix *work, double *result)
{
	/* multivariate normal distribution random number generator */
	/*
	*	n	dimension of the random vetor
	*	mean	vector of means of size n
	*	var	variance matrix of dimension n x n
	*	result	output variable with a sigle random vector normal distribution generation
	*/
	int k;
	for(int i = 0; i < n; i++){
		for(int j = 0; j < n; j++){
			mset(work, i, j, var(i, j));
		}
	}

	gsl_linalg_cholesky_decomp(work);

	for(k=0; k<n; k++){
		gsl_vector_set( res1, k, gsl_ran_ugaussian(r));
	}

	gsl_blas_dtrmv(CblasLower, CblasNoTrans, CblasNonUnit, work, res1);

	for(k = 0; k < n; k++){
		result[k] = vget(res1, k) +  mean(k);
	}

	return 0;
}

/*****************************************************************************************************************/
/*****************************************************************************************************************/
double dmvnorm(const int n, const gsl_vector *x, const gsl_vector *mean, const gsl_matrix *var)
{
	/* multivariate normal density function    */
	/*
	*	n	dimension of the random vetor
	*	mean	vector of means of size n
	*	var	variance matrix of dimension n x n
	*/
	int s;
	double ax,ay;
	gsl_vector *ym, *xm;
	gsl_matrix *work = gsl_matrix_alloc(n,n),
			   *winv = gsl_matrix_alloc(n,n);
	gsl_permutation *p = gsl_permutation_alloc(n);

	gsl_matrix_memcpy( work, var );
	gsl_linalg_LU_decomp( work, p, &s );
	gsl_linalg_LU_invert( work, p, winv );
	ax = gsl_linalg_LU_det( work, s );
	gsl_matrix_free( work );
	gsl_permutation_free( p );

	xm = gsl_vector_alloc(n);
	gsl_vector_memcpy( xm, x);
	gsl_vector_sub( xm, mean );
	ym = gsl_vector_alloc(n);
	gsl_blas_dsymv(CblasUpper,1.0,winv,xm,0.0,ym);
	gsl_matrix_free( winv );
	gsl_blas_ddot( xm, ym, &ay);
	gsl_vector_free(xm);
	gsl_vector_free(ym);
	ay = exp(-0.5*ay)/sqrt( pow((2*M_PI),n)*ax );

	return ay;
}

/*****************************************************************************************************************/
/*****************************************************************************************************************/
int rmvt(const gsl_rng *r, const int n, const gsl_vector *location, const gsl_matrix *scale, const int dof, gsl_vector *result)
{
	/* multivariate Student t distribution random number generator */
	/*
	*	n	 dimension of the random vetor
	*	location vector of locations of size n
	*	scale	 scale matrix of dimension n x n
	*	dof	 degrees of freedom
	*	result	 output variable with a single random vector normal distribution generation
	*/
	int k;
	gsl_matrix *work = gsl_matrix_alloc(n,n);
	double ax = 0.5*dof;

	ax = gsl_ran_gamma(r,ax,(1/ax));     /* gamma distribution */

	gsl_matrix_memcpy(work,scale);
	gsl_matrix_scale(work,(1/ax));       /* scaling the matrix */
	gsl_linalg_cholesky_decomp(work);

	for(k=0; k<n; k++)
		gsl_vector_set( result, k, gsl_ran_ugaussian(r) );

	gsl_blas_dtrmv(CblasLower, CblasNoTrans, CblasNonUnit, work, result);
	gsl_vector_add(result, location);

	gsl_matrix_free(work);

	return 0;
}

/*****************************************************************************************************************/
/*****************************************************************************************************************/
double dmvt(const int n, const gsl_vector *x, const gsl_vector *location, const gsl_matrix *scale, const int dof)
{
	/* multivariate Student t density function */
	/*
	*	n	 dimension of the random vetor
	*	location vector of locations of size n
	*	scale	 scale matrix of dimension n x n
	*	dof	 degrees of freedom
	*/
	int s;
	double ax,ay,az=0.5*(dof + n);
	gsl_vector *ym, *xm;
	gsl_matrix *work = gsl_matrix_alloc(n,n),
			   *winv = gsl_matrix_alloc(n,n);
	gsl_permutation *p = gsl_permutation_alloc(n);

	gsl_matrix_memcpy( work, scale );
	gsl_linalg_LU_decomp( work, p, &s );
	gsl_linalg_LU_invert( work, p, winv );
	ax = gsl_linalg_LU_det( work, s );
	gsl_matrix_free( work );
	gsl_permutation_free( p );

	xm = gsl_vector_alloc(n);
	gsl_vector_memcpy( xm, x);
	gsl_vector_sub( xm, location );
	ym = gsl_vector_alloc(n);
	gsl_blas_dsymv(CblasUpper,1.0,winv,xm,0.0,ym);
	gsl_matrix_free( winv );
	gsl_blas_ddot( xm, ym, &ay);
	gsl_vector_free(xm);
	gsl_vector_free(ym);

	ay = pow((1+ay/dof),-az)*gsl_sf_gamma(az)/(gsl_sf_gamma(0.5*dof)*sqrt( pow((dof*M_PI),n)*ax ));

	return ay;
}
/*****************************************************************************************************************/
/*****************************************************************************************************************/
int rwishart(const gsl_rng *r, const int n, const int dof, const gsl_matrix *scale, gsl_matrix *result)
{
	/* Wishart distribution random number generator */
	/*
	*	n	 gives the dimension of the random matrix
	*	dof	 degrees of freedom
	*	scale	 scale matrix of dimension n x n
	*	result	 output variable with a single random matrix Wishart distribution generation
	*/
	int k,l;
	gsl_matrix *work = gsl_matrix_calloc(n,n);

	for(k=0; k<n; k++){
		gsl_matrix_set( work, k, k, sqrt( gsl_ran_chisq( r, (dof-k) ) ) );
		for(l=0; l<k; l++){
			gsl_matrix_set( work, k, l, gsl_ran_ugaussian(r) );
		}
	}
	gsl_matrix_memcpy(result,scale);
	gsl_linalg_cholesky_decomp(result);
	gsl_blas_dtrmm(CblasLeft,CblasLower,CblasNoTrans,CblasNonUnit,1.0,result,work);
	gsl_blas_dsyrk(CblasUpper,CblasNoTrans,1.0,work,0.0,result);

	return 0;
}


static double LogLik_ptr(gsl_vector *x)
{
	return 0;
}

//void Ellip_slice(gsl_vector *f, double (*loglik)(gsl_vector *x), const gsl_vector *mean, const gsl_matrix *var)
void Ellip_slice(gsl_vector *f, const gsl_vector *mean, const gsl_matrix *var)
{
	int n = f->size;
	gsl_vector *v = gsl_vector_alloc(n);
	gsl_vector *fcos = gsl_vector_alloc(n);
	gsl_vector *vsin = gsl_vector_alloc(n);
	rmvnorm(glob_r, n, mean, var, v);
	double u = next_uniform();
	double logy = LogLik_ptr(f) + log(u);
	double theta = 2 * M_PI * next_uniform();
	double theta_min = theta - 2 * M_PI;
	double theta_max = theta;
	while(1){
		gsl_vector_memcpy(fcos, f);
		gsl_vector_scale (fcos, cos(theta));
		gsl_vector_memcpy(vsin, v);
		gsl_vector_scale (vsin, sin(theta));
		gsl_vector_add (fcos, vsin);
		if(LogLik_ptr(fcos) > logy){
			gsl_vector_memcpy(f, fcos);
			break;
		}else{
			if(theta < 0){
				theta_min = theta;
			}else{
				theta_max = theta;
			}
			theta = theta_min + (theta_max - theta_min) * next_uniform();
		}
	}
	gsl_vector_free(v);
	gsl_vector_free(vsin);
	gsl_vector_free(fcos);
}


/*
 * generalized inverse Gaussian
 */

static double phix(double alpha, double lambda, double x)
{
	return -alpha * (cosh(x) - 1) - lambda * (exp(x) - x - 1);
}
static double phix_(double alpha, double lambda, double x)
{
	return -alpha * sinh(x) - lambda * (exp(x) - 1);
}
static double chiX(double s_, double t_, double eta, double zeta,
		double theta, double xi, double s, double t, double x)
{
	if(x >= -s_ && x <= t_){
		return 1;
	}else if(x > t_){
		return exp(-eta - zeta * (x - t));
	}else{
		return exp(-theta + xi * (x + s));
	}
}
double GIGRND(double pp, double a, double b)
{
	double sample;
	double lambda, omega, alpha, res, t, s, eta, zeta, theta, xi, p, r, t_, s_, q, U, V, W, X;
	int iter;

	if(pp >= 0){
    	lambda = pp;
	}else{
    	lambda = -pp;
	}

	omega = sqrt(a * b);

	alpha = sqrt(omega * omega + lambda * lambda) - lambda;

	res = -phix(alpha, lambda, 1);
	if(res >= 0.5 && res <= 2){
    	t = 1;
    	s = 1;
	}else if(res > 2){
    	t = sqrt(2 / (alpha + lambda));
    	s = sqrt(4 / (alpha * cosh(1) + lambda));
	}else{
    	t = log(4 / (alpha + 2 * lambda));
    	s = min(1 / lambda, log(1 + 1 / alpha + sqrt(1 / alpha / alpha + 2 / alpha)));
	}

	eta = -phix(alpha, lambda, t);
	zeta = -phix_(alpha, lambda, t);
	theta = -phix(alpha, lambda, -s);
	xi = phix_(alpha, lambda, -s);
	p = 1 / xi;
	r = 1 / zeta;

	t_ = t - r * eta;
	s_ = s - p * theta;
	q = t_ + s_;

	iter = 0;
	while(1){
		U = drand48();
		V = drand48();
		W = drand48();
		if(U < q / (p + q + r)){
			X = -s_ + q * V;
		}else if(U < (q + r) / (p + q + r)){
			X = t_ + r * log(1 / V);
		}else{
			X = -s_ - p * log(1 / V);
		}
		if(W * chiX(s_, t_, eta, zeta, theta, xi, s, t, X) <= exp(phix(alpha, lambda, X))){
			break;
		}
		iter++;
		if(iter > 100){ //too many iterations, set to its mode
			X = 0;
			break;
		}
	}

	if(pp < 0){
		sample = sqrt(b / a) / ((lambda / omega + sqrt(1 + lambda * lambda / omega / omega)) * exp(X));
	}else{
		sample = sqrt(b / a) * (lambda / omega + sqrt(1 + lambda * lambda / omega / omega)) * exp(X);
	}
	return sample;
}
///////////////
