import greenfoot.*;  // (World, Actor, GreenfootImage, Greenfoot and MouseInfo)
import java.util.HashSet; //Needed to store the aliens

/**
 * This class implements some of the more complicated functionality of the alien block, and provides
 * helper methods to use in the Alien subclass.
 * 
 * It is provided complete and does not need to be modified.
 * 
 * @author Michael Berry 
 * @version 21/03/2008
 */
public abstract class MasterAlien extends Actor
{

    private boolean direction;      //This is true if the aliens are travelling right, false if they're travelling left.
    private AlienWorld world;       //The world the aliens are in
    /**The step moved by the aliens horizontally. The bigger this is, the faster the aliens will move.*/
    protected final int HSTEP = 1; 
    /**The step moved by the aliens vertically. The bigger this is, the faster the aliens will move downwards.*/
    protected final int YSTEP = 5;
    private final int ROWS = 3;     //The number of rows of aliens
    private HashSet<Alien> aliens;  //The HashSet containing all the invidiual aliens    
    
    /**
     * The individual alien Actor. This class is provided complete and does not need to be modified.
     * It deals with the aliens shooting randomly, and disappearing if they are shot.
     * 
     * @author Michael Berry
     * @version 21/03/2008
     */
    private class Alien extends Actor
    {
        private AlienWorld world;
        
        /**
         * Randomly shoot downwards, and remove self from the world if shot.
         */
        public void act() 
        {
            Shot shot;
            //If shot, then disappear, remove the shot, and add 20 to the score
            if((shot = (Shot)getOneIntersectingObject(Shot.class))!=null && shot.getDirection()) {
                world.removeObject(this);
                world.removeObject(shot);
                aliens.remove(this);
                world.updateScore(20);
            }
            //There is a small chance (1/5000) that the alien will shoot.
            if(Greenfoot.getRandomNumber(5000)==1) {
                shot = new Shot(false);
                world.addObject(shot, getX(), getY());
            }
        }

        /**
         * Called when the object is added into the world, you can safely ignore this method.
         */
        public void addedToWorld(World world)
        {
            this.world = (AlienWorld)world;
            setImage("images/alien.bmp");
        }
        
        /**
         * Returns whether this alien is colliding with a ship.
         * @return true if the slien is colliding with a ship.
         */
        public Ship shipIntersect()
        {
            return (Ship)getOneIntersectingObject(Ship.class);
        }
    }
    
    /**
     * Called when the object is added into the world, you can safely ignore this method.
     */
    public void addedToWorld(World world)
    {
        this.world = (AlienWorld)world;
        setImage(new GreenfootImage(1, 1));
        aliens = new HashSet<Alien>();        
        for(int i = getX() ; i<=getX()+800 ; i+=50) {
            for(int j = getY() ; j<getY()+(ROWS*50) ; j+=50) {
                Alien alien = new Alien();
                world.addObject(alien, i, j);
                aliens.add(alien);
            }
        }
    }
    
    /**
     * Move all the aliens by x steps horizontally and y steps vertically.
     * @param x the number of steps to move horizontally
     * @param y the number of steps to move vertically
     */
    protected void move(int x, int y)
    {
        for(Alien alien : aliens) {
            alien.setLocation(alien.getX()+x, alien.getY()+y);
        }
    }
    
    /**
     * Return whether there are any more aliens.
     * @return true if there all the aliens are gone.
     */
    protected boolean allGone()
    {
        return aliens.size()==0;
    }
    
    /**
     * Return whether the aliens are moving right or left.
     * @return true if the aliens are moving right.
     */
    protected boolean getDirection()
    {
        return direction;
    }
    
    /**
     * Reverse the direction of travel. If the aliens were moving right, set the direction to left, and vice versa.
     */
    protected void reverseDirection()
    {
        direction = !direction;
    }
    
    /**
     * Return whether the alien block is at the horiztonal edges of the screen.
     * @return true if the aliens are at the edge of the screen.
     */
    protected boolean atEdge()
    {
        for(Alien alien : aliens) {
            if(alien.getX()<=10 || alien.getX()>=world.getWidth()-10 ) return true;
        }
        return false;
    }
    
    /**
     * Return whether the alien block is at the bottom of the screen
     * @return true if the aliens are at the bottom of the screen.
     */
    protected boolean atBottom()
    {
        for(Alien alien : aliens) {
            if(alien.getY() >= world.getHeight() || alien.shipIntersect()!=null) {
                return true;
            }
        }
        return false;
    }
    
    /**
     * Called when the game has ended.
     */
    protected void gameOver()
    {
        world.gameOver();
    }

}
