package edu.buffalo.nsf.hippo;

import java.util.Map;

import org.apache.log4j.Logger;

import edu.buffalo.nsf.hippo.data.config.Config;
import edu.buffalo.nsf.hippo.data.config.ConnectionParams;
import edu.buffalo.nsf.hippo.data.config.DataSourceParams;
import edu.buffalo.nsf.hippo.data.config.DbDescription;
import edu.buffalo.nsf.hippo.data.config.Engine;
import edu.buffalo.nsf.hippo.data.config.EngineDescription;
import edu.buffalo.nsf.hippo.data.config.IntegrityConstraints;
import edu.buffalo.nsf.hippo.data.config.ProverFactory;
import edu.buffalo.nsf.hippo.data.config.Relation;
import edu.buffalo.nsf.hippo.data.config.ResultInterpretation;
import edu.buffalo.nsf.hippo.data.config.ResultMap;
import edu.buffalo.nsf.hippo.data.config.Schema;
import edu.buffalo.nsf.hippo.prolog.Elaboration;
import edu.buffalo.nsf.hippo.util.Util;


/**
 *  Basic implementation based on JAXB components 
 */
public class ConfigConfiguration implements Configuration
{

    protected static Logger logger = Logger.getLogger(ConfigConfiguration.class);

    protected static String getDataSourceName(Config config)
    {
        DbDescription db     = config.getDbDescription();
        DataSourceParams dsp = db.getDataSourceParams();
        return dsp.getName();
    }

    protected static boolean isCached(Config config)
    {
        try {
            DbDescription db     = config.getDbDescription();
            DataSourceParams dsp = db.getDataSourceParams();
            return dsp.isCached();
        } catch (NullPointerException exc) {
            return false;
        }
    }

    protected static int getCacheSize(Config config)
    {
        try {
            DbDescription db     = config.getDbDescription();
            DataSourceParams dsp = db.getDataSourceParams();
            return dsp.getCacheSize();
        } catch (NullPointerException exc) {
            return 0;
        }
    }
    
    protected static Map getSchema(Config config)
    {
        Map           result    = Util.createMap();

        try {
            DbDescription db     = config.getDbDescription();
            Schema        schema = db.getSchema();
            Relation[]    rel    = schema.getRelation();

            for (int i = 0; i < rel.length; i++) {
                result.put(rel[i].getName(),rel[i]);
            }

            return result;

        } catch (NullPointerException exc) {
            logger.error("Can not access schema data from the configuration");
            logger.error("Reason:" + exc);
            return result;
        }
    }

    protected static String getDriverName(Config config)
    {
        DbDescription    db     = config.getDbDescription();
        ConnectionParams params = db.getConnectionParams();

        return params.getDriverName();
    }

    protected static String getConnectURI(Config config)
    {
        DbDescription    db     = config.getDbDescription();
        ConnectionParams params = db.getConnectionParams();

        String dbName = params.getDbName();
        String host   = params.getHost();
        int    port   = params.getPort();
        
        String result = "jdbc:postgresql:";

        if (host != null) {
            result += "//" + host;
            
            if (port != 0) {
                result += ":" + port;
            }

            result += "/";
        }

        return result + dbName;
    }

    protected static String getDBName(Config config)
    {
        DbDescription    db     = config.getDbDescription();
        ConnectionParams params = db.getConnectionParams();

        String dbName = params.getDbName();

        return dbName;
    }

    protected static String getUser(Config config)
    {
        DbDescription    db     = config.getDbDescription();
        ConnectionParams params = db.getConnectionParams();

        return params.getUser(); 
    }
    
    protected static String getPassword(Config config)
    {
        DbDescription    db     = config.getDbDescription();
        ConnectionParams params = db.getConnectionParams();

        return params.getPassword();
    }

    protected static IntegrityConstraints getIntegrityConstraints(Config config)
    {
        return config.getDbDescription().getIntegrityConstraints();
    }

    protected static String getProverFactoryName(Config config)
    {
        EngineDescription edsc = config.getEngineDescription();
        Engine            e    = edsc.getEngine();
        ProverFactory     p    = e.getProverFactory();

        if (p == null) {
            return null;
        }
        
        String      proverName = p.getName();

        return proverName;
    }

    protected static String getEngineName(Config config)
    {
        EngineDescription edsc = config.getEngineDescription();
        Engine            e    = edsc.getEngine();

        return e.getName();
    }


    protected static Map getResultInterpretation(Config config)
    {
        Map ri = Util.createMap();

        EngineDescription edsc = config.getEngineDescription();
        Engine               e = edsc.getEngine();
        ResultInterpretation r = e.getResultInterpretation();

        if (r == null) {
            return null;
        }
        
        for (int i = 0; i < r.getResultMapLength(); i++) {
            ResultMap map = r.getResultMap(i);
            
            Elaboration key   = Elaboration.create(map.getValue());
            Boolean     value = new Boolean(map.isAccepted());
            
            ri.put(key,value);
        }        

        return ri;
    }

    protected static String getCoreSuffix(Config config)
    {
        EngineDescription edsc = config.getEngineDescription();
        Engine               e = edsc.getEngine();
        
        return e.getCoreSuffix();
    }

    protected static String getTranslatorName(Config config)
    {
        EngineDescription edsc = config.getEngineDescription();
        Engine               e = edsc.getEngine();
        
        return e.getTranslatorName();
    }

    protected static String getCoreOperatorName(Config config)
    {
        EngineDescription edsc = config.getEngineDescription();
        Engine               e = edsc.getEngine();
        
        return e.getCoreOperatorName();
    }
    
    protected static String getShellOperatorName(Config config)
    {
        EngineDescription edsc = config.getEngineDescription();
        Engine               e = edsc.getEngine();
        
        return e.getShellOperatorName();
    }

    protected static String getEnvelopeOperatorName(Config config)
    {
        EngineDescription edsc = config.getEngineDescription();
        Engine               e = edsc.getEngine();
        
        return e.getEnvelopeOperatorName();
    }
    

    protected String  sourceName;
    protected boolean isCached;
    protected int     cacheSize;

    protected Map     schema;
    protected String  dbName;
    protected String  driverName;
    protected String  connectURI;
    protected String  user;
    protected String  password;
    protected IntegrityConstraints ics;


    protected String  engineName;
    protected String  factoryName;
    protected Map     ri;
    
    protected String coreSuffix;
    protected String translatorName;
    protected String coreOperatorName;
    protected String shellOperatorName;
    protected String envelopeOperatorName;


    public ConfigConfiguration(Config config)
    {
        this.sourceName = getDataSourceName (config);
        this.isCached   = isCached          (config);
        this.cacheSize  = getCacheSize      (config);

        this.schema     = getSchema               (config);
        this.dbName     = getDBName               (config);
        this.driverName = getDriverName           (config);
        this.connectURI = getConnectURI           (config);
        this.user       = getUser                 (config);
        this.password   = getPassword             (config);
        this.ics        = getIntegrityConstraints (config);

        this.engineName  = getEngineName           (config);
        this.factoryName = getProverFactoryName    (config);
        this.ri          = getResultInterpretation (config);
        this.coreSuffix  = getCoreSuffix(config);

        this.translatorName        = getTranslatorName   (config);
        this.coreOperatorName      = getCoreOperatorName (config);
        this.shellOperatorName     = getShellOperatorName (config); 
        this.envelopeOperatorName  = getEnvelopeOperatorName(config); 
    }

    public String getDataSourceName()
    {
        return sourceName;
    }

    public boolean isDataSourceCached()
    {
        return isCached;
    }

    public int getDataSourceCacheSize()
    {
        return cacheSize;
    }

    public Map getSchema()
    {
        return schema;
    }

    public String getDBName()
    {
        return dbName;
    }

    public String getDBConnectURI()
    {
        return connectURI;
    }

    public String getDBUser()
    {
        return user;
    }

    public String getDBPassword()
    {
        return password;
    }

    public String getDBDriverName()
    {
        return driverName;
    }

    public IntegrityConstraints getIntegrityConstraints()
    {
        return ics;
    }

    public String getTranslatorName()
    {
        return translatorName;
    }

    public String getProverFactoryName()
    {
        return factoryName;
    }

    public String getCoreSuffix()
    {
        return coreSuffix;
    }

    public String getEngineName()
    {
        return engineName;
    }

    public Map getResultInterpretation()
    {
        return ri;
    }

    public String getCoreOperatorName()
    {
        return coreOperatorName;
    }
    
    public String getShellOperatorName()
    {
        return shellOperatorName;
    }

    public String getEnvelopeOperatorName()
    {
        return envelopeOperatorName;
    }
}
