package edu.buffalo.nsf.hippo;

import java.lang.reflect.Constructor;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.SQLException;
import java.util.Iterator;

import org.apache.log4j.Logger;

import edu.buffalo.nsf.hippo.data.CachedDataSource;
import edu.buffalo.nsf.hippo.data.DataSource;
import edu.buffalo.nsf.hippo.data.DataSourceCache;
import edu.buffalo.nsf.hippo.data.query.traverse.QueryTransformer;
import edu.buffalo.nsf.hippo.data.query.traverse.QueryTranslator;
import edu.buffalo.nsf.hippo.prolog.ProverFactory;
import edu.buffalo.nsf.hippo.util.ConfigurableContainer;
import edu.buffalo.nsf.hippo.util.DataSourceContainer;

/**
 *  Environment object.
 */
public class EnvironmentImpl implements Environment
{
    protected static Logger logger = Logger.getLogger(EnvironmentImpl.class);

    protected DataSourceContainer   sources;
    protected ConfigurableContainer cs;

    public EnvironmentImpl()
    {
        sources = new DataSourceContainer(100);
        cs      = new ConfigurableContainer(100);
    }

    public void shutdown()
    {
        logger.info("Shutting down the enviroment.");

        Iterator ic = cs.iterator();

        while (ic.hasNext()) {
            ((Configurable) ic.next()).shutdown();
        }

        Iterator ds = sources.iterator();
        
        while (ds.hasNext()) {
            ((Configurable) ds.next()).shutdown();
        }
        
        sources = null;

        logger.info("Environment shutdown complete.");
    }


    public Connection getConnection(Configuration config)
    {
        try {
            Class.forName(config.getDBDriverName());
            
            String connectURI = config.getDBConnectURI();
            String user       = config.getDBUser();
            String password   = config.getDBPassword();
            
            return DriverManager.getConnection(connectURI,user,password);
        } catch (Throwable t) {
            throw new InternalProblem("Exception caught while creating " + 
                                      "a new database connection.",t);
        }
    }

    protected DataSource createDataSource(Configuration config)
    {
        logger.info("creating datasource!");

        Configurable s = getConfigurable(config.getDataSourceName(),config);

        return (DataSource) s;
    }

    public DataSource getDataSource (Configuration config)
    {
        synchronized (sources) {
            if (! sources.contains(config)) {
                sources.put(config,createDataSource(config));
            }
        }

        DataSource s = sources.get(config);

        if (config.isDataSourceCached()) {
            int cacheSize = config.getDataSourceCacheSize();
            
            DataSourceCache cache    = new DataSourceCache(cacheSize); 

            s = new CachedDataSource(s,cache);
        }

        logger.info("dispatching data source " + s);
        
        return s;
    }


    protected void createConfigurable(String name, Configuration config)
    {        
        try {            
            //logger.info("creating configurable " + name);
            
            Class c = Class.forName(name);

            Class[] paramTypes = new Class[1];
            paramTypes[0] = Environment.class;
            
            Constructor constr = c.getConstructor(paramTypes);
            
            Object[] params = new Object[1];
            params[0] = this;
            
            Configurable o = (Configurable) constr.newInstance(params);            
            
            cs.put(name,config,o);

            o.init(config);
            
        } catch (Exception exc) {
            logger.error("Exception caught while creating Configurable instance (" + 
                         name + ")", exc);
            throw new InternalProblem(exc);
        }
    }

    protected Configurable getConfigurable(String name, Configuration config)
    {
        synchronized (cs) {
            if (! cs.contains(name,config)) {
                createConfigurable(name,config);
            }
        }

        return cs.get(name,config);
        
    }


    public ProverFactory getProverFactory(String name, Configuration config)
    {
        return (ProverFactory) getConfigurable(name,config);
    }

    public ProverFactory getProverFactory(Configuration config)
    {
        return getProverFactory(config.getProverFactoryName(),config);
    }

    public Engine getEngine(Configuration config)
    {        
        return (Engine) getConfigurable(config.getEngineName(),config);
    }

    public QueryTranslator getTranslator(Configuration config)
    {
        return (QueryTranslator) getConfigurable(config.getTranslatorName(),config);
    }
    
    public QueryTransformer getCoreOperator(Configuration config)
    {
        return (QueryTransformer) getConfigurable(config.getCoreOperatorName(),config);
    }
    
    public QueryTransformer getShellOperator(Configuration config)
    {
        return (QueryTransformer) getConfigurable(config.getShellOperatorName(),config);
    }

    public QueryTransformer getEnvelopeOperator(Configuration config)
    {
        return (QueryTransformer) getConfigurable(config.getEnvelopeOperatorName(),config);
    }

    public void dispose(Configurable c)
    {
        logger.info("disposing configurable object " + c);
        
        if (! (c instanceof DataSource)) {
            c.shutdown();
        }
    }

    public void release(Connection db)
    {
        try {
            db.close();
        } catch (SQLException exc) {
            logger.error("Exception caught.",exc); 
        }        
    }
}

