package edu.buffalo.nsf.hippo.data;

import java.util.Map;

import org.apache.commons.collections.LRUMap;
import org.apache.log4j.Logger;

import edu.buffalo.nsf.hippo.data.graph.Vertex;
import edu.buffalo.nsf.hippo.util.Util;

/**
 *  Stores information about the vertices contained by datasource.
 */
public class DataSourceCache
{   

    protected static Logger logger = Logger.getLogger(DataSourceCache.class);

    /**
     *  Maps vertices into Boolean values.
     */
    protected Map vertices;


    public DataSourceCache()
    {
        this.vertices = Util.createMap();
    }

    public DataSourceCache(DataSourceCache cache1, DataSourceCache cache2)
    {
        this.vertices = Util.createMap();

        this.vertices.putAll(cache1.vertices);
        this.vertices.putAll(cache2.vertices);
    }

    public DataSourceCache(int capacity)
    {
        if (capacity == 0) {
            this.vertices = Util.createMap();
        } else {
            this.vertices = new LRUMap(capacity);
        }
    }

    /**
     *  Checks if information on a given vertex is contained by this cache.
     */
    public boolean check(Vertex v)
    {
        return vertices.containsKey(v);
    }

    /**
     *  Stores particular information on a 
     *  given vertex in this cache. 
     *  The old information (if nay) is being disposed. 
     */
    public void store(Vertex v, boolean b)
    {
        vertices.put(v,new Boolean(b));
    }


    /** 
     *   Returns Boolean value of information 
     *   corresponding to a given vertex. If no information is
     *   stored on the given tuple, then <code>null</code> 
     *   value is returned.
     */
    public Boolean search(Vertex v)
    {
        return (Boolean) vertices.get(v);
    }

    
    /** 
     *   Returns <code>boolean</code> value of information 
     *   corresponding to a given vertex. If no information is
     *   stored on the given tuple, then 
     *   <code>IllegalArgumentException</code> exception is thrown.
     */   
    public boolean read(Vertex v)
    {
        try {
            return search(v).booleanValue();
        } catch (NullPointerException exc) {
            throw new IllegalArgumentException( "Information on given vertex " + 
                                                "is not contained in this cache.");
        }
    }

    /**
     *  Returns number of entires contained in this cache. 
     */
    public int size()
    {
        return vertices.size();
    }

    /**
     *  Removes all entries for this cache.
     */
    public void clear()
    {
        vertices.clear();
    }

    public String toString()
    {
        return "Cache: " + vertices;
    }
}
