package edu.buffalo.nsf.hippo.data.query.traverse;

import edu.buffalo.nsf.hippo.data.query.Difference;
import edu.buffalo.nsf.hippo.data.query.FromExpr;
import edu.buffalo.nsf.hippo.data.query.FromList;
import edu.buffalo.nsf.hippo.data.query.FromRelation;
import edu.buffalo.nsf.hippo.data.query.FromSubquery;
import edu.buffalo.nsf.hippo.data.query.GroupByList;
import edu.buffalo.nsf.hippo.data.query.Intersection;
import edu.buffalo.nsf.hippo.data.query.Query;
import edu.buffalo.nsf.hippo.data.query.Select;
import edu.buffalo.nsf.hippo.data.query.SelectList;
import edu.buffalo.nsf.hippo.data.query.Union;
import edu.buffalo.nsf.hippo.data.query.WhereCompValue;
import edu.buffalo.nsf.hippo.data.query.WhereComparator;
import edu.buffalo.nsf.hippo.data.query.WhereComparison;
import edu.buffalo.nsf.hippo.data.query.WhereCondition;
import edu.buffalo.nsf.hippo.data.query.WhereConjunction;
import edu.buffalo.nsf.hippo.data.query.WhereConstant;
import edu.buffalo.nsf.hippo.data.query.WhereDisjunction;
import edu.buffalo.nsf.hippo.data.query.WhereNegation;
import edu.buffalo.nsf.hippo.data.query.WhereSelectValue;

public abstract class PhonyTransformer implements QueryTransformer
{
    
    public Query transform(Query q)
    {
        if (q instanceof Select ) {
            Select s = (Select) q;
            
            return transformSelect(s.getSelectList(), 
                                   s.getFromList(), 
                                   s.getWhereCondition(),
                                   s.getGroupByList());
        } else if (q instanceof Difference) {
            Difference d = (Difference) q;
            
            return transformDifference(d.getExpr1(),d.getExpr2());
        } else if (q instanceof Intersection) {
            Intersection i = (Intersection) q;
            
            return transformIntersection(i.getExpr1(), i.getExpr1());
        } else if (q instanceof Union) {
            Union u = (Union) q;
            
            return transformUnion(u.getExpr1(),u.getExpr2());
        } else {
            throw new IllegalArgumentException("Unknown Query subclass.");
        }
    }

    protected FromExpr transform(FromExpr fromExpr)
    {
        if (fromExpr instanceof FromRelation) {
            FromRelation r = (FromRelation) fromExpr;

            return transformFromRelation(r.getRelationName(),
                                         r.getBindingName());
        } else if (fromExpr instanceof FromSubquery) {
            FromSubquery s = (FromSubquery) fromExpr;

            return transformFromSubquery(s.getSubquery(),
                                         s.getBindingName());
        } else {
            throw new IllegalArgumentException("Unknown FromExpr subclass.");
        }
    }

    protected WhereCondition transform(WhereCondition whereCondition)
    {
        if (whereCondition == null) {
            return null;
        } else if (whereCondition instanceof WhereComparison) {
            WhereComparison cmp = (WhereComparison) whereCondition;
            
            return transformWhereComparison(cmp.getComparedValue1(),
                                     cmp.getComparator(),
                                     cmp.getComparedValue2());
        } else if (whereCondition instanceof WhereConjunction) {
            WhereConjunction c = (WhereConjunction) whereCondition;
            
            return transformWhereConjunction(c.getCondition1(),c.getCondition2());
        } else if (whereCondition instanceof WhereDisjunction) {
            WhereDisjunction d = (WhereDisjunction) whereCondition;
            
            return transformWhereDisjunction(d.getCondition1(),d.getCondition2());
        } else if (whereCondition instanceof WhereNegation) {
            WhereNegation n = (WhereNegation) whereCondition;
            
            return transformWhereNegation(n.getCondition());
        } else {
            throw new IllegalArgumentException("Unknown WhereCondition subclass.");
        }
    }

    protected WhereCompValue transform(WhereCompValue val)
    {
        if (val instanceof WhereSelectValue) {
            WhereSelectValue s = (WhereSelectValue) val;

            return transformWhereSelectValue(s.getBindingName(),s.getFieldName());
        } else if (val instanceof WhereConstant) {
            WhereConstant c = (WhereConstant) val;
            
            return transformWhereConstant(c.getConstant());
        } else {
            throw new IllegalArgumentException("Unknown WhereCompValue subclass.");
        }
    }

    protected Query transformSelect(SelectList selectList,
                                 FromList fromList,
                                 WhereCondition whereCondition,
                                 GroupByList groupByList)
    {
        if (groupByList != null) {
            throw new IllegalArgumentException(
              "This translator can be used only for SJUID queires."
            );
        }

        return new Select(selectList,
                          transformFromList(fromList.getFromExpr()),
                          transform(whereCondition),
                          groupByList);
    }

    protected FromList transformFromList(FromExpr[] fromExpr)
    {
        FromExpr[] f = new FromExpr[fromExpr.length];
        
        for (int i = 0; i < fromExpr.length; i++) {
            f[i] = transform(fromExpr[i]);
        }
        
        return new FromList(f);
    }

    protected FromExpr transformFromRelation(String relName, String bindName)
    {
        return new FromRelation(relName,bindName);
    }

    protected FromSubquery transformFromSubquery(Query q, String bindName)
    {
        return new FromSubquery(transform(q),bindName);
    }

    protected WhereCondition transformWhereConjunction(WhereCondition cond1,
                                          WhereCondition cond2)
    {
        return new WhereConjunction(transform(cond1), transform(cond2));
    }

    protected WhereCondition transformWhereDisjunction(WhereCondition cond1,
                                          WhereCondition cond2)
    {
        return new WhereDisjunction(transform(cond1), transform(cond2));
    }

    protected WhereCondition transformWhereNegation(WhereCondition cond)
    {
        return new WhereNegation(transform(cond));
    }


    protected WhereCondition transformWhereComparison(WhereCompValue val1, 
                                                   WhereComparator comp,
                                                   WhereCompValue val2)
    {
        WhereCompValue v1 = transform(val1);
        WhereCompValue v2 = transform(val2);

        return new WhereComparison(v1,comp,v2);
    }

    protected WhereCompValue transformWhereSelectValue(String bindName, String fieldName)
    {
        return new WhereSelectValue(bindName,fieldName);
    }

    protected WhereCompValue transformWhereConstant(Object c)
    {
        return new WhereConstant(c);
    }

    protected Query transformDifference (Query expr1, Query expr2)
    {
        return new Difference(transform(expr1),transform(expr2));
    }

    protected Query transformIntersection (Query expr1, Query expr2)
    {
        return new Intersection(transform(expr1),transform(expr2));
    }

    protected Query transformUnion (Query expr1, Query expr2)
    {
        return new Union(transform(expr1),transform(expr2));
    }
}
