package edu.buffalo.nsf.hippo.prolog;

import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.log4j.Logger;

import edu.buffalo.nsf.hippo.data.formula.Atom;
import edu.buffalo.nsf.hippo.data.formula.Conjunction;
import edu.buffalo.nsf.hippo.data.formula.Disjunction;
import edu.buffalo.nsf.hippo.data.formula.Formula;
import edu.buffalo.nsf.hippo.data.formula.Negation;
import edu.buffalo.nsf.hippo.data.graph.Edge;
import edu.buffalo.nsf.hippo.data.graph.Vertex;
import edu.buffalo.nsf.hippo.data.query.traverse.SJUIDTranslator;
import edu.buffalo.nsf.hippo.data.tuple.Tuple;
import edu.buffalo.nsf.hippo.util.Util;


public final class HProver extends Prover
{
    protected static Logger logger = Logger.getLogger(HProver.class);

    protected Formula   f;
    protected ProverEnv basic;

    public HProver(Formula f, ProverEnv basic)
    {
        this.f     = f.toCNF();
        this.basic = basic;
    }

    public Elaboration elaborate(Tuple t)
    {
        return elaborate(basic,t);
    }

    public Elaboration elaborate(ProverEnv env, Tuple t)
    {
        Map s = SJUIDTranslator.createSubstitution(t);

//         logger.debug("before subst.:" + f);

//         logger.debug("tuple is: " + t);
//         logger.debug("Substitution is: " + s);

        Formula g = f.substitute(s);
        
//         logger.debug("after subst.:" + g);

        return Elaboration.create(isConsistent(env,g));
    }

    protected boolean isConsistent(ProverEnv env, Formula g)
    {
        try {
            List conjs = ((Conjunction)  g).getConjuncts();

            boolean result = true;

            for (int i = 0; result && i < conjs.size(); i++) {
                Disjunction d = (Disjunction) conjs.get(i);

                result = result && ! isInconsistent(env,d);
            }

            return result;

        } catch (ClassCastException exc) {
            logger.error("class cast exception",exc);
            throw new IllegalArgumentException(
                  "Formula not in CNF or " +
                  "wrong work of the negation operation."
            );
        }
    }
    
    protected boolean isInconsistent(ProverEnv env, Disjunction d)
    {
        boolean result = true;

        Conjunction c = (Conjunction) d.negate();

        Set  pos = toVertexSet  (getPositiveLiterals(c));
        List neg = toVertexList (getNegativeLiterals(c));

        result = result && env.getHypergraph().contains(pos);

        result = result && consistentModelCheck(env,neg,pos);

        return result;        
    }


    /**
     *  @param negative - set of vertices from negative literals that
     *  have to be processed.
     *  @param test - set of vertices that should be tested against
     *  independence.
     */
    protected boolean consistentModelCheck(ProverEnv env, List negative, Set s)
    {
        boolean result = false;

        if (negative.isEmpty()) {
            result = env.getHypergraph().isIndependent(s);

        } else {
            Vertex v = (Vertex) negative.remove(0);

            if (! env.getHypergraph().contains(v)) {
                result = consistentModelCheck(env,negative,s);

            } else {
                Set edges = env.getHypergraph().incident(v);
                
                //what if edges empty???? -- false :. unable to build consistent model.

                boolean exists = false;

                Iterator iter = edges.iterator();

                while (iter.hasNext() && ! exists) {
                    Edge e = (Edge) iter.next();
                    
                    Set tmp = Util.union(s,e.getNeighbors(v));
                    
                    exists = exists || consistentModelCheck(env,negative,tmp);
                    
                }

                result = exists;
            }

        }

        return result;

    }

    protected List getPositiveLiterals(Conjunction c)
    {
        List cts = c.getConjuncts();
        List ls = Util.createList();

        for (int i = 0; i < cts.size(); i++) {
            if (cts.get(i) instanceof Atom) {
                ls.add(cts.get(i));
            }
        }

        return ls;
    }

    protected List getNegativeLiterals(Conjunction c)
    {
        List cts = c.getConjuncts();
        List ls = Util.createList();

        for (int i = 0; i < cts.size(); i++) {
            if (cts.get(i) instanceof Negation) {
                ls.add(((Negation)cts.get(i)).getNegatedFormula());
            }
        }

        return ls;        
    }

    protected List toVertexList(List gs)
    {
        List vs = Util.createList(gs.size());

        for (int i = 0; i < gs.size(); i++) {
            Atom g = (Atom) gs.get(i);
            String name = g.getRelationName();
            Tuple  t    = g.getValueVector().toTuple();

            vs.add(Vertex.create(name,t));
        }

        return vs;
    }

    protected Set toVertexSet(List gs)
    {
        return Util.toSet(toVertexList(gs));
    }
}
