package edu.buffalo.nsf.hippo.util;

import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Logger;

import edu.buffalo.nsf.hippo.Configurable;
import edu.buffalo.nsf.hippo.Configuration;

public class ConfigurableContainer
{
    protected Logger logger = Logger.getLogger(ConfigurableContainer.class);

    protected class CKey
    {
        protected Configuration config;
        protected int           hashCode;
        
        public CKey(Configuration config)
        {
            this.config   = config;            
            this.hashCode = HashUtil.hashCode(config);
        }

        public int hashCode()
        {
            return hashCode;
        }

        public boolean equals(Object o)
        {
            boolean result;
            if (o instanceof CKey) {
                result =  HashUtil.equalEngine(config,((CKey)o).config);
            } else {
                result =  false;
            }

            return result;
        }

        public String toString()
        {
            return "key:" + hashCode; 
        }        
    }

    /**
     *  This map holds associations between Configuration objects and 
     *  corresponding CKey objects corresponging. This map allows to  
     *  avoid overhead caused by expensive creating of CKey object.
     *  This map is a map with limited capacity, as there can be 
     *  unlimited number of Configuration objects refering to the 
     *  same DataSource object, what may lead to exhaust of resources.     
     */
    protected Map keys;

    /**
     *  This map holds associations between CKey objects and 
     *  Configurable objects corresponding to Configuration obejct 
     *  encapsulated in CKey objects.
     */
    protected Map maps;

    public ConfigurableContainer(int rawMaxCapacity)
    {
        if (! (rawMaxCapacity > 0)) {
            throw new IllegalArgumentException(
                "rawMaxCapacity must be a positive integer " + 
                "in order to this object work properly.");
        }
        this.keys = Util.createLRUMap(rawMaxCapacity);
        this.maps   = Util.createMap();
    }

    public CKey key(Configuration config)
    {
        if (! keys.containsKey(config)) {
            keys.put(config, new CKey(config));
        }

        return (CKey) keys.get(config);
    }

    protected Map map(Configuration config)
    {
        CKey key = key(config);

        if (! maps.containsKey(key)) {
            maps.put(key, Util.createMap());
        }

        return (Map) maps.get(key);
    }

    public boolean contains(String name, Configuration config)
    {
        return maps.containsKey(key(config)) && map(config).containsKey(name);
    }

    public Configurable get(String name, Configuration config)
    {
        return (Configurable) map(config).get(name);
    }

    public void remove(Configuration config)
    {
        maps.remove(key(config));
    }

    public void put(String name, Configuration config, Configurable c)
    {
        //logger.debug("\nPutting : name=" + name  + 
        //                     "\n        config=" + config + 
        // "\n          c   =" + c);
                     
        map(config).put(name,c);
    }

    public Iterator iterator()
    {
        List vals = Util.createList();
        Iterator i = maps.values().iterator();

        while (i.hasNext()) {
            Map m = (Map) i.next();
            
            vals.addAll(m.values());
        }
        return vals.iterator();
    }
}
