package edu.buffalo.nsf.hippo.data;


import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Map;
import java.util.Set;

import org.apache.log4j.Logger;

import edu.buffalo.nsf.hippo.Configuration;
import edu.buffalo.nsf.hippo.Environment;
import edu.buffalo.nsf.hippo.data.config.Bind;
import edu.buffalo.nsf.hippo.data.config.DenialConstraint;
import edu.buffalo.nsf.hippo.data.config.Field;
import edu.buffalo.nsf.hippo.data.config.IntegrityConstraints;
import edu.buffalo.nsf.hippo.data.config.Relation;
import edu.buffalo.nsf.hippo.data.graph.Edge;
import edu.buffalo.nsf.hippo.data.graph.Hypergraph;
import edu.buffalo.nsf.hippo.data.graph.InternalHypergraphImpl;
import edu.buffalo.nsf.hippo.data.graph.Vertex;
import edu.buffalo.nsf.hippo.data.query.Query;
import edu.buffalo.nsf.hippo.data.tuple.Tuple;
import edu.buffalo.nsf.hippo.data.tuple.TupleStream;
import edu.buffalo.nsf.hippo.util.FormatUtil;
import edu.buffalo.nsf.hippo.util.Util;

public class SQLDataSource implements DataSource
{
    private static Logger logger = Logger.getLogger(SQLDataSource.class);

    protected Environment          env;
    protected Connection           db;
    protected Map                  schema;
    protected IntegrityConstraints ics;
    protected Hypergraph           h;

    public SQLDataSource(Environment env)
    {
        this.env = env;
    }

    public void init(Configuration config)
    {
        this.schema = config.getSchema();
        this.db     = env.getConnection(config);
        this.ics    = config.getIntegrityConstraints();
    }

    public void shutdown()
    {
        env.release(db);
        this.h = null;
     }

    protected TupleStream executeSQLQuery(String sqlQuery)
    {
        try {
            Statement stmt = db.createStatement();

            //logger.debug("Executing SQL query: " + sqlQuery);

            ResultSet rs = stmt.executeQuery(sqlQuery);

            //StatsManager.peekStats().touch(Stats.QUERIES_EXEC);

            TupleStream result = TupleStream.create(rs,stmt);

//             rs.close();

//             stmt.close();
            
            //logger.debug("Query result: "  + result);

            return result;
        } catch (SQLException exc) {
            logger.error("Exception caught while executing the query: " + sqlQuery ,exc);
            throw new DataSourceError("SQLException caught",exc);
        }        
    }

    public TupleStream executeQuery(Query query)
    {
        //TODO:: implement the whole support for differen database systems
        return executeSQLQuery(query.toString());
    }

    protected TupleStream detectConflicts(DenialConstraint dc)
    {
        String sqlQuery = "SELECT * FROM ";
        Bind[] b = dc.getBinding();

        if (b == null || b.length == 0) {
            logger.warn("Constraint with no bindings.");
            return TupleStream.EMPTY_TUPLE_STREAM;
        }

        for (int i = 0; i < b.length; i++) {
            
            sqlQuery += b[i].getRelationName() + " " + b[i].getVariableName();
            
            if (i < b.length - 1) {
                sqlQuery += ", ";
            }            
        }

        sqlQuery += " ";

        sqlQuery += "WHERE NOT (" + dc.getDefinition() + ")";

        return executeSQLQuery(sqlQuery);
    }

    /** For memoization purposes.*/
    protected Hypergraph _getHypergraph()
    {
        logger.debug("Creating hypergraph!");

        //Date start = new Date();

        //StatsManager.pushStats(Stats.HYPERGRAPH_STATS);

        DenialConstraint[] dc = ics.getIntegrityConstraint();
        Set edges = Util.createSet();

        for (int i = 0; i < dc.length; i++) {
            TupleStream ts = detectConflicts(dc[i]);

            while (ts.hasNext()) {
                edges.add(Edge.create(dc[i],schema,ts.getNext()));
            }
        }


        Hypergraph h = new InternalHypergraphImpl(edges);
        
        //Date stop = new Date();

        //Date rt = new Date(stop.getTime() - start.getTime());
        
        //DateFormat f = new SimpleDateFormat("m:ss.SSS");
        
        //logger.warn("Hypergraph created in: " + f.format(rt) +"\n");

        //StatsManager.popStats();

        return h;
    }

    public synchronized Hypergraph getHypergraph()
    {
        if (h == null) {
            h = _getHypergraph();
        }

        return h;
    }

    public boolean checkVertex (Vertex v)
    {

        String relName = v.getRelationName();
        Tuple  t       = v.getTuple();
            


        Relation r = (Relation) schema.get(relName);
        String   query = "SELECT COUNT(*) FROM " + relName + " WHERE ";

        if (r == null) {
            logger.warn("Trying to check tuple of relation " + relName + " " +
                        "not contained in schema data.");
            
            return false;
        }

        Field[] f = r.getField();

        for (int i = 0; i < f.length; i++) {
        	if (t.getObject(i) != null){
        		query += relName + "." + f[i].getName() + " = ";
        		if (t.getObject(i) instanceof String) {
        			query += FormatUtil.escapeSQLString(t.getObject(i).toString());
        		} else {
        			query += t.getObject(i).toString();
        		}        		
            } else {
        		query += relName + "." + f[i].getName() + " is NULL";            	
            }

            if (i < f.length - 1) {
                query += " AND ";
            }
        }


        TupleStream ts = executeSQLQuery(query);
        
        if (! ts.hasNext()) {
            throw new InternalError("Inappropriate SQL backend behavior.");
        }

        return ts.getNext().getInt(0) != 0;
    }
}
