package edu.buffalo.nsf.hippo.util;

import java.util.Iterator;
import java.util.Map;

import edu.buffalo.nsf.hippo.Configuration;
import edu.buffalo.nsf.hippo.data.DataSource;

public class DataSourceContainer
{
    protected class DBKey
    {
        protected Configuration config;
        protected int           hashCode;
        
        public DBKey(Configuration config)
        {
            this.config   = config;            
            this.hashCode = HashUtil.hashCodeDB(config);
        }

        public int hashCode()
        {
            return hashCode;
        }

        public boolean equals(Object o)
        {
            boolean result;
            if (o instanceof DBKey) {
                result =  HashUtil.equalDB(config,((DBKey)o).config);
            } else {
                result =  false;
            }

            return result;
        }

        public String toString()
        {
            return "key:" + hashCode; 
        }        
    }

    /**
     *  This map holds associations between Configuration objects and 
     *  corresponding DBKey objects corresponging. This map allows to 
     *  avoid overhead caused by expensive creating of DBKey object.
     *  This map is a map with limited capacity, as there can be 
     *  unlimited number of Configuration objects refering to the 
     *  same DataSource object, what may lead to exhaust of resources.     
     */
    protected Map keys;

    /**
     *  This map holds associations between DBKey objects and 
     *  DataSource objects corresponding to Configuration obejct 
     *  encapsulated in DBKey objects.
     */
    protected Map sources;

    public DataSourceContainer(int rawMaxCapacity)
    {
        if (! (rawMaxCapacity > 0)) {
            throw new IllegalArgumentException(
                "rawMaxCapacity must be a positive integer " + 
                "in order to this object work properly.");
        }
        this.keys    = Util.createLRUMap(rawMaxCapacity);
        this.sources = Util.createMap();
    }

    public DBKey key(Configuration config)
    {
        if (! keys.containsKey(config)) {
            keys.put(config, new DBKey(config));
        }

        return (DBKey) keys.get(config);
    }

    public boolean contains(Configuration config)
    {
        return sources.containsKey(key(config));
    }

    public DataSource get(Configuration config)
    {
        return (DataSource) sources.get(key(config));
    }

    public void remove(Configuration config)
    {
        sources.remove(key(config));
    }

    public void put(Configuration config, DataSource source)
    {
        sources.put(key(config),source);
    }

    public Iterator iterator()
    {
        return sources.values().iterator();
    }
}
