//This is mostly adapted from Mahmoud's chained hash code.

#ifndef CHAINEDHASHJOS_H_
#define CHAINEDHASHJOS_H_

#include "DListJOS.h"
#include <vector>
#include <string>

using std::vector;
using std::string;

// CLASS INV: The find() method always returns a pointer from the vector to the hash.
//            Hash() is only ever called within this class in order to establish the
//            location of the desired bucket.
template <typename I>
class ChainedHash
{
public:
	class iterator
	{
		typename DList<I>::iterator list_iter;

		iterator(const typename DList<I>::iterator& l_iter)
			:list_iter(l_iter){}

		friend class ChainedHash<I>;

	public:
		iterator(const iterator& other)
			:list_iter(other.list_iter){}

		iterator& operator=(const iterator& rhs)
		{
			list_iter = rhs.list_iter;
			return *this;
		}

		I& operator*() const
		{
			return *list_iter;
		}

		iterator& operator++()
		{
			++list_iter;
			return *this;
		}

		iterator operator++(int)
		{
			iterator oldMe = *this;
			list_iter++;
			return oldMe;
		}

		bool operator==(const iterator& rhs) const
		{
			return list_iter == rhs.list_iter;
		}

		bool operator!=(const iterator& rhs) const
		{
			return list_iter != rhs.list_iter;
		}
	};

private:
	DList<I> list;
	vector<iterator> vec;
	size_t n, buckets;

public:
	explicit ChainedHash(size_t buckets = 10000)
		:list(DList<I>())
		,vec(vector<iterator>())
		,n(0), buckets(buckets)
	{
		for (size_t bucket = 0; bucket < buckets; bucket++)
		{
			iterator iter = iterator(list.insert(list.end(), I()));
			vec.push_back(iter);
		}
		vec.push_back(iterator(list.end()));
	}

	~ChainedHash(){}

	iterator begin() const
	{
		return iterator(list.begin());
	}

	iterator end() const
	{
		return iterator(list.end());
	}

    // REQ: This find method requires a working hash method in order to function.
    // ENS: Nothing is changed within the hash, a pointer to the sought after ticker is returned.
    // LOOP INV: 'iter' is never equal to a value less than or greater than the value hashed from the ticker. Thus the
    //            loop exits when the entire bucket has been checked.
	iterator find(const I& item) const
	{
        size_t h = hash(item) % buckets;
        for (iterator iter = vec[h]; iter != vec[h+1]; iter++)
            if (*iter == item)
                return iter;

        return end();
    }

	iterator insert(const I& item)
	{
		size_t h = hash(item) % buckets;
        for (iterator iter = vec[h]; iter != vec[h+1]; iter++)
			if (*iter == item)
				return iter;
		++n;
		return iterator(list.insert(vec[h+1].list_iter, item));
	}

	size_t size() const
	{
		return n;
	}

	bool empty() const
	{
		return size() == 0;
	}

	string str() const
	{
		return list.toString();
	}

    // REQ: This find method requires a working toString() method to hash correctly.
    // ENS: A hash is returned based on the positional values of the characters within the string.
    //      The 31 is present as a prime in order to prevent the hash getting stuck, or 'fed up.'
    // LOOP INV: The hashed value returned is always 0 - 9,999. The loop index is always less than the size
    //           of the string, thus only ever iterates through the full size of said string.
    size_t hash(const I& item) const
    {
        string x = item.toString();
        for (size_t i, hash = 0; i < x.size(); i++)
        {
            hash = hash * 31 + x[i];
            return hash;
        }
    }
};

#endif

