#include <cstdlib>
#include <fstream>
#include <iostream>
#include <math.h>
#include <math.h>
#include <sstream>
#include <time.h>
#include <vector>

#include "ChainedHashJOS.h"
#include "CommandAliasANB.h"
#include "HeapANB.h"
#include "StockANB.h"

using std::cout;
using std::endl;
using std::getline;
using std::ifstream;
using std::string;
using std::stringstream;
using std::vector;

size_t processTokens(const vector<string>& tokens)
{
    if ((tokens[0])[tokens[0].length() - 1] == '?')
    {
        if (tokens[0] == "add?") {return ADD;}
        else if (tokens[0] == "printAll?") {return PRINT;}
        else if (tokens[0] == "printTopByVolume?") {return PRINTVOLUME;}
        else if (tokens[0] == "printTopByPercentUp?") {return PRINTUP;}
        else if (tokens[0] == "printTopByPercentDown?") {return PRINTDOWN;}
        else if (tokens[0] == "printTopByPercentChange?") {return PRINTCHANGE;}
        else if (tokens[0] == "printTopByMomentum?") {return PRINTMOMENTUM;}
        else if (tokens[0] == "printTopByTrendTrades?") {return PRINTTREND;}
        // This one is the extra credit
        //else if (tokens[0] == "printTopByTrendShares?") {return PRINTTRENDSHARES;}
        // ----------------------------
        else if (tokens[0] == "pause?") {return PAUSE;}
        else {return UNRECOGNIZED;}
    }
    else
    {
        if (tokens.size() == 2)
        {
            return TRADE;
        }
        else
        {
            return UNRECOGNIZED;
        }
    }
}

vector<string> parseTokens(const string& s, char delimiter)
{
    vector<string> vec;
    stringstream* parser = new stringstream;

    for (size_t i = 0; i < s.length(); i++)
    {
        if (s[i] != delimiter) {
            *parser << s[i];
        }
        else
        {
            vec.push_back(parser->str());
            parser = new stringstream;
        }
    }
    if (parser->str() != "") {vec.push_back(parser->str());}

    return vec;
}

double readNumberFromString(const string& s)
{
    double number = 0;

    int radix = s.find_first_of('.');
    if (radix == string::npos) {radix = s.length();}

    for (int i = 0; i < s.length(); i++)
    {
        if (s[i] >= '0' && s[i] <= '9')
        {
            number += double(s[i] - 48) * pow(10, (radix - i) - 1);
        }
        else if (s[i] == '.')
        {
            radix += 1;
        }
        else
        {
            return number;
        }
    }

    return number;
}

int main(int argc, char* argv[])
{
    if (argc > 1)
    {
        ChainedHash<Stock> chainedHash;
        Heap<Proxy_byName> heapName;
        Heap<Proxy_byVolume> heapVolume;
        Heap<Proxy_byPercentUp> heapPercentUp;
        Heap<Proxy_byPercentDown> heapPercentDown;
        Heap<Proxy_byPercentChange> heapPercentChange;
        Heap<Proxy_byMomentum> heapMomentum;
        Heap<Proxy_byTrend> heapTrend;

        ifstream fin(argv[1]);
        string line;

        while (!fin.eof())
        {
            getline(fin,line);
            vector<string> tokens = parseTokens(line,' ');
            size_t code = processTokens(tokens);
            Stock* dummyStock;

            switch (code)
            {
                case UNRECOGNIZED: cout << "Unrecognized input: " << line << endl; break;

                case ADD:
                    {
                        vector<string> stockInfo = parseTokens(tokens[2],'s');
                        dummyStock = new Stock(tokens[1],(int) (readNumberFromString(stockInfo[0])),readNumberFromString(stockInfo[1]));
                        chainedHash.insert(*dummyStock);
                        heapName.push(Proxy_byName(dummyStock));
                        heapVolume.push(Proxy_byVolume(dummyStock));
                        heapPercentUp.push(Proxy_byPercentUp(dummyStock));
                        heapPercentDown.push(Proxy_byPercentDown(dummyStock));
                        heapPercentChange.push(Proxy_byPercentChange(dummyStock));
                        heapMomentum.push(Proxy_byMomentum(dummyStock));
                        heapTrend.push(Proxy_byTrend(dummyStock));

                        cout << "Added stock: " << dummyStock->toString() << "   " << dummyStock->capital << " shares @ $" << dummyStock->CurrentPrice() << endl;
                    }; break;

                case TRADE:
                    {
                        vector<string> tradeInfo = parseTokens(tokens[1],'s');
                        Stock* tradedStock = &*chainedHash.find(Stock(tokens[0]));
                        tradedStock->process_trade((int) (readNumberFromString(tradeInfo[0])),readNumberFromString(tradeInfo[1]));
                        cout << "Trade: " << tradedStock->toString() << "   " << readNumberFromString(tradeInfo[0]) << " shares @ $" << readNumberFromString(tradeInfo[1]) << endl;

                    }; break;

                case PRINT: // It is of note that all of my print...? commands
                    {       // might only be usable once because I pop from each
                            // respective heap until it is empty without pushing
                            // any of the data back. This is simply because
                            // every time I tried to pop to a vector, then
                            // re-insert all the data at the end, I would get
                            // horrific runtime errors, so I have decided it
                            // best to go this route.
                        cout << "Printing all stocks and current price..." << endl;
                        while (heapName.size() > 0)
                        {
                            Proxy_byName myProxy = heapName.pop();
                            cout << (*myProxy).toString() << "   $" << (*myProxy).CurrentPrice() << endl;
                        }
                    }; break;

                case PRINTVOLUME:
                    {
                        int runLength = readNumberFromString(tokens[1]);
                        cout << "Printing top " << runLength << " stocks by volume..." << endl;
                        while (runLength > 0)
                        {
                            Proxy_byVolume myProxy = heapVolume.pop();
                            cout << (*myProxy).toString() << "   " << (*myProxy).Volume() << " shares" << endl;
                            runLength -= 1;
                        }
                    }; break;

                case PRINTUP:
                    {
                        int runLength = readNumberFromString(tokens[1]);
                        cout << "Printing top " << runLength << " stocks by percentage increase..." << endl;
                        while (runLength > 0)
                        {
                            Proxy_byPercentUp myProxy = heapPercentUp.pop();
                            cout << (*myProxy).toString() << "   " << (*myProxy).PercentUp() << "%" << endl;
                            runLength -= 1;
                        }
                    }; break;

                case PRINTDOWN:
                    {
                        int runLength = readNumberFromString(tokens[1]);
                        cout << "Printing top " << runLength << " stocks by percentage decrease..." << endl;
                        while (runLength > 0)
                        {
                            Proxy_byPercentDown myProxy = heapPercentDown.pop();
                            cout << (*myProxy).toString() << "   -" << (*myProxy).PercentDown() << "%" << endl;
                            runLength -= 1;
                        }
                    }; break;

                case PRINTCHANGE:
                    {
                        int runLength = readNumberFromString(tokens[1]);
                        cout << "Printing top " << runLength << " stocks by change in percentage..." << endl;
                        while (runLength > 0)
                        {
                            Proxy_byPercentChange myProxy = heapPercentChange.pop();
                            cout << (*myProxy).toString() << "   " << (*myProxy).PercentChange() << "%" << endl;
                            runLength -= 1;
                        }
                    }; break;

                case PRINTMOMENTUM:
                    {
                        int runLength = readNumberFromString(tokens[1]);
                        cout << "Printing top " << runLength << " stocks by momentum..." << endl;
                        while (runLength > 0)
                        {
                            Proxy_byMomentum myProxy = heapMomentum.pop();
                            cout << (*myProxy).toString() << "   " << (*myProxy).Momentum() << endl;
                            runLength -= 1;
                        }
                    }; break;

                case PRINTTREND:
                    {
                        int runLength = readNumberFromString(tokens[1]);
                        cout << "Printing top " << runLength << " stocks by trend..." << endl;
                        while (runLength > 0)
                        {
                            Proxy_byTrend myProxy = heapTrend.pop();
                            cout << (*myProxy).toString() << "  " << (*myProxy).Trend() << endl;
                            runLength -= 1;
                        }
                    }; break;

                case PAUSE: break;

            }
        }
        fin.close();
    }

    return 0;
}

/** Report Questions (1A) for Hasher (Josh Studt)
1a) The hasher stores data, we used a separate vector filled with pointers
pointing to the individual start of each bucket node within the hasher.

1b) The hash table is chained; this means that each bucket can vary in size
independently. We gave the hash table a starting size (number of buckets) of
10,000.

1c) Having filled our vector with pointers, our hasher did simply use null data
as a way of differentiating between dummy and data nodes.

1d) Our vector did fill the dummy node the first time something was inserting
into that respective bucket, thus our pointers in the vector need not be
refreshed because they always pointed to the same node on the hash, empty or
not.

1e) Our hasher simply returned null for end() if the last bucket was empty.

1f) For our find() method comparisons our hasher simply did a == comparison for
the string class value (stock ticker) entered to be found.

1g) There is a simple operator*() call that returns a pointer within the chained
hash class.

1h) As far as figuring out the method was supposed to do/how to do it: yes. Making it faster?
No, not really.
*/

/** Report Questions (1B) for Heaper (Adam Badawy)
(a) My Heap class stores proxy objects.

(b) The Heap and the ChainedHash do not interact. They are completely
    decoupled.

(c) If you are referring to every item in the Heap being greater than or equal
    to its children, then yes, my Heap is always a Heap.

(d) I did not make use of function objects, but let's illustrate a hypothetical
    situation in which I did have them; instead of having a multitude of Proxy
    classes, I simply have one Proxy class, which has a field which is a pointer
    to a Stock object, and a field which is a function object used to determine
    what value of the Stock object we are concerned about. I.e., Proxy_byVolume
    would be replaced by a Proxy class, and upon construction of this Proxy, I
    would pass it a pointer to a stock, and a function object which overloads
    operator() to return the value returned in a call to ptr->Volume(). In this
    instance, I would have my function objects take an argument of a pointer to
    a Stock, for more simplified use with the Proxy class.

(e) See above answer for how I would have used function objects, had I thought
    it to be a more efficient design, given time constraints and the help
    offered from Mahmoud. I couldn't be bothered to reinvent the wheel.

(f) I wrote a handy little function in my client to parse tokens from any string
    with any delimiter I so choose. I think this was a good design choice
    because it makes things much less messy when searching for multiple
    delimiters (e.g., ' ' and 's'). I also wrote a function to convert a string
    containing only digits and decimal points to a double value. Using these two
    function in my client, I then simply passed the values Stock's process_trade
    method needed to work with - no strings, no messy conversions.

(g) My Stock class maintains fields for all calculations. I don't keep any
    "records", per se.

(h) I wouldn't let a hasher talk me into anything, Dr. Regan. Plus, I'm always
    aiming to further decouple components. I tend to write ravioli code, but I
    always thought that better than spaghetti code. Now, to ask if this is an
    application where one could get away with "that", begs the question, "How
    much should a programmer really be 'getting away with'?". I know this all
    seems like beating around the bush of the question, but if I'm understanding
    what you're asking correctly, it seems like such a poor design choice I
    can't even begin to justify it.
*/

