/**     "ValliRC.h", by Raymond Chen    CSE250
 
 This is a header file for Valli. Valli combines a vector and linked list with iterator to store
 and sort the reversed words. It performs this task by storing milepost (iterators) for every
 rth(the ratio) cell of a list in a vector. It performs binary search on the vector to narrow down the
 milepost and then linear search on the linked list. With this implementation, the running time is
 O(log(n/r)+r).
 
 */


// ENS/INV: Valli will preserve sortness

#include <vector>
#include <iostream>
#include "DList.h"

using std::string;
using std::vector;

template <typename I>       // REQ: I has I(), operator<<
class Valli {
public:
    // iterator for the Valli class
    // encapsulates the iterator from "DList.h"
	class iterator {
		typename DList<I>::iterator list_iter;
        
		iterator(const typename DList<I>::iterator& l_iter)
        :list_iter(l_iter){}
        
		friend class Valli<I>;
        
	public:
		iterator(){}
        
		iterator(const iterator& other)
        :list_iter(other.list_iter){}
        
		iterator& operator=(const iterator& rhs) {
			list_iter = rhs.list_iter;
			return *this;
		}
        
		I& operator*() const {
			return *list_iter;
		}
        
		iterator& operator++() {
			++list_iter;
			return *this;
		}
        
		iterator operator++(int) {
			iterator oldMe = *this;
			list_iter++;
			return oldMe;
		}
        
		bool operator==(const iterator& rhs) const {
			return list_iter == rhs.list_iter;
		}
        
		bool operator!=(const iterator& rhs) const {
			return list_iter != rhs.list_iter;
		}
	};
    
    /*  @param list the list storing the reversed words
        @param vec the vector that stores iterators as mileposts
        @param n number of entries
        @param r the ratio
     */
private:
    DList<I> list;
    vector<iterator> vec;
    size_t n, r;
    
public:
    explicit Valli(size_t ratio)
    :list(DList<I>()), vec(vector<iterator>()), n(0), r(ratio) {}
    
    // default ratio is 20
    Valli()
    :list(DList<I>()), vec(vector<iterator>()), n(0), r(20) {}
    
    ~Valli() {}
    
    // insert function performs binary search on the vector for mileposts(iterators)
    // then it performs linear search on the linked list
    // returns an iterator on the just-inserted item
    // calls refresh when necessary
    
    // INV: List will remain sorted
    //      Inserted item(const) will not be modified
    
    iterator insert(const I& item) {
        if(vec.size() == 0) {
            vec.push_back(list.begin());
        }else{
            vec[0] = iterator(list.begin());
        }
        size_t left = 0;
        size_t right = vec.size();
        while(right > left+1) {
            size_t mid = (right+left)/2;
            if(item < *vec[mid]) {
                right = mid;
            }else{
                left = mid;
            }
        }
        iterator itr = vec[left];
        iterator itr_stop;
        if(right == vec.size()) {
            itr_stop = list.end();
        }else{
            itr_stop = vec[right];
        }
        for(; itr != itr_stop; itr++) {
            if(*itr >= item) {
                break;
            }
        }
        iterator result = list.insert(itr.list_iter, item);
        if(++n > 2 * r * vec.size()) {
            refresh();
        }
        return result;
    }
    
    // the find function performs binary search on the vector to find the milepost(iterators)
    // then it performs linear search on the linked list
    // returns the iterator on item, end() if not found
    
    // INV: const, will not modify data
    iterator find(const I& item) const {
        size_t left = 0;
        size_t right = vec.size();
        while(right > left+1) {
            size_t mid = (right+left)/2;
            if(item < *vec[mid]) {
                right = mid;
            }else{
                left = mid;
            }
        }
        iterator itr_start = vec[left];
        iterator itr_stop;
        if(right == vec.size()) {
            itr_stop = list.end();
        }else{
            itr_stop = vec[right];
        }
        for(iterator itr = itr_start; itr != itr_stop; itr++) {
            if(*itr == item) {
                return itr;
            }else if(*itr > item) {
                return end();
            }
        }
        return end();
    }
    
    // the erase function removes what the arguement(iterator) is pointing at
    // does binary search on the vector to check if the iterator is a milepost
    // INV: list will remain sorted

    void erase(iterator itr) {
        size_t left = 0;
        size_t right = vec.size();
        while(right > left+1) {
            size_t mid = (right+left)/2;
            if(*itr < *vec[mid]) {
                right = mid;
            }else{
                left = mid;
            }
        }
        iterator start_itr = vec[left];
        if(itr == start_itr) {
            vec[left]++;
            list.erase(itr.list_iter);        
        }else{
            list.erase(itr.list_iter);        
        }
        n--;
    }
    
    //return iterator on first item
    iterator begin() const {
		return iterator(list.begin());
	}
    
    //return iterator on last item
	iterator end() const {
		return iterator(list.end());
	}
    
    size_t size() const {
		return n;
	}
    
	bool empty() const {
		return size() == 0;
	}
    
    //refresh mileposts by taking a new ratio arguement
    void refresh(size_t newRatio) {
        r = newRatio;
        vec.clear();
        size_t count = r;
        for(iterator itr = begin(); itr != end(); itr++, count++) {
            if(count == r) {
                count = 0;
                vec.push_back(itr);
            }
        }
    }
    
    void refresh() {
		refresh(r);
	}
	
    string str() const {
		return list.toString();
	}
};
