import java.awt.*;
import java.awt.geom.*;
import java.util.*;



import net.tinyos.prowler.Simulator;

class PolygonDoubleIterator implements PathIterator
{
	PolygonDouble poly;
	Iterator<Point2D.Double> i;
	boolean firstSegment;
	Point2D.Double p;
	AffineTransform at;
	PolygonDoubleIterator(PolygonDouble poly,AffineTransform at)
	{
		this.poly=poly;
		this.at=at;
		i=poly.points.iterator();
		p=null;
		firstSegment=true;
		if (i.hasNext())
		{
			p=i.next();
		}
	}
	public int currentSegment(float[] arg0) 
	{
		arg0[0]=(float)p.x;
		arg0[1]=(float)p.y;
		if (at!=null)
			at.transform(arg0,0,arg0,0,arg0.length/2);
		if (firstSegment)
			return PathIterator.SEG_MOVETO;
		else
			return PathIterator.SEG_LINETO;
	}

	public int currentSegment(double[] arg0) 
	{
		arg0[0]=p.x;
		arg0[1]=p.y;
		if (at!=null)
			at.transform(arg0,0,arg0,0,4);
		if (firstSegment)
			return PathIterator.SEG_MOVETO;
		else
			return PathIterator.SEG_LINETO;
	}

	public int getWindingRule() 
	{
		return PathIterator.WIND_EVEN_ODD;
	}

	public boolean isDone() 
	{
		return !(i.hasNext());
	}

	public void next() 
	{
		if (firstSegment)
			firstSegment=false;
		else if (i.hasNext())
		{
			p=i.next();
		}
	}
	
}

class PolygonDouble implements Shape
{
	Vector<Point2D.Double> points;
	PolygonDouble()
	{
		points=new Vector<Point2D.Double>();
	}
	public void addPoint(Point2D p)
	{
		points.add(new Point2D.Double(p.getX(),p.getY()));
	}
	
	public Point2D getCenterOfMass()
	{
		Point2D.Double com=new Point2D.Double();
		for (Point2D.Double p:points)
		{
			com.x+=p.x;
			com.y+=p.y;
		}
		com.x/=points.size();
		com.y/=points.size();
		return com;
	}
	
	public boolean contains(Point2D p) 
	{
		return contains(p.getX(),p.getY());
	}

	public boolean contains(Rectangle2D r) 
	{
		return contains(r.getMinX(),r.getMinY(),r.getWidth(),r.getHeight());
	}

	public boolean contains(double x, double y) 
	{
		if (points.size()<3)
			return false;
		for (int i=0;i<points.size()-1;i++)
		{
			Point2D.Double p=points.get(i);
			Point2D.Double q=points.get(i+1);
			if (Line2D.Double.relativeCCW(p.x,p.y,q.x,q.y,x, y)==-1)
			{
				return false;
			}
		}
		return true;
	}

	public boolean contains(double x, double y, double w, double h) 
	{
		return contains(x,y) &&
			contains(x+w,y) &&
			contains(x,y+h) &&
			contains(x+w,y+h);
	}

	public Rectangle getBounds() 
	{
		return getBounds2D().getBounds();
	}

	public Rectangle2D getBounds2D() 
	{
		Rectangle2D.Double r=new Rectangle2D.Double();
		for (int i=0;i<points.size();i++)
		{
			r.add(points.get(i));
		}
		return r;
	}

	public PathIterator getPathIterator(AffineTransform at) 
	{
		return new PolygonDoubleIterator(this,at);
	}

	public PathIterator getPathIterator(AffineTransform at, double flatness) 
	{
		return new PolygonDoubleIterator(this,at);
	}

	public boolean intersects(Rectangle2D r) 
	{
		PathIterator pi=getPathIterator(null);
		double [] arr=new double[6];
		while(!pi.isDone())
		{
			if (pi.currentSegment(arr)==PathIterator.SEG_LINETO)
			{
				if (r.intersectsLine(arr[0], arr[1],arr[2],arr[3]))
					return true;
			}
			pi.next();
		}
		return false;
	}

	public boolean intersects(double x, double y, double w, double h) 
	{
		return intersects(new Rectangle2D.Double(x,y,w,h));
	}
	
}

public class GeomNode
{
	/** location */
	Point2D.Double pos;
	/** approximated location */
	Point2D.Double approxPos;
	/** number of samples used for position approximation */
	int approxSampleCount; 
	/** next node in routing table */
	GeomNode next;
	/** voronoi cell neighbors */
	Vector<DelunayNeighbor> neighs;
	/** 1 hop neighbors */
	Vector<GeomNode> oneHopNodes;
	double voronoiCellArea;
	double voronoiCellCircum;
	/** total length of bad edges including indirects */
	double badEdgeLength;
	/** total length of bad edges excluding indirects */
	double badIndirectEdgeLength;
	/** expected ratio of handoffs to happen in this cell */
	double handoffRatio;
	/** expected ratio of proper handoffs from this cell */
	double connRatio;
	/** expected ratio of proper handoffs from this cell including indirects */
	double connIndirectRatio;
	/** the buffer of packets with their timestamps. All overflowing packets are dropped @see TrailKernel.bufferSize*/
	double [] packetBuffer;
	/** front of the packet queue */
	int queueFront;
	/** end of the packet queue */
	int queueEnd;
	/** is this node generating packets now */
	boolean active;
	/** number of packets allowed to be sent this time step */
	int bwAllocated;
	/** energy used in joules */
	double energyUsed;
	
	static long FIRST_FREE_ID=0;
	long id;
	int colorIdx;
	PolygonDouble voronoiCellShape;
	PolygonDouble extendedVoronoiCellShape;
	int handoffCount;
	int badHandoffCount;
	TrailApplication trailApp; 
	GeomNode()
	{
		pos=null;
		next=null;
		id=FIRST_FREE_ID++;
		voronoiCellArea=0;
		voronoiCellCircum=0;
		colorIdx=0;
		handoffCount=0;
		badHandoffCount=0;
		badEdgeLength=0;
		badIndirectEdgeLength=0;
		handoffRatio=0;
		connRatio=0;
		connIndirectRatio=0;
		neighs=new Vector<DelunayNeighbor>();
		oneHopNodes=new Vector<GeomNode>();
		packetBuffer=new double[TrailKernel.krn.bufferSize];
		active=false;
		queueFront=0;
		queueEnd=0;
		bwAllocated=0;
		energyUsed=0;
		approxPos=new Point2D.Double();
		approxSampleCount=0;
	}
	
	GeomNode(double x, double y)
	{
		this();
		pos=new Point2D.Double(x,y);
	}
	
	void setPos(double x, double y)
	{
		if (pos==null)
			pos=new Point2D.Double(x,y);
		else
		{
			pos.x=x;
			pos.y=y;
		}
	}
	
	void createVoronoiCellShape()
	{
		voronoiCellShape=new PolygonDouble();
		for (DelunayNeighbor dn:neighs)
		{
			if (Line2D.relativeCCW(pos.x, pos.y, dn.p.x, dn.p.y, dn.q.x, dn.q.y)==1)
			{
				Point2D.Double t=dn.p;
				dn.p=dn.q;
				dn.q=t;				
			}
		}		
		Collections.sort(neighs, new Comparator<DelunayNeighbor>()
				{
					public int compare(DelunayNeighbor arg0, DelunayNeighbor arg1) 
					{
						if (Math.atan2(arg0.p.y-pos.y,arg0.p.x-pos.x)<Math.atan2(arg1.p.y-pos.y,arg1.p.x-pos.x))
							return -1;
						return 1;
					}
		});
		voronoiCellShape.addPoint(neighs.get(0).p);
		for (DelunayNeighbor dn:neighs)
		{
			voronoiCellShape.addPoint(dn.q);
		}
		voronoiCellShape.addPoint(neighs.get(0).p);
		fillExtendedVoronoiCells();
	}
	
	void fillExtendedVoronoiCells()
	{
		double R=1/TrailKernel.krn.anchorUpdateProbability*TrailKernel.krn.bsSpeed;
		extendedVoronoiCellShape=new PolygonDouble();
		extendedVoronoiCellShape.addPoint(neighs.get(0).p);
		for (DelunayNeighbor dn:neighs)
		{
			// x=k*a_x+b_x, y=k*a_y+b_y take let l=|p-q| then use k=1+d/l less
			double dx=dn.q.x-dn.p.x;
			double dy=dn.q.y-dn.p.y;
			double l=Math.sqrt(dx*dx+dy*dy);
			dx/=l;
			dy/=l;
			double x1=dn.p.x+dy*R;
			double y1=dn.p.y-dx*R;
			double x2=dn.q.x+dy*R;
			double y2=dn.q.y-dx*R;
			extendedVoronoiCellShape.addPoint(new Point2D.Double(x1,y1));
			extendedVoronoiCellShape.addPoint(new Point2D.Double(x2,y2));
			extendedVoronoiCellShape.addPoint(dn.q);
		}		
		extendedVoronoiCellShape.addPoint(neighs.get(0).p);				
	}
	
	int getBufferedPacketCount()
	{
		if (TrailKernel.krn.useProwler)
			return trailApp.getQueuedPacketCount();
		else
			return (queueFront+TrailKernel.krn.bufferSize-queueEnd)%TrailKernel.krn.bufferSize;
	}

	@Override
	public String toString()
	{
		return this.getClass().getName()+"["+id+"]";
	}
}


class DelunayNeighbor
{
	GeomNode node; // neighboring node
	Point2D.Double p; // two points on boundary
	Point2D.Double q;
	double edgeWeight;
	double indirectCoverage;
	enum DelunayNeighborType {ProperEdge, IndirectEdge, BadEdge};
	DelunayNeighborType type;
	
	DelunayNeighbor(GeomNode n,double x1, double y1, double x2, double y2)
	{
		node=n;
		type=DelunayNeighborType.ProperEdge;
		p=new Point2D.Double(x1,y1);
		q=new Point2D.Double(x2,y2);
	}
	
	DelunayNeighbor(GeomNode n,Point2D.Double p,Point2D.Double q)
	{
		this(n,p.x,p.y,q.x,q.y);
	}
}

class MobileNode extends GeomNode
{
	Point2D.Double dir;
	Point2D.Double target;
	TrailKernel krn;
	double speed;
	void init()
	{
		this.krn=TrailKernel.krn;
		target=null;
		dir=new Point2D.Double();	
		dir.x=krn.rnd.nextDouble()*2-1;
		dir.y=krn.rnd.nextDouble()*2-1;
		double len=dir.distance(0, 0);
		dir.x/=len;
		dir.y/=len;
		speed=1;
	}
	
	MobileNode()
	{
		super();
		init();
	}
	
	MobileNode(double x,double y)
	{		
		super(x,y);
		init();
	}
	
	void navigate()
	{
		navigate(krn.rnd);
	}
	
	void navigate(Random rnd)
	{
		// deal with motion
		if (target==null)
		{
			if (krn.useRandomWPMethod)
			{
				target=new Point2D.Double((int)(rnd.nextDouble()*krn.width),(int)(rnd.nextDouble()*krn.height));
			}
			else
			{
				if (this.pos.x+this.dir.x<0 || this.pos.x+this.dir.x>krn.width ||
						this.pos.y+this.dir.y<0 || this.pos.y+this.dir.y>krn.height)
				{
					Point2D.Double newdir=new Point2D.Double();
					do
					{
						newdir.x=rnd.nextDouble()*2-1;
						newdir.y=rnd.nextDouble()*2-1;
						double len=newdir.distance(0, 0);
						newdir.x/=(len);
						newdir.y/=(len); // turn around
					}while (this.pos.x+newdir.x*speed<0 || this.pos.x+newdir.x*speed>krn.width ||
							this.pos.y+newdir.y*speed<0 || this.pos.y+newdir.y*speed>krn.height);
					this.dir=newdir;
				}
				this.pos.x+=this.dir.x*speed;
				this.pos.y+=this.dir.y*speed;
			}
		}
		else
		{
			this.dir.x=target.x-this.pos.x;
			this.dir.y=target.y-this.pos.y;
			double len=this.dir.distance(0,0);
			this.dir.x/=len;
			this.dir.y/=len;
			if (speed>len)
			{
				this.pos.x+=this.dir.x*(len-0.01);
				this.pos.y+=this.dir.y*(len-0.01);
			}
			else
			{
				this.pos.x+=this.dir.x*speed;
				this.pos.y+=this.dir.y*speed;
			}
			if (len<0.1)
				target=null;
		}		
	}
}

class ROI extends MobileNode
{
	static Random roiRand=null;
	double size;
	ROI()
	{
		super();
		speed=krn.roiSpeed;
		if (krn.useShareROISize)
			size=krn.ROIsize/Math.sqrt(krn.roiCount);
		else
			size=krn.ROIsize;	
		if (roiRand==null)
			roiRand=new Random(krn.seed);
	}
	
	
	ROI(double x,double y)
	{
		super(x,y);
		if (krn.useShareROISize)
			size=krn.ROIsize/Math.sqrt(krn.roiCount);
		else
			size=krn.ROIsize;	
		speed=krn.roiSpeed;
		if (roiRand==null)
			roiRand=new Random(krn.seed);
	}
	
	void activateInRange()
	{
		for (GeomNode n:krn.nodes)
		{
			if (Math.abs(n.pos.x-pos.x)<=size && Math.abs(n.pos.y-pos.y)<=size && n.pos.distance(pos)<size)
			{
				n.active=true;
			}
		}
	}
	void step()
	{
		navigate(roiRand);
		activateInRange();
	}
}

class TeleportingROI extends ROI
{
	int tLeft;
	TeleportingROI()
	{
		super();
		speed=0;
		tLeft=0;
		if (roiRand==null)
			roiRand=new Random(krn.seed);
	}
	
	TeleportingROI(double x,double y)
	{
		super(x,y);
		speed=0;
		tLeft=0;
		if (roiRand==null)
			roiRand=new Random(krn.seed);
	}	
	void step()
	{
		//navigate(roiRand);
		if (tLeft<=0)
		{
			pos.x=roiRand.nextDouble()*TrailKernel.krn.width;
			pos.y=roiRand.nextDouble()*TrailKernel.krn.height;
			tLeft=krn.roiTeleportPeriod*10/9+roiRand.nextInt(krn.roiTeleportPeriod*2/10);
		}
		tLeft--;
		activateInRange();
	}
}

class SmartROI extends ROI
{
	Vector<Point2D.Double> Communities;
	static double N=600;
	static double Cn=60;
	static double Cc=60;
	static double vmax=15;
	static double vmin=5;
	static double Tmaxn=100;
	static double Tmaxc=100;
	static double Lr=312;
	static double Ll=48;
	static double pln=0.5;
	static double prn=0.2;
	static double plc=0.8;
	static double prc=0.2;
	static double Tn=5760;
	static double Tc=2880;
	
	boolean isRoaming;
	boolean isPausing;
	boolean isNormalMovement;
	int duration;
	int sleepDuration;
	Point2D.Double currentCommunity;
	
	SmartROI()
	{
		super();
		initSmart();
	}
	
	SmartROI(double x,double y)
	{
		super(x,y);
		initSmart();
	}
	
	private void initSmart()
	{
		isRoaming=true;
		isPausing=false;
		isNormalMovement=true;
		duration=0;
		sleepDuration=0;
		Communities=new Vector<Point2D.Double>();
		for (int i=0;i<20;i++)
		{
			Point2D.Double d=new Point2D.Double(roiRand.nextDouble()*krn.width,roiRand.nextDouble()*krn.height);
			Communities.add(d);
		}
		currentCommunity=getNextCommunity();
	}
	
	double getExpRandom(double lambda)
	{
		return Math.log(roiRand.nextDouble())/(1/lambda);
	}
	
	double getMovementDuration()
	{
		if (isRoaming)
			return getExpRandom(Lr);
		else
			return getExpRandom(Ll);
	}

	double getPauseDuration()
	{
		if (isNormalMovement) 
			return roiRand.nextDouble()*Tmaxn;
		else
			return roiRand.nextDouble()*Tmaxc;
	}
	
	double getPl()
	{
		if (isNormalMovement)
			return pln;
		else
			return plc;
	}
	
	double getPr()
	{
		if (isNormalMovement)
			return prn;
		else
			return prc;
	}
	
	double getC()
	{
		if (isNormalMovement)
			return Cn/2;
		else
			return Cc/2;
	}
	
	double getRandomSpeed()
	{
		return vmin+roiRand.nextDouble()*(vmax-vmin);
	}
	
	double getRandomDirection()
	{
		return roiRand.nextDouble()*Math.PI*2;
	}
	
	Point2D.Double getNextCommunity()
	{
		 double denom=55835135.0/15519504;
		 double val=roiRand.nextDouble()*denom;
		 for (int i=0;i<Communities.size();i++)
		 {
			 val-=1.0/(i+1);
			 if (val<0)
				 return Communities.get(i);
		 }
		 return null;
	}
	
	void setNextTarget()
	{
		double length=getMovementDuration();
		double angle=getRandomDirection();
		target=new Point2D.Double(pos.x+length*Math.cos(angle),pos.y+length*Math.sin(angle));
		if (!isRoaming)
		{
			if (Math.abs(target.x-currentCommunity.x)>getC())
			{
				target.x=currentCommunity.x+(roiRand.nextDouble()-0.5)*getC();
			}
			if (Math.abs(target.y-currentCommunity.y)>getC())
			{
				target.y=currentCommunity.y+(roiRand.nextDouble()-0.5)*getC();
			}
		}
		
		if (target.x<0)
			target.x=0;
		if (target.y<0)
			target.y=0;
		if (target.x>krn.width)
			target.x=krn.width;
		if (target.y>krn.height)
			target.y=krn.height;
	}

	@Override
	void step()
	{		
		Point2D.Double oldTarget=null;
		if (target!=null)
			oldTarget=new Point2D.Double(this.target.x,this.target.y);
		boolean endOfEpoch=false;
		if (sleepDuration>0)
		{
			sleepDuration--;
			super.activateInRange();
			return;
		}
		super.step();
		if (target==null || !target.equals(oldTarget))
		{
			endOfEpoch=true;
		}
		duration++;
		if (isNormalMovement)
		{
			if (duration>=Tn)
			{
				isNormalMovement=false;
				duration=0;
			}
		}
		else 
		{
			if (duration>=Tc)
			{
				isNormalMovement=true;
				duration=0;
				// assign a new community
				currentCommunity=getNextCommunity();
			}
		}
		if (endOfEpoch)
		{
			if (isRoaming)
			{
				if (roiRand.nextDouble()<getPl())
					isRoaming=false;
			}
			else
			{
				if (roiRand.nextDouble()<getPr())
					isRoaming=true;
			}
			setNextTarget();
			sleepDuration=(int)getPauseDuration();
			//System.out.println("Going to sleep for "+sleepDuration+" seconds");
		}
		
	}
}

class BS extends MobileNode
{
	GeomNode prev;
	BS()
	{
		super();
		prev=null;
	}
	
	BS(double x,double y)
	{
		super(x,y);
		prev=null;
		speed=TrailKernel.krn.bsSpeed;
	}
	
	void simpleNavigate()
	{
		Point2D.Double newTarget=new Point2D.Double();
		if (target==null)
		{
			target= new Point2D.Double(0,0);
		}
		newTarget.x=0;
		newTarget.y=0;
		for (GeomNode n:krn.nodes)
		{
			if (n.pos.distance(pos)<krn.range && prev!=n)
			{
				newTarget.x+=(n.pos.x-pos.x)*(n.getBufferedPacketCount()+n.bwAllocated);
				newTarget.y+=(n.pos.y-pos.y)*(n.getBufferedPacketCount()+n.bwAllocated);
			}
		}
		double len=newTarget.distance(0,0);
		if (len<1e-5)
		{
			target=null;
		}
		else
		{
			newTarget.x+=pos.x;
			newTarget.y+=pos.y;
//			System.out.println("target="+target+"  pos="+pos);
			target.x=target.x*0.5+newTarget.x*0.5;
			target.y=target.y*0.5+newTarget.y*0.5;
		}		
	}
	
	void prowlerNavigate()
	{
		if (target==null)
			target= new Point2D.Double(0,0);
		Point2D.Double newTarget=((TrailBSApplication)trailApp).getTargetPosition();
		
		if (((TrailBSApplication)trailApp).recentMessages.size()>0 || TrailKernel.krn.useOracle)
		{
			target.x=newTarget.x;
			target.y=newTarget.y;
		}
		else
		{
			if (TrailKernel.krn.useUnicastRouteUpdate)
			{
				target.x=newTarget.x;
				target.y=newTarget.y;
			}
			else
				target=null;
		}
	}
	
	void navigate()
	{
		if (TrailKernel.krn.prowlerDriver.sim.getSimulationTime()<Simulator.ONE_SECOND*500)
			speed=0;
		else
			speed=TrailKernel.krn.bsSpeed;

		if (krn.useFollowData || krn.useFollowFlow || krn.useOracle)
		{
			if (!krn.useProwler)
				simpleNavigate();
			else
				prowlerNavigate();
		}
		if (krn.useSmartRandomWalk)
			smartRandomWalk();
		else
			super.navigate();
	}
	
	private void smartRandomWalk()
	{
		boolean isBadDirection=true;
		for (int i=0;i<1000 && isBadDirection;i++)
		{
			isBadDirection=false;
			for (Line2D.Double l:krn.linesNotToCross)
			{
				if (l.intersectsLine(pos.x, pos.y, pos.x+dir.x*krn.range*2, pos.y+dir.y*krn.range*2))
				{
					isBadDirection=true;
					break;
				}
			}
			if (isBadDirection)
			{
				double theta=krn.rnd.nextDouble()*Math.PI*2;
				dir.x=Math.cos(theta);
				dir.y=Math.sin(theta);
			}
		}
		super.navigate();
	}

	GeomNode getClosestTo(Point2D.Double p)
	{
		double mindist=Double.MAX_VALUE;
		GeomNode newPrev=null;
		for (GeomNode n:krn.nodes)
		{
			if (Math.abs(n.pos.x-p.x)<=krn.range && Math.abs(n.pos.y-p.y)<=krn.range)
			{
				double d=n.pos.distance(p);
				if (d<krn.range)
				{
					if (d<mindist)
					{
						newPrev=n;
						mindist=d;
					}
				}
			}
		}
		return newPrev;
	}
	
	GeomNode getClosestNode()
	{
		return getClosestTo(pos);
	}
	
	void simpleTrack()
	{
		// deal with handoffs
		
		double mindist=Double.MAX_VALUE;
		GeomNode oldPrev=this.prev;
		GeomNode newPrev=oldPrev;

		// this could be done much better with voronoi diagram
		Vector<GeomNode> inRange=new Vector<GeomNode>();
		for (GeomNode n:krn.nodes)
		{
			if (Math.abs(n.pos.x-pos.x)<=krn.range && Math.abs(n.pos.y-pos.y)<=krn.range)
			{
				double d=n.pos.distance(this.pos);
				if (d<krn.range)
				{
					if (d<mindist)
					{
						newPrev=n;
						mindist=d;
					}
					inRange.add(n);
				}
			}
		}
		if (krn.useRandomAnchors)
		{
			if (this.prev==null || krn.rnd.nextDouble()<krn.anchorUpdateProbability)
			{
				if (!inRange.contains(this.prev) || krn.rnd.nextDouble()<0.1)
					this.prev=inRange.get(krn.rnd.nextInt(inRange.size()));
			}
		}
		else
			this.prev=newPrev;
		if (krn.usePositionApproximation)
		{
			for (GeomNode n:inRange)
			{
				n.approxPos.x=(n.approxPos.x*n.approxSampleCount+pos.x)/(n.approxSampleCount+1);
				n.approxPos.y=(n.approxPos.y*n.approxSampleCount+pos.y)/(n.approxSampleCount+1);
				n.approxSampleCount++;
			}
		}
		if (oldPrev!=null && oldPrev!=this.prev)
		{
			// this means we have handoff
			krn.handoffcount++;
			oldPrev.handoffCount++;
			if (oldPrev.pos.distance(this.prev.pos)<krn.range)
				oldPrev.next=this.prev;
			else
				oldPrev.next=null;
		}
		if (this.prev!=null)
		{
			this.prev.next=this;
			for (GeomNode n:inRange)
			{
				if (n!=this.prev)
				{
					if (n.pos.distance(this.prev.pos)<krn.range)// next node should be in range
						n.next=this.prev;
				}
			}
			// do indirect handoff
			if (krn.useIndirectHandoff && oldPrev!=null && oldPrev.next==null)
			{
				for (GeomNode n:oldPrev.oneHopNodes)
				{
					if (n.next==this.prev && n.pos.distance(oldPrev.pos)<krn.range)
					{
						oldPrev.next=n;
						break;
					}
				}
			}
			// if still this case persist it means we are disconnected
			if (oldPrev!=null && oldPrev.next==null)
			{
				krn.disconnectcount++;
				oldPrev.badHandoffCount++;
			}
			if (this.prev!=null && this.prev.pos.distance(this.pos)>krn.range)
			{
				this.prev.handoffCount++;
				this.prev.badHandoffCount++;
				// you are so far away from me
				this.prev.next=null;
				this.prev=null;
				krn.disconnectcount++;
				krn.handoffcount++;
			}
		}						
	}
	
	void step()
	{
		navigate();
		if (!krn.useProwler)
			simpleTrack();
	}
}