
import java.util.LinkedList;
import java.util.TreeMap;

import net.tinyos.prowler.*;


public class ProwlerDriver
{
	TrailKernel krn;
	Simulator sim;
	MobileGaussianRadioModel radioModel;
	TrailNode tnodes[];
	TrailApplication tapps[];
	TrailNode bsnode[];
	TrailBSApplication bsapp[];
	double lastLocationUpdate;
	static ProwlerDriver theProwlerDriver;
	LinkedList<DataMessage> generatedMessages;
	
	
	ProwlerDriver(TrailKernel krn)
	{
		this.krn=krn;
		theProwlerDriver=this;
		generatedMessages=new LinkedList<DataMessage>();
		constructSim();
	}
	
	private void constructSim()
	{
		long time0;
		Simulator.random.setSeed(TrailKernel.krn.seed);
		sim = new Simulator();

		// creating the desired radio model, uncomment the one you need 
		radioModel = new MobileGaussianRadioModel(sim);
		//radioModel = new RayleighRadioModel(sim);

		tnodes = new TrailNode[krn.nNodes];
		tapps = new TrailApplication[krn.nNodes];
		time0 = System.currentTimeMillis();
		try
		{
			for (int i = 0; i < krn.nNodes; i++)
			{
				GeomNode n = krn.nodes.get(i);
				TrailNode rwn = (TrailNode) sim.createNode(TrailNode.class, radioModel, (int)n.id,
								n.pos.x / 5, n.pos.y / 5, 5);
				TrailApplication rapp = new TrailApplication(rwn);
				// never forget to set the base node 
				rapp.setBaseNode(n);
				tnodes[i] = rwn;
				tapps[i] = rapp;
			}
			bsnode=new TrailNode[krn.bsCount];
			bsapp=new TrailBSApplication[krn.bsCount];
			int i=0;
			for (BS bs:krn.bss)
			{
				bsnode[i]=(TrailNode)sim.createNode(TrailNode.class, radioModel, (int)bs.id,
						bs.pos.x / 5, bs.pos.y / 5, 5);
				bsapp[i]= new TrailBSApplication(bsnode[i]);
				bsapp[i].setBaseNode(bs);
				i++;
			}
		} catch (Exception e)
		{
			e.printStackTrace();
			System.out.println("Error Creating nodes");
			System.exit(1);
		}
		// This call is a must, please do not forget to call it whenever the mote 
		// field is set up
		radioModel.updateNeighborhoods();

		System.out.println("creation time: "
				+ (System.currentTimeMillis() - time0) + " millisecs");
		
	}
	
	void updateBsNodes()
	{
		for (TrailApplication ta:bsapp)
		{
			ta.getNode().setPosition(ta.baseNode.pos.x/5,ta.baseNode.pos.y/5,5);
		}
		if (krn.time-lastLocationUpdate > 20)
		{
			radioModel.updateNeighborhoods();
			lastLocationUpdate=krn.time;
		}
	}
	
	private void clearAllBefore(TreeMap<Long,DataMessage> data,long cutOff)
	{
		while (data.size()>0 && data.firstKey()<cutOff)
			data.remove(data.firstKey());
	}
	
	void runSim(double timeSeconds)
	{
		long messageTrailWindow=20*Simulator.ONE_SECOND;
		if (krn.showPacketTrails)
		{
			long cutOff=sim.getSimulationTime()-messageTrailWindow;
			for (TrailApplication p:tapps)
			{
				clearAllBefore(p.sendData, cutOff);
				clearAllBefore(p.recvData, cutOff);
			}
			for (TrailApplication p:bsapp)
			{
				clearAllBefore(p.sendData, cutOff);
				clearAllBefore(p.recvData, cutOff);
			}
		}
		sim.runTill((long)(krn.time*Simulator.ONE_SECOND));
		
		int totalMessagesSent=0;
		double maxEnergyUse=0;
//		int totalMessagesReceived=0;
//		int totalPacketsGenerated=0;
//		int totalQueuedPackets=0;
		for (TrailApplication p:tapps)
		{
			totalMessagesSent+=p.messagesSent;
			p.updateEnergyUse();
			if (!(p.baseNode.next instanceof BS) && (p.baseNode.energyUsed>maxEnergyUse))
				maxEnergyUse=p.baseNode.energyUsed;
//			totalMessagesReceived+=p.messagesReceived;
//			totalPacketsGenerated+=p.currentPacketIndex;
//			totalQueuedPackets+=p.msgQueue.size();
		}
		krn.maxEnergyUse=maxEnergyUse;
		double power=maxEnergyUse/krn.time;
		double totalEnergy=2.5*3600*3;// 2.5 ampere hours in joules
		krn.expectedLifeTime=totalEnergy/power/(3600*24);// lifetime in days
//		System.out.println(sim.getSimulationTime()+": PacketsGenerated="+totalPacketsGenerated+" messagesReceived="+bsapp.messagesReceived+" packetsQueued="+totalQueuedPackets+" BS window size="+((TrailBSApplication)krn.bs.trailApp).recentMessages.size());
		krn.messagesSent=totalMessagesSent;
		krn.averageDelay=(double)krn.totalDelay/krn.packetsDelivered;
		krn.deliveryRatio=(double)krn.packetsDelivered/krn.packetsGenerated;
		krn.throughput=krn.packetsDelivered/krn.time;
		updateBsNodes();
	}
	
}
