import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.Random;
import java.util.TreeMap;
import java.util.Vector;
import java.util.Map.Entry;

import net.tinyos.prowler.Application;
import net.tinyos.prowler.Event;
import net.tinyos.prowler.Simulator;

abstract class Message
{
	TrailNode from;
	TrailNode to;
	long tReceive;

	Message(net.tinyos.prowler.Node from, net.tinyos.prowler.Node to)
	{
		this.from=(TrailNode)from;
		this.to=(TrailNode)to;
	}
	
	abstract void handleMessage(TrailApplication app);
}

class DataMessage extends Message
{
	long generationTime;
	int packetId;
	TrailNode originator;
	int hopCount;
	boolean isDelivered;
	DataMessage(net.tinyos.prowler.Node from, net.tinyos.prowler.Node to, 
			net.tinyos.prowler.Node originator,long generationTime, int packetId)
	{
		super(from,to);
		this.originator=(TrailNode) originator;
		this.generationTime=generationTime;
		this.packetId=packetId;
		this.hopCount=0;
		this.isDelivered=false;
	}
	DataMessage(net.tinyos.prowler.Node from, net.tinyos.prowler.Node to,
			DataMessage origMessage)
	{
		super(from,to);
		this.originator=origMessage.originator;
		this.generationTime=origMessage.generationTime;
		this.packetId=origMessage.packetId;
		this.hopCount=origMessage.hopCount+1;
		this.isDelivered=false;
	}
	void handleMessage(TrailApplication app)
	{
		app.handleMessage(this);
	}
}

class DataAckMessage extends Message
{
	DataMessage origMessage;
	DataAckMessage(net.tinyos.prowler.Node from, DataMessage origMessage)
	{
		super(from,origMessage.from);
		this.origMessage=origMessage;
	}
	void handleMessage(TrailApplication app)
	{
		app.handleMessage(this);
	}
}

class FloodMessage extends Message
{
	int hopCount;
	public FloodMessage(net.tinyos.prowler.Node from, net.tinyos.prowler.Node to, int hopCount)
	{
		super(from,to);
		this.hopCount=hopCount;
	}
	void handleMessage(TrailApplication app)
	{
		app.handleMessage(this);
	}
}

class RouteUpdateMessage extends Message
{
	TrailNode anchor;
	public RouteUpdateMessage(net.tinyos.prowler.Node from, net.tinyos.prowler.Node to, net.tinyos.prowler.Node anchor)
	{
		super(from,to);
		this.anchor=(TrailNode)anchor;
	}
	void handleMessage(TrailApplication app)
	{
		app.handleMessage(this);
	}
}

class NeighborhoodDeclaration extends Message
{
	public NeighborhoodDeclaration(net.tinyos.prowler.Node from)
	{
		super(from,null);
	}
	void handleMessage(TrailApplication app)
	{
		app.handleMessage(this);
	}
}

class NeighborhoodReport extends Message
{
	HashMap<TrailApplication,Integer> knownNeighbors;
	public NeighborhoodReport(net.tinyos.prowler.Node from)
	{
		super(from,null);
		this.knownNeighbors=this.from.getTrailApp().declarationCount;
	}
	void handleMessage(TrailApplication app)
	{
		app.handleMessage(this);
	}
}

class TrailEvent extends Event
{
	boolean isSet;
	public TrailEvent()
	{
		time=0;
		isSet=false;
	}
	public TrailEvent(long time)
	{
		super(time);
	}
	public void execute()
	{
		super.execute();
		isSet=false;
	}
	void setTime(long time)
	{
		this.time=time;
	}
	void setTimer(long timeTicks)
	{
		setTime(timeTicks);
		ProwlerDriver.theProwlerDriver.sim.addEvent(this);
		isSet=true;
	}
	void setPeriodicTimer(long meanTime)
	{
		int minTime=(int)meanTime/2;
		long now=ProwlerDriver.theProwlerDriver.sim.getSimulationTime();
		setTimer(now+minTime+Simulator.random.nextInt(minTime*2));
	}
}

public class TrailApplication extends Application 
{
	@SuppressWarnings("serial")
	class MsgQueueType extends LinkedList<Message>
	{

		@Override
		public boolean add(Message e)
		{
			if (!msgQueueEvent.isSet)
				setMsgQueueTimer();
			return super.add(e);
		}
		
	}
	TrailEvent dataGenerationEvent;
	TrailEvent msgQueueEvent;
	TrailEvent neighborhoodDeclarationEvent;
	TrailEvent neighborhoodReportEvent;
	Simulator sim;
	int id;
	Random rnd;
	GeomNode baseNode;
	int messagesSent;
	int messagesReceived;
	int currentPacketIndex;
	int myHopCount;
	int reportsMade;
	int declarationsMade;
	int neighborhoodRepeatCount;
	int msgCheckPeriod;
	int dataGenerationPeriod;
	MsgQueueType msgQueue;
	TreeMap<Long,DataMessage> recvData;
	TreeMap<Long,DataMessage> sendData;
	HashMap<TrailApplication,Integer> declarationCount;
	HashMap<TrailApplication,Integer> reportCount;
	HashSet<TrailApplication> neighbors;
	boolean isAnchor;
	long anchorSince;
	// Power variables
	static final double sleepPower=0.03e-3*3;
	static final double samplePower=17.3e-6/100e-3;// watts
	static final double idlePower=sleepPower+samplePower;
	static final double transmitPower=15e-3*3-idlePower;
	static final double receivePower=20e-3*3-idlePower;
	static final double t_byte=416e-6;
	static final int L_preamble=271;
	static final int L_packet=36;
	static final double transmitPacketCost=transmitPower*(L_preamble+L_packet)*t_byte;
	static final double receivePacketCost=receivePower*(L_preamble+L_packet)*t_byte;	
	public TrailApplication(net.tinyos.prowler.Node node)
	{
		super(node);
		rnd=Simulator.random;
		this.sim=node.getSimulator();
		this.id=node.getId();
		msgQueue=new MsgQueueType();
		recvData=new TreeMap<Long,DataMessage>();
		sendData=new TreeMap<Long,DataMessage>();
		dataGenerationEvent=new TrailEvent()
		{			
			public void execute()
			{
				super.execute();
				stepGenerateData();
			}
		};
		msgQueueEvent=new TrailEvent()
		{
			public void execute()
			{
				super.execute();
				stepMsgQueue();
			}
		};
		neighborhoodDeclarationEvent=new TrailEvent()
		{
			public void execute()
			{
				super.execute();
				stepNeighborhoodDeclaration();
			}
		};
		neighborhoodReportEvent=new TrailEvent()
		{
			public void execute()
			{
				super.execute();
				stepNeighborhoodReporting();
			}
		};
		msgQueueEvent.setTimer(rnd.nextInt(Simulator.ONE_SECOND)+Simulator.ONE_SECOND+sim.getSimulationTime());
		neighborhoodDeclarationEvent.setTimer(rnd.nextInt(Simulator.ONE_SECOND)+Simulator.ONE_SECOND+sim.getSimulationTime());
//		setDataGenerationTimer(rnd.nextInt(Simulator.ONE_SECOND*10));
		currentPacketIndex=0;
		myHopCount=Integer.MAX_VALUE;
		declarationCount=new HashMap<TrailApplication, Integer>();
		reportCount=new HashMap<TrailApplication, Integer>();
		neighbors=new HashSet<TrailApplication>();
		neighborhoodRepeatCount=5;
		isAnchor=false;
		msgCheckPeriod=(int)(Simulator.ONE_SECOND/TrailKernel.krn.messageQueueCheckRate);
		dataGenerationPeriod=(int)(Simulator.ONE_SECOND/TrailKernel.krn.dataGenerationRate);
		dataGenerationEvent.setTimer(sim.getSimulationTime()+Simulator.ONE_SECOND*1000+5000+node.getId());
	}
	
	protected void stepNeighborhoodReporting()
	{
		NeighborhoodReport nr=new NeighborhoodReport(this.getNode());
		msgQueue.add(nr);
		reportsMade++;
		if (reportsMade<neighborhoodRepeatCount)
			neighborhoodDeclarationEvent.setTimer(sim.getSimulationTime()+Simulator.ONE_SECOND*(10+rnd.nextInt(10)));
		else
		{
			fillNeighbors();
		}
	}

	private void fillNeighbors()
	{
		for (Entry<TrailApplication, Integer> ent:reportCount.entrySet())
		{
			if (ent.getValue()>=neighborhoodRepeatCount-1 && declarationCount.containsKey(ent.getKey()) 
					&& declarationCount.get(ent.getKey())>=neighborhoodRepeatCount-1 
					&& ent.getKey().baseNode.pos.distance(baseNode.pos)<TrailKernel.krn.range)
			{
				neighbors.add(ent.getKey());
			}
		}
		//System.out.println("Topology Discovery for "+baseNode+" is complete with "+neighbors.size()+" neighbors");
	}

	protected void stepNeighborhoodDeclaration()
	{
		NeighborhoodDeclaration nd=new NeighborhoodDeclaration(this.getNode());
		msgQueue.add(nd);
		declarationsMade++;
		if (declarationsMade<neighborhoodRepeatCount)
			neighborhoodDeclarationEvent.setTimer(sim.getSimulationTime()+Simulator.ONE_SECOND*(10+rnd.nextInt(10)));
		else
			neighborhoodReportEvent.setTimer(sim.getSimulationTime()+Simulator.ONE_SECOND*(10+rnd.nextInt(10)));
	}

	public void setBaseNode(GeomNode baseNode)
	{
		this.baseNode=baseNode;
		baseNode.trailApp=this;
	}
	
	public void setMsgQueueTimer()
	{
		msgQueueEvent.setPeriodicTimer(msgCheckPeriod);
	}

	protected void stepMsgQueue()
	{
		if (msgQueue.size()>0)
		{
			if (msgQueue.getFirst() instanceof DataMessage)
			{
				DataMessage dm=(DataMessage)msgQueue.getFirst();
				TrailNode you=null;
				if (baseNode.next!=null)
					you=(TrailNode) baseNode.next.trailApp.getNode();
				dm.to=you;
			}
			if (!((msgQueue.getFirst() instanceof DataMessage) && (msgQueue.getFirst().to==null))
				&& getNode().sendMessage(msgQueue.getFirst(), this))
			{
				if (!(TrailKernel.krn.useAcknowledgments && (msgQueue.getFirst() instanceof DataMessage)))
					msgQueue.remove(0);
			}
			if (msgQueue.size()>0)
				setMsgQueueTimer();
		}
	}
	
	void generatePacket()
	{
		TrailNode me=((TrailNode)getNode());
		TrailNode you=null;
		if (baseNode.next!=null)
			you=(TrailNode) baseNode.next.trailApp.getNode();
		DataMessage m=new DataMessage(me,you,me,sim.getSimulationTime(),currentPacketIndex++);
		ProwlerDriver.theProwlerDriver.generatedMessages.add(m);
		TrailKernel.krn.packetsGenerated++;
		msgQueue.add(m);		
	}

	protected void stepGenerateData()
	{
		if (baseNode.active)
		{
			if (ROI.roiRand.nextDouble()<=TrailKernel.krn.dataLocality)
				generatePacket();
			else
			{
				int idx=ROI.roiRand.nextInt(ProwlerDriver.theProwlerDriver.tapps.length);
				ProwlerDriver.theProwlerDriver.tapps[idx].generatePacket();
			}
		}
		dataGenerationEvent.setTimer(sim.getSimulationTime()+ROI.roiRand.nextInt(3*dataGenerationPeriod/4)+dataGenerationPeriod/4);
	}

	public String toString()
	{
		return "TrailApplication["+getNode().getId()+"]"; 
	}
	
	public void updateRouteUnicast(RouteUpdateMessage rum)
	{
		GeomNode anchor=null;
		if (rum.anchor!=null)
			anchor=rum.anchor.getTrailApp().baseNode;
		if (isAnchor)
		{
			if (anchor!=baseNode)
			{
				baseNode.next=anchor;
				isAnchor=false;
			}
		}
		else
		{
			if (anchor==baseNode)
			{
				isAnchor=true;
				baseNode.next=rum.from.getTrailApp().baseNode;
			}
		}
	}
	
	public void updateRoute(RouteUpdateMessage rum)
	{
		if (TrailKernel.krn.useUnicastRouteUpdate)
			updateRouteUnicast(rum);
		else
		{
			GeomNode anchor;
			if (rum.anchor!=null)
				anchor=rum.anchor.getTrailApp().baseNode;
			else
				anchor=null;
			isAnchor=false;
			if ((baseNode.next instanceof BS) && anchor!=baseNode)
			{
				baseNode.next=null;
			}
			if (anchor!=null)
			{
				if (anchor==baseNode)
				{
					baseNode.next=rum.from.getTrailApp().baseNode;
					isAnchor=true;
				}
				else if (neighbors.contains(anchor.trailApp))
				{
					baseNode.next=anchor;
				}
				else if (TrailKernel.krn.useIndirectHandoff)
				{
					for (TrailApplication ta:neighbors)
					{
						if (ta.neighbors.contains(anchor.trailApp))
						{
							if (ta.baseNode.pos.distance(rum.from.getTrailApp().baseNode.pos)<
									baseNode.pos.distance(rum.from.getTrailApp().baseNode.pos))
									{
										baseNode.next=ta.baseNode;
										break;
									}
						}
					}
				}
			}
		}
		if (isAnchor)
		{
			anchorSince=sim.getSimulationTime();
		}
	}
	void handleMessage(DataMessage orig)
	{
		if (sim.getSimulationTime()-anchorSince>Simulator.ONE_SECOND/TrailKernel.krn.anchorUpdateProbability*2)
		{
			isAnchor=false;
			if (baseNode.next instanceof BS)
				baseNode.next=null;
		}
		TrailNode me=((TrailNode)getNode());
		TrailNode you=null;
		if (baseNode.next!=null)
			you=(TrailNode) baseNode.next.trailApp.getNode();
		boolean found=false;
		if (TrailKernel.krn.showPacketTrails)
			recvData.put(orig.tReceive,orig);
		for (Message m2:msgQueue)
		{
			if (m2 instanceof DataMessage)
			{
				DataMessage queued=((DataMessage)m2);
				if (queued.originator == orig.originator && queued.packetId == orig.packetId)
				{
					found=true;
					break;
				}
			}
		}
		if (!found)
		{
			if (!(TrailKernel.krn.useBSSnooping && isAnchor))
			{
				DataMessage dm=new DataMessage(me,you,orig);
				msgQueue.add(dm);
				if (TrailKernel.krn.useAcknowledgments)
				{
					if (orig.to!=null)
					{
						DataAckMessage ackBar=new DataAckMessage(me,orig);
						getNode().sendMessage(ackBar, this);
					}
				}
			}
		}				
	}
	
	void handleMessage(FloodMessage fm)
	{
		if (myHopCount>fm.hopCount && 
			((fm.from.getTrailApp() instanceof TrailBSApplication)|| neighbors.contains(fm.from.getTrailApp())))
		{
			myHopCount=fm.hopCount;
			FloodMessage relay=new FloodMessage((TrailNode)getNode(),null,fm.hopCount+1);
			baseNode.next=fm.from.getTrailApp().baseNode;
			Vector<Message> remove=new Vector<Message>();
			for (Message msg:msgQueue)
				if (msg instanceof FloodMessage)
					remove.add(msg);
			msgQueue.removeAll(remove);
			msgQueue.add(relay);
		}		
	}
	
	void handleMessage(RouteUpdateMessage rum)
	{
		updateRoute(rum);
	}
	
	void handleMessage(DataAckMessage ackBar)
	{
		msgQueue.remove(ackBar.origMessage);		
	}
		
	void handleMessage(NeighborhoodReport neighborhoodReport)
	{
		if (neighborhoodReport.knownNeighbors.containsKey(this))
		{
			TrailApplication ta=neighborhoodReport.from.getTrailApp();
			reportCount.put(ta, neighborhoodReport.knownNeighbors.get(this));
		}
	}

	void handleMessage(NeighborhoodDeclaration neighborHoodDeclaration)
	{
		TrailApplication ta=neighborHoodDeclaration.from.getTrailApp();
		if (!declarationCount.containsKey(ta))
			declarationCount.put(ta, 1);
		else
			declarationCount.put(ta, declarationCount.get(ta)+1);		
	}

	/**
	 * Stores the sender from which it first receives the message, and passes 
	 * the message.
	 */
	public void receiveMessage(Object message)
	{
		Message m=(Message) message;
		// for now ignore the bcast packets
		if (m.to==getNode() || m.to==null)
		{
			m.tReceive=sim.getSimulationTime();
			messagesReceived++;
			m.handleMessage(this);
		}
	}

	/**
	 * Sets the sent flag to true. 
	 */
	public void sendMessageDone()
	{
		// the sniplet below can be used to keep track of which message has been sent 
		Message m=(Message)((TrailNode)getNode()).getLastMessageSent();
		if (TrailKernel.krn.showPacketTrails)
		{
			if (m instanceof DataMessage)
				sendData.put(m.tReceive,(DataMessage)m);
		}
		messagesSent++;
	}

	public int getQueuedPacketCount()
	{
		return msgQueue.size();
	}
	
	public void updateEnergyUse()
	{
		// energy use is the sum of packets recvd and sent with the sleep duration
		// we use watts and joules and use differential power to arrive at correct figures
		final double timeSec=sim.getSimulationTimeInMillisec()/1000.0;
		final double idleEnergy=timeSec*idlePower;
		final double transmitEnergy=messagesSent*transmitPacketCost;
		final double receptionEnergy=messagesReceived*receivePacketCost;
		baseNode.energyUsed=idleEnergy+transmitEnergy+receptionEnergy;
	}
}

