import java.awt.geom.Point2D;
import java.util.HashMap;
import java.util.TreeMap;
import java.util.Map.Entry;

import net.tinyos.prowler.Simulator;

public class TrailBSApplication extends TrailApplication
{
	TrailEvent floodEvent;
	TrailEvent routeUpdateEvent;
	TreeMap<Long,DataMessage> recentMessages;
	TreeMap<Long,DataMessage> snoopedMessages;
	HashMap<GeomNode,Integer> salmonCounts;
	long timeFrameTicks;
	public TrailBSApplication(net.tinyos.prowler.Node node)
	{
		super(node);
		recentMessages=new TreeMap<Long, DataMessage>();
		snoopedMessages=new TreeMap<Long, DataMessage>();
		salmonCounts=new HashMap<GeomNode, Integer>();
		timeFrameTicks=Simulator.ONE_SECOND*600;
		// BS shouldn't generate data
		sim.removeEvent(dataGenerationEvent);
		sim.removeEvent(neighborhoodDeclarationEvent);
		floodEvent=new TrailEvent()
		{			
			public void execute()
			{
				super.execute();
				sendFloodMsg();
			}
		};
		floodEvent.setTime(sim.getSimulationTime()+Simulator.ONE_SECOND/5+Simulator.ONE_SECOND*300);
		sim.addEvent(floodEvent);
		routeUpdateEvent=new TrailEvent()
		{
			public void execute()
			{
				super.execute();
				stepRouteUpdate();
			}
		};
		setRouteUpdateTimer(sim.getSimulationTime()+Simulator.ONE_SECOND*30+node.getId()*400+Simulator.ONE_SECOND*300);
	}

	public void setRouteUpdateTimer(long timeTicks)
	{
		routeUpdateEvent.setTime(timeTicks);
		sim.addEvent(routeUpdateEvent);		
	}

	protected void stepRouteUpdate()
	{
		GeomNode n=null;
		if (TrailKernel.krn.useUnicastRouteUpdate)
		{
			BS bs=(BS)baseNode;
			GeomNode n1=null;
			GeomNode n2=null;
			double d1=Double.MAX_VALUE;
			double d2=Double.MAX_VALUE;

			if (bs.target!=null)
			{
				n1=bs.getClosestTo(bs.target);
				d1=bs.pos.distance(n1.pos);
			}
			else 
			{
				n1=bs.getClosestNode();
				d1=bs.pos.distance(n1.pos);
			}
			if (bs.prev!=null)
			{
				n2=bs.prev;
				d2=bs.pos.distance(n2.pos);
			}
			
			if (d1<d2)
				n=n1;
			else 
				n=n2;
/*			else
				n=bs.getClosestNode();*/
		}
		else
			n=((BS)baseNode).getClosestNode();
		RouteUpdateMessage rum;
		if (n==null || n.pos.distance(baseNode.pos)>TrailKernel.krn.range)
		{
			((BS)baseNode).prev=null;
			rum=new RouteUpdateMessage(this.getNode(),null,null);
		}
		else
		{
			((BS)baseNode).prev=n;
			rum=new RouteUpdateMessage(this.getNode(),null,n.trailApp.getNode());
		}
		
		msgQueue.add(rum);
		long nextTime=(long)(sim.getSimulationTime()+Simulator.ONE_SECOND/TrailKernel.krn.anchorUpdateProbability);
		setRouteUpdateTimer(nextTime);
	}
	
	protected void sendFloodMsg()
	{
		BS bs=(BS)baseNode;
		GeomNode n=bs.getClosestNode();
		if (TrailKernel.krn.useUnicastRouteUpdate)
			bs.prev=n;
		FloodMessage fm=new FloodMessage( getNode(),n.trailApp.getNode(),0);
		msgQueue.add(fm);
	}
	public String toString()
	{
		return "TrailBSApplication["+getNode().getId()+"]"; 
	}
	
	void markDelivery(DataMessage orig)
	{
		long time=sim.getSimulationTime();
		for (DataMessage d:ProwlerDriver.theProwlerDriver.generatedMessages)
		{
			if (orig.originator==d.originator && orig.packetId==d.packetId)
			{
				if (!d.isDelivered)
				{
					d.isDelivered=true;
					TrailKernel.krn.packetsDelivered++;
					TrailKernel.krn.totalDelay+=((double)time-orig.generationTime)/Simulator.ONE_SECOND;
				}
			}
		}
	}
	
	@Override
	void handleMessage(DataMessage orig)
	{
		if (TrailKernel.krn.showPacketTrails)
			recvData.put(orig.tReceive,orig);
		recentMessages.put(orig.tReceive,orig);
		markDelivery(orig);
		messagesReceived++;
		if (TrailKernel.krn.useAcknowledgments)
		{
			TrailNode me=((TrailNode)getNode());
			if (orig.to!=null)
			{
				DataAckMessage ackBar=new DataAckMessage(me,orig);
				getNode().sendMessage(ackBar, this);
			}
		}
	}
	
	@Override
	void handleMessage(FloodMessage fm)
	{
		// ignore flood messages
	}
	
	@Override
	void handleMessage(RouteUpdateMessage rum)
	{
		// ignore route update messages
	}
	
	@Override
	void handleMessage(DataAckMessage ackBar)
	{
		// ignore acknowledgements		
	}
	
	@Override
	public void sendMessageDone()
	{
		super.sendMessageDone();
		//System.out.println("Flooding Started");
	}
	
	public Point2D.Double getFlowTarget()
	{
		long oldBegin=sim.getSimulationTime()-timeFrameTicks/20;
		// clear old messages
		while (recentMessages.size()>0 && recentMessages.firstKey()<oldBegin)
			recentMessages.remove(recentMessages.firstKey());
		while (snoopedMessages.size()>0 && snoopedMessages.firstKey()<oldBegin)
			snoopedMessages.remove(snoopedMessages.firstKey());
		BS bs=(BS)baseNode;
		Point2D.Double newPos=new Point2D.Double(bs.target.x,bs.target.y);
		for (DataMessage dm:snoopedMessages.values())
		{
			GeomNode n=((TrailNode)dm.from).getTrailApp().baseNode;
			newPos.x+=n.pos.x;
			newPos.y+=n.pos.y;
		}
		newPos.x/=snoopedMessages.size()+1;
		newPos.y/=snoopedMessages.size()+1;
		return newPos;				
	}

	public Point2D.Double getSalmonTarget()
	{
		long oldBegin=sim.getSimulationTime()-timeFrameTicks/20;
		// clear old messages
		while (recentMessages.size()>0 && recentMessages.firstKey()<oldBegin)
			recentMessages.remove(recentMessages.firstKey());
		while (snoopedMessages.size()>0 && snoopedMessages.firstKey()<oldBegin)
			snoopedMessages.remove(snoopedMessages.firstKey());
		salmonCounts.clear();
		BS bs=(BS)baseNode;
		Point2D.Double newPos=new Point2D.Double(0,0);
		for (DataMessage dm:snoopedMessages.values())
		{
			GeomNode n=((TrailNode)dm.from).getTrailApp().baseNode;
			if (!salmonCounts.containsKey(n))
				salmonCounts.put(n, 1);
			else
				salmonCounts.put(n, salmonCounts.get(n)+1);
		}
		if (bs.prev!=null)
		{
			newPos.x=bs.prev.pos.x;
			newPos.y=bs.prev.pos.y;
		}
		//TrailNode gm=null;
		for (Entry<GeomNode,Integer> e:salmonCounts.entrySet())
		{
			if (e.getValue()>snoopedMessages.size()/2)
			{
				newPos.x=e.getKey().pos.x;
				newPos.y=e.getKey().pos.y;
				//gm=(TrailNode) e.getKey().trailApp.getNode();
				break;
			}
		}
/*		System.out.print("Salmon calculation with "+salmonCounts.size()+" neighbors: ");
		for (Entry<GeomNode,Integer> e:salmonCounts.entrySet())
		{
			System.out.print(e.getKey().id+"="+e.getValue()+",");
		}
		System.out.println(" Target if found="+gm+" Anchor="+bs.prev);*/
		return newPos;				
	}

	public Point2D.Double getDataTarget()
	{
		long oldBegin=sim.getSimulationTime()-timeFrameTicks;
		// clear old messages
		while (recentMessages.size()>0 && recentMessages.firstKey()<oldBegin)
			recentMessages.remove(recentMessages.firstKey());
		//BS bs=(BS)baseNode;
		Point2D.Double newPos=new Point2D.Double(0,0);
		double wtotal=0;
		for (DataMessage dm:recentMessages.values())
		{
			GeomNode n=((TrailNode)dm.originator).getTrailApp().baseNode;
			double w=(dm.generationTime-oldBegin-timeFrameTicks*3)/(timeFrameTicks*4);
			w=w>0?w+0.1:0.1;
			newPos.x+=n.pos.x*w;
			newPos.y+=n.pos.y*w;
			wtotal+=w;
		}
		wtotal=wtotal>0.05?wtotal:0.05;
//		System.out.println(wtotal);
		newPos.x/=wtotal;
		newPos.y/=wtotal;
		return newPos;		
	}
	public Point2D.Double getOracleTarget()
	{
		Point2D.Double newPos=new Point2D.Double(0,0);
		for (ROI roi:TrailKernel.krn.rois)
		{
			newPos.x+=roi.pos.x;
			newPos.y+=roi.pos.y;
		}
		newPos.x/=TrailKernel.krn.roiCount;
		newPos.y/=TrailKernel.krn.roiCount;
		return newPos;
	}
	
	
	public Point2D.Double getTargetPosition()
	{
		if (TrailKernel.krn.useOracle)
			return getOracleTarget();
		if (TrailKernel.krn.useFollowData)
			return getDataTarget();
		else if (TrailKernel.krn.useFollowFlow && TrailKernel.krn.useUnicastRouteUpdate)
			return getSalmonTarget();
		else if (TrailKernel.krn.useFollowFlow)
			return getFlowTarget();
		return null;
	}

	@Override
	public void receiveMessage(Object message)
	{
		super.receiveMessage(message);
		if (TrailKernel.krn.useBSSnooping && (message instanceof DataMessage))
		{
			if (((BS)this.baseNode).prev!=null)
			{
				TrailNode anchor=(TrailNode)(((BS)this.baseNode).prev.trailApp.getNode());
				if (((Message)message).to==anchor)
				{
					Message m=(Message)message;
					m.tReceive=sim.getSimulationTime();
					messagesReceived++;
					m.handleMessage(this);
				}
			}
		}
		if (TrailKernel.krn.useFollowFlow && (message instanceof DataMessage))
		{
			DataMessage m=(DataMessage)message;
			TrailNode anchor=null;
			if (((BS)this.baseNode).prev!=null)
				anchor=(TrailNode)(((BS)this.baseNode).prev.trailApp.getNode());
//			System.out.println("Anchor="+anchor+" snooped packet target="+m.to);
			if (m.to==anchor)
			{
//				System.out.println("Packing message="+m);
				m.tReceive=sim.getSimulationTime();
				snoopedMessages.put(m.tReceive, m);
			}
		}
	}
}
