import javax.imageio.ImageIO;
import javax.swing.*;
import java.awt.*;
import java.awt.geom.*;
import java.awt.event.*;
import java.awt.image.*;
import java.awt.print.*;
import java.io.File;
import java.io.IOException;
import java.util.*;

class TrailDisplay extends JPanel implements MouseListener, MouseMotionListener,Printable, MouseWheelListener 
{

	/**
	 * 
	 */
	private static final long serialVersionUID = -7303331365125199955L;
	
	TrailKernel krn;
	TrailFrame parent;
	Image img;
	Point mousePos;
	/** node nearest to mouse */
	GeomNode nearest;
	Random rnd;
	BufferedImage bsFigure;
	volatile boolean isPainting;
	AffineTransform af;
	boolean useFastPaint;
	class ActionQueue
	{
		Vector<MouseEvent> actions;
		ActionQueue()
		{
			actions=new Vector<MouseEvent>();
		}
		synchronized void addItem(MouseEvent e)
		{
			actions.add(e);
		}
		synchronized void process()
		{
			for (MouseEvent e:actions)
			{
				if (e instanceof MouseWheelEvent)
					handleMouseWheel((MouseWheelEvent)e);
				else
					handleMouseMotion(e);
			}
			actions.clear();
		}
	};
	ActionQueue actionQueue;

	TrailDisplay(TrailFrame parent)
	{
		this.krn=TrailKernel.krn;
		this.parent=parent;
		this.setDoubleBuffered(true);
		af=new AffineTransform();
		img=null;
		mousePos=null;
		nearest=null;
		this.addMouseListener(this);
		this.addMouseMotionListener(this);
		this.addMouseWheelListener(this);
		this.rnd=new Random(krn.seed);
		//bsFigure=loadImage("RoombaPicSmall.png");
		actionQueue=new ActionQueue();
		isPainting=false;
		useFastPaint=true;
	}
	
	BufferedImage loadImage(String fname)
	{
		File file = new File(fname);
		BufferedImage bf=null;
		try
		{
			bf=ImageIO.read(file);
		} catch (IOException e)
		{
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		return bf;
	}

	
    /**
     * Draw a line segment.
     * @param endA one endpoint
     * @param endB the other endpoint
     */
    final void draw (Graphics2D g, Pnt endA, Pnt endB) 
    {
        g.drawLine((int)endA.coord(0), (int)endA.coord(1),
                   (int)endB.coord(0), (int)endB.coord(1));
    }
	
	void drawKernel(Graphics2D g)
	{
		g.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
		AffineTransform a=g.getTransform();
		g.transform(af);
		//g.setXORMode(Color.white);
		Color [] colArr={Color.cyan,Color.green,Color.magenta,Color.orange,Color.pink,Color.yellow,Color.lightGray};
		if (krn.colorCells)
		{
			for (GeomNode n:krn.nodes)
			{
				g.setColor(colArr[n.colorIdx]);
				g.fill(n.voronoiCellShape);
			}
		}
		if (krn.showCellBorders)
		{
			g.setColor(Color.gray);
			for (GeomNode n:krn.nodes)
			{
				g.draw(n.voronoiCellShape);
			}
			for (GeomNode n:krn.nodes)
			{
				for (DelunayNeighbor dn:n.neighs)
				{
					
					if (dn.node!=null && dn.node.id<n.id)
					{
						if (dn.type==DelunayNeighbor.DelunayNeighborType.BadEdge)
						{
							g.setColor(Color.red);
							Stroke s=g.getStroke();
							float []dash={3,3};
							g.setStroke(new BasicStroke(3.0f,BasicStroke.CAP_BUTT,BasicStroke.JOIN_ROUND,1.0f,dash,0));
							g.drawLine((int)n.pos.x, (int)n.pos.y, (int)dn.node.pos.x, (int)dn.node.pos.y);
							g.setColor(Color.blue);
							g.drawLine((int)dn.p.x, (int)dn.p.y, (int)dn.q.x, (int)dn.q.y);
							g.setStroke(s);
//							g.drawString(""+dn.indirectCoverage, (int)(dn.p.x+dn.q.x)/2, (int)(dn.p.y+dn.q.y)/2);
						}
						else if (dn.type==DelunayNeighbor.DelunayNeighborType.IndirectEdge)
						{
							Stroke s=g.getStroke();
							float []dash={3,3};
							g.setStroke(new BasicStroke(2.0f,BasicStroke.CAP_BUTT,BasicStroke.JOIN_ROUND,1.0f,dash,0));
							g.setColor(Color.green);
							g.drawLine((int)n.pos.x, (int)n.pos.y, (int)dn.node.pos.x, (int)dn.node.pos.y);
							g.setColor(Color.cyan);
							g.drawLine((int)dn.p.x, (int)dn.p.y, (int)dn.q.x, (int)dn.q.y);
							g.setStroke(s);
						}
					}
				}
			}
		}
		g.setTransform(a);
	}

	public void drawCoverage(Graphics2D greal)
	{
		BufferedImage buf=new BufferedImage(this.getWidth(), this.getHeight(),BufferedImage.TYPE_INT_RGB);
		Graphics g=buf.createGraphics();
		g.setColor(Color.white);
		g.fillRect(0, 0, getWidth(), getHeight());
		double range=krn.range/Math.sqrt(3);
		for (GeomNode n:krn.nodes)
		{
			int minx=(int)(n.pos.x-range);
			int miny=(int)(n.pos.y-range);
			int maxx=(int)(n.pos.x+range);
			int maxy=(int)(n.pos.y+range);
			if (minx<0)
				minx=0;
			if (maxx>getWidth())
				maxx=getWidth();
			if (miny<0)
				miny=0;
			if (maxy>getHeight())
				maxy=getHeight();
			for (int i=miny;i<maxy;i++)
			{
				for (int j=minx;j<maxx;j++)
				{
					if (n.pos.distance(j, i)<range)
					{
						int c=buf.getRGB(j, i);
						if (c==0xFFFFFFFF)
							buf.setRGB(j, i, 0xFF0000FF);
						else if (c==0xFF0000FF)
							buf.setRGB(j, i, 0x00FF00FF);
						else
							buf.setRGB(j, i, 0x777777);
					}
				}
			}
		}
		img=buf;
	}
	
	void drawImage(Graphics2D greal)
	{
		if (img==null)
		{
			//drawCoverage(greal);
			BufferedImage buf=new BufferedImage(this.getWidth(), this.getHeight(),BufferedImage.TYPE_INT_RGB);
			Graphics2D g=buf.createGraphics();
			g.setBackground(Color.white);
			g.clearRect(0, 0, getWidth(), getHeight());
			drawKernel(g);
			img=buf;
		}
		greal.drawImage(img,0,0,null);
	}
	
	final void drawArrow(Graphics2D g, double x1,double y1,double x2,double y2)
	{
		int arrowLen=7;
		g.drawLine((int)x1, (int)y1, (int)x2, (int)y2);
		double theta=Math.atan2(y1-y2, x1-x2);
		double thetatop=theta+Math.PI/6;
		double thetabottom=theta-Math.PI/6;
		g.drawLine((int)x2, (int)y2, 
				(int)(x2+arrowLen*Math.cos(thetatop)), 
				(int)(y2+arrowLen*Math.sin(thetatop))
				);
		g.drawLine((int)x2, (int)y2, 
				(int)(x2+arrowLen*Math.cos(thetabottom)), 
				(int)(y2+arrowLen*Math.sin(thetabottom))
				);		
	}
	final void drawArrow(Graphics2D g, Point2D.Double p, Point2D.Double q)
	{
		drawArrow(g,p.x,p.y,q.x,q.y);
	}

	/** draws and arrow between two nodes */
	final void drawArrow(Graphics2D g, GeomNode n, GeomNode m)
	{
		Point2D.Double v=new Point2D.Double();
		v.x=m.pos.x-n.pos.x;
		v.y=m.pos.y-n.pos.y;
		double len=v.distance(0,0);
		double x1=n.pos.x+v.x/len*3;
		double x2=n.pos.x+v.x/len*(len-6);
		double y1=n.pos.y+v.y/len*3;
		double y2=n.pos.y+v.y/len*(len-6);
		drawArrow(g,x1,y1,x2,y2);		
	}

	/** draws an arrow to a point from a node */
	final void drawArrow(Graphics2D g, GeomNode n, double x,double y)
	{
		Point2D.Double v=new Point2D.Double();
		v.x=x-n.pos.x;
		v.y=y-n.pos.y;
		double len=v.distance(0,0);
		double x1=n.pos.x+v.x/len*3;
		double x2=n.pos.x+v.x/len*(len-6);
		double y1=n.pos.y+v.y/len*3;
		double y2=n.pos.y+v.y/len*(len-6);
		drawArrow(g,x1,y1,x2,y2);		
	}
	
	void drawLightning(Graphics2D g, GeomNode n, GeomNode m)
	{
		Point2D.Double v=new Point2D.Double();
		v.x=m.pos.x-n.pos.x;
		v.y=m.pos.y-n.pos.y;
		double len=v.distance(0,0);
		double x1=n.pos.x+v.x/len*3;
		double x2=n.pos.x+v.x/len*(len-6);
		double y1=n.pos.y+v.y/len*3;
		double y2=n.pos.y+v.y/len*(len-6);
		int nparts=10;
		Vector<Point2D.Double> points=new Vector<Point2D.Double>(nparts);
		points.add(new Point2D.Double(x1,y1));
		for (int i=0;i<nparts-1;i++)
		{
			double boxSize=((double)nparts-i)/nparts*10;
			double nx=points.get(i).x+(m.pos.x-points.get(i).x)/nparts+(rnd.nextDouble()-0.5)*boxSize;
			double ny=points.get(i).y+(m.pos.y-points.get(i).y)/nparts+(rnd.nextDouble()-0.5)*boxSize;
			points.add(new Point2D.Double(nx,ny));
		}
		Stroke[] strokes=new Stroke[3];
		strokes[0]=new BasicStroke(6);
		strokes[1]=new BasicStroke(3);
		strokes[2]=g.getStroke();
		Color[] colors=new Color[3];
		colors[2]=g.getColor();
		colors[0]=new Color((float)colors[2].getRed()/255,(float)colors[2].getGreen()/255,(float)colors[2].getBlue()/255,0.2f);
		colors[1]=new Color((float)colors[2].getRed()/255,(float)colors[2].getGreen()/255,(float)colors[2].getBlue()/255,0.4f);
		for (int idx=0;idx<3;idx++)
		{
			g.setStroke(strokes[idx]);
			g.setColor(colors[idx]);
			for (int i=0;i<nparts-1;i++)
			{
				g.drawLine((int)points.get(i).x, (int)points.get(i).y, (int)points.get(i+1).x, (int)points.get(i+1).y);
			}
			drawArrow(g, points.lastElement().x, points.lastElement().y, x2,y2);
		}
	}
	
	void drawArrows(Graphics2D g)
	{
		double avgHops=0;
		int MaxHops=50;
		for (GeomNode n:krn.nodes)
		{
			if (n.next!=null)
			{
				g.setColor(Color.black);
				if (krn.showArrows)
				{
					drawArrow(g,n,n.next);
				}
				Vector<GeomNode> seen=new Vector<GeomNode>();
				GeomNode cur=n.next;
				int hops=1;
				boolean loop=false;
				boolean found=false;
				while(cur!=null && !(cur instanceof BS))
				{
					if (seen.contains(cur))
					{
						loop=true;
						break;
					}
					seen.add(cur);
					hops++;
					cur=cur.next;
				}
				if (cur instanceof BS)
					found=true;
				if (found)
				{
					avgHops+=hops;
					g.setColor(Color.blue);
				}
				else
				{
					avgHops+=MaxHops;
					g.setColor(Color.cyan);
				}
				if (krn.showHopCount)
				{
					if (loop)
					{
						Color c=g.getColor();
						g.setColor(Color.magenta);
						g.drawString("L",(int)n.pos.x, (int)n.pos.y-7);
						g.setColor(c);
					}
					else
						g.drawString(""+hops,(int)n.pos.x, (int)n.pos.y-7);
				}
			}
			else
				avgHops+=MaxHops; // hopelessly lost
		}
		avgHops/=krn.nodes.size();
	}
	
	void drawRois(Graphics2D g,boolean useAlpha)
	{
		if (!(krn.rois.firstElement() instanceof SmartROI) || !krn.showHumanLikeROIDetail)
		{
			if (useAlpha)
				g.setColor(new Color(0x55ff00ff,true));
			else
				g.setColor(new Color(0xffaaff,false));
			for (ROI roi:krn.rois)
				g.fillOval((int)(roi.pos.x-roi.size), (int)(roi.pos.y-roi.size), (int)(roi.size*2), (int)(roi.size*2));
		}
		else
		{
			for (ROI roi:krn.rois)
			{
				SmartROI r=(SmartROI)roi;
				if (useAlpha)
					g.setColor(new Color(0x55ff00ff,true));
				else
					g.setColor(new Color(0xffaaff,false));
				g.fillOval((int)(roi.pos.x-roi.size), (int)(roi.pos.y-roi.size), (int)(roi.size*2), (int)(roi.size*2));
				g.setColor(Color.blue);
				for (int i=0;i<r.Communities.size();i++)
				{
					if (r.currentCommunity==r.Communities.get(i))
						g.setColor(Color.magenta);
					else
						g.setColor(Color.blue);						
					Point2D.Double p=r.Communities.get(i);
					double size=(21.0-i);
					g.fillOval((int)(p.x-size), (int)(p.y-size), (int)(2*size), (int)(2*size));
				}
				if (r.target!=null)
					drawArrow(g,r.pos,r.target);
			}
		}
	}
	
	void drawBS(Graphics2D g,boolean useAlpha)
	{
		for (BS bs:krn.bss)
		{
			if (bs.prev!=null)
			{
				if (useAlpha)
					g.setColor(new Color(0x77000000,true));
				else
					g.setColor(new Color(0x888888,false));
				g.fill(bs.prev.voronoiCellShape);
			}
		}
		
		g.setColor(Color.red);
		for (BS bs:krn.bss)
		{
			double theta=0;
			if (bs.dir!=null)
			{
				theta=Math.atan2(bs.dir.y, bs.dir.x)+Math.PI/2;
			}
//			AffineTransform a=new AffineTransform();
//			a.translate(bs.pos.x,bs.pos.y);
//			a.scale(0.5, 0.5);
//			a.rotate(theta);
//			a.translate(-bsFigure.getWidth()/2, -bsFigure.getHeight()/2);
			//g.drawImage(bsFigure,a, null);
			g.drawOval((int)bs.pos.x-5, (int)bs.pos.y-5, 10, 10);
			drawArrow(g,bs,bs.dir.x*30+bs.pos.x,bs.dir.y*30+bs.pos.y);
			//g.drawOval((int)(bs.pos.x-krn.range), (int)(bs.pos.y-krn.range), (int)krn.range*2, (int)krn.range*2);
			g.setColor(Color.magenta);
			if (bs.target!=null)
			{
				g.setColor(Color.magenta);
				g.drawOval((int)bs.target.x-3, (int)bs.target.y-3, 6, 6);
			}
		}
	}
	
	void drawNearest(Graphics2D g,boolean useAlpha)
	{
		if (useAlpha)
			g.setColor(new Color(0x7700ff00,true));
		else
			g.setColor(new Color(0x88ff88,false));
		g.fill(nearest.extendedVoronoiCellShape);
		for (int i=2;i<nearest.extendedVoronoiCellShape.points.size()-2;i+=3)
		{
			double R=1/krn.anchorUpdateProbability*krn.bsSpeed;
			Point2D.Double p=nearest.extendedVoronoiCellShape.points.elementAt(i);
			Point2D.Double c=nearest.extendedVoronoiCellShape.points.elementAt(i+1);
			Point2D.Double q;
			if (i<nearest.extendedVoronoiCellShape.points.size()-3)
				q=nearest.extendedVoronoiCellShape.points.elementAt(i+2);
			else
				q=nearest.extendedVoronoiCellShape.points.elementAt(1);
			if (p.distance(q)<0.01)
				continue;
			Arc2D.Double arc=new Arc2D.Double();
			arc.setArcByCenter(c.x, c.y, R, 20, 40, Arc2D.PIE);
			arc.setAngles(q, p);
			g.fill(arc);
		}
		g.drawOval((int)(nearest.pos.x-krn.range), (int)(nearest.pos.y-krn.range), (int)(krn.range*2), (int)(krn.range*2));
		if (useAlpha)
			g.setColor(new Color(0x4400ff00,true));
		else
			g.setColor(new Color(0xbbffbb,false));
		if (krn.showInRange)
		{
			for (GeomNode n:krn.nodes)
			{
				if (n!=nearest && n.pos.distance(mousePos)<krn.range)
					g.fill(n.voronoiCellShape);
			}
		}
		if (krn.showNeighbors)
		{
			g.setColor(Color.black);
			for (TrailApplication ta:nearest.trailApp.neighbors)
			{
				GeomNode n=ta.baseNode;
				g.drawLine((int)nearest.pos.x, (int)nearest.pos.y, (int)n.pos.x, (int)n.pos.y);
			}
		}
	}
	
	void drawMessages(Graphics2D g)
	{
		for (GeomNode n:krn.nodes)
		{
			int divisionCount=n.trailApp.sendData.size()+2;
			TrailApplication from=n.trailApp;
			int i=0;
			try
			{
				for (DataMessage dm:from.sendData.values())
				{
					TrailApplication to=((TrailNode)dm.to).getTrailApp();
					double dx=(to.baseNode.pos.x-from.baseNode.pos.x)/divisionCount;
					double dy=(to.baseNode.pos.y-from.baseNode.pos.y)/divisionCount;
					if (to.recvData.containsValue(dm))
					{
						g.setColor(Color.cyan);
					}
					else
					{
						g.setColor(Color.magenta);
					}
					drawArrow(g, n, to.baseNode);
					int x=(int)(from.baseNode.pos.x+dx*(i+1));
					int y=(int)(from.baseNode.pos.y+dy*(i+1));
					g.fillRect(x-2, y-2, 4, 4);
					i++;
				}
			}catch(ConcurrentModificationException e)
			{
				// yen icinde kalsin.
			}
		}
	}
	
	void paintHandoffShade(Graphics2D g)
	{
		Color redShadow=new Color(0x77ff0000,true);
		Color blueShadow=new Color(0x770000ff,true);
		for (GeomNode n:krn.nodes)
		{
			double handoffRatioSim=((double)n.handoffCount)/krn.handoffcount;
			double handoffRatioTheory=n.handoffRatio;
			if (handoffRatioSim<handoffRatioTheory*0.5)
			{
				g.setColor(redShadow);
				g.fill(n.voronoiCellShape);
			}
			else if (handoffRatioSim>handoffRatioTheory*1.5)
			{
				g.setColor(blueShadow);
				g.fill(n.voronoiCellShape);				
			}
			else
			{
				double min=handoffRatioTheory*0.5;
				double max=handoffRatioTheory*1.5;

				double lambda=(handoffRatioSim-min)/(max-min);
				if (lambda<0.5)
					g.setColor(new Color(1f,0f,0f,(float)(0.5-lambda)));
				else
					g.setColor(new Color(0f,0f,1,(float)(lambda-0.5)));
				g.fill(n.voronoiCellShape);				
			}
		}		
	}
	
	void paintEnergyShade(Graphics2D g)
	{
		double max=1;
		for (GeomNode n:krn.nodes)
		{
			if (n.energyUsed>max)
				max=n.energyUsed;
		}		
		for (GeomNode n:krn.nodes)
		{
			double lambda=n.energyUsed/max;
			if (lambda<0.5)
				g.setColor(new Color(1f,0f,0f,(float)(0.5-lambda)));
			else
				g.setColor(new Color(0f,0f,1,(float)(lambda-0.5)));
			g.fill(n.voronoiCellShape);				
		}		
	}

	void paintPredictionShade(Graphics2D g)
	{
		Color redShadow=new Color(0x77ff0000,true);
		Color blueShadow=new Color(0x770000ff,true);
		for (GeomNode n:krn.nodes)
		{
			double conRatioSim=1-((double)n.badHandoffCount/n.handoffCount);
			double conRatioTheory;
			if (krn.useIndirectHandoff)
				conRatioTheory=n.connIndirectRatio;
			else
				conRatioTheory=n.connRatio;
			if (conRatioSim<conRatioTheory*0.5)
			{
				g.setColor(redShadow);
				g.fill(n.voronoiCellShape);
			}
			else if (conRatioSim>conRatioTheory*1.5)
			{
				g.setColor(blueShadow);
				g.fill(n.voronoiCellShape);				
			}
			else
			{
				double min=conRatioTheory*0.5;
				double max=conRatioTheory*1.5;

				double lambda=(conRatioSim-min)/(max-min);
				if (lambda<0.5)
					g.setColor(new Color(1f,0f,0f,(float)(0.5-lambda)));
				else
					g.setColor(new Color(0f,0f,1,(float)(lambda-0.5)));
				g.fill(n.voronoiCellShape);				
			}
		}		
	}
	
	void realPainter(Graphics gold, boolean fast)
	{
		isPainting=true;
		actionQueue.process();
		Graphics2D g=(Graphics2D)gold;
		if (fast)
		{
			g.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_OFF);
			drawImage(g);
			g.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
		}
		else
		{
			AffineTransform a=g.getTransform();
			g.transform(af);
			if (nearest!=null)
			{
				drawNearest(g,false);
			}
	//		g.setXORMode(Color.white);
			// draw base station
			drawBS(g,true);
			if (krn.showRoi)
			{
				drawRois(g,false);
			}
			g.setTransform(a);
			drawKernel(g);
		}
		g.transform(af);
		// drawNodes
		if (krn.showHoffShade)
		{
			paintHandoffShade(g);
		}
		if (krn.showPredictionShade)
		{
			paintPredictionShade(g);
		}
		if (krn.showEnergyShade)
		{
			paintEnergyShade(g);
		}
		g.setColor(Color.gray);
		drawArrows(g);
		for (GeomNode n:krn.nodes)
		{
			g.setColor(Color.black);
			if (n.active)
				g.fillOval((int)n.pos.x-3, (int)n.pos.y-3, 6, 6);
			else
			{
				g.setColor(Color.white);
				g.fillOval((int)n.pos.x-3, (int)n.pos.y-3, 6, 6);
				g.setColor(Color.black);
				g.drawOval((int)n.pos.x-3, (int)n.pos.y-3, 6, 6);
			}
			if (krn.showApproxPosition)
			{
				g.setColor(new Color(0x770000ff,true));
				g.fillOval((int)n.approxPos.x-3, (int)n.approxPos.y-3, 6, 6);
			}
			if (krn.showBuffers)
			{
				g.setColor(Color.white);
				g.fillRect((int)n.pos.x+5, (int)n.pos.y-5, 5, 10);
				g.setColor(Color.black);
				g.drawRect((int)n.pos.x+5, (int)n.pos.y-5, 5, 10);
				int nBuffer=n.getBufferedPacketCount();
				g.fillRect((int)n.pos.x+5, (int)n.pos.y+5-nBuffer*10/krn.bufferSize, 5, nBuffer*10/krn.bufferSize);
			}
		}
		
		if (krn.showCellStats)
		{
			g.setColor(Color.black);
			for (GeomNode n:krn.nodes)
			{
				Point2D p=n.voronoiCellShape.getCenterOfMass();
				g.drawString(n.badHandoffCount+"/"+n.handoffCount, (int)p.getX(), (int)p.getY());
			}
		}
		if (mousePos!=null)
		{
			//g.drawLine((int)krn.bs.pos.x, (int)krn.bs.pos.y, mousePos.x,mousePos.y);
//			g.drawOval(mousePos.x-(int)krn.range,mousePos.y-(int)krn.range, (int)krn.range*2, (int)krn.range*2);
//			g.drawOval(mousePos.x-(int)krn.range/2,mousePos.y-(int)krn.range/2, (int)krn.range, (int)krn.range);
//			g.drawOval(mousePos.x-(int)krn.range+20,mousePos.y-(int)krn.range+20, (int)krn.range*2-40, (int)krn.range*2-40);
			//g.drawLine((int)mousePos.x, (int)mousePos.y, (int)nearest.pos.x, (int)nearest.pos.y);
		}

		if (fast)
		{
			if (nearest!=null)
			{
				drawNearest(g,true);
			}
	//		g.setXORMode(Color.white);
			// draw base station
			drawBS(g,true);
			if (krn.showRoi)
			{
				drawRois(g,true);
			}
		}

		if (krn.useProwler)
		{
			if (krn.showPacketOrigins)
			{
				g.setColor(Color.blue);
				for (BS bs:krn.bss)
				{
					TrailBSApplication bsApp=(TrailBSApplication)bs.trailApp;
					DataMessage []mic=new DataMessage[0];
					mic=bsApp.recentMessages.values().toArray(mic);
					for (DataMessage dm:mic)
					{
						GeomNode n=((TrailNode)dm.originator).getTrailApp().baseNode;
						GeomNode m=bs;
						drawArrow(g, n, m);
					}
				}
			}
			if (krn.showSnoopedPackets)
			{
				g.setColor(Color.green);
				Stroke s=g.getStroke();
				Stroke s2=new BasicStroke(5);
				g.setStroke(s2);
				for (BS bs:krn.bss)
				{
					TrailBSApplication bsApp=(TrailBSApplication)bs.trailApp;
					DataMessage []mic=new DataMessage[0];
					mic=bsApp.snoopedMessages.values().toArray(mic);
					for (DataMessage dm:mic)
					{
						GeomNode n=((TrailNode)dm.from).getTrailApp().baseNode;
						GeomNode m=((TrailNode)dm.to).getTrailApp().baseNode;
						drawArrow(g, n, m);
					}
				}
				g.setStroke(s);
			}
			
			for (GeomNode gn:krn.nodes)
			{
				g.setColor(Color.yellow);
				if (((TrailNode)gn.trailApp.getNode()).isTransmitting())
				{
					for (int i=1;i<3;i++)
						g.drawOval((int)(gn.pos.x-i*10), (int)(gn.pos.y-i*10), i*20, i*20);
				}
			}
		}
		drawMessages(g);
/*		String disp="AverageHops="+avgHops;
		if (nearest!=null)
		{
			disp=disp+" NearestNode= "+nearest.id+" VoronoiArea="+nearest.voronoiCellArea;
		}
		g.setColor(Color.blue);
		g.drawString(disp,0,10);*/
		isPainting=false;		
	}

	public void paintComponent(Graphics gold)
	{
		realPainter(gold, useFastPaint);
	}
	
	void handleMouseMotion(MouseEvent e)
	{
		mousePos=e.getPoint();
		try
		{
			af.inverseTransform(mousePos,mousePos);
		} catch (NoninvertibleTransformException e1)
		{
			// TODO Auto-generated catch block
			e1.printStackTrace();
		}
		// deal with the mouse move here
		// nearest=null;
		double mindist=Double.MAX_VALUE;
		for (GeomNode n:krn.nodes)
		{
			if (n.pos.distance(mousePos)<mindist)
			{
				nearest=n;
				mindist=n.pos.distance(mousePos);
			}
		}
		//setToolTipText("Node "+nearest.id);
		parent.doUpdates();
	}
	
	void handleMouseWheel(MouseWheelEvent e)
	{
		Point mousePos=e.getPoint();
		try
		{
			af.inverseTransform(mousePos,mousePos);
		} catch (NoninvertibleTransformException e1)
		{
			// TODO Auto-generated catch block
			e1.printStackTrace();
		}
		af.translate(mousePos.x, mousePos.y);
		if (e.getWheelRotation()<0)
			af.scale(0.9, 0.9);
		else if (e.getWheelRotation()>0)
			af.scale(1/0.9,1/0.9);
		af.translate(-mousePos.x, -mousePos.y);
		img=null;
	}

	public void mouseClicked(MouseEvent e) 
	{
		// TODO Auto-generated method stub
	}

	public void mouseEntered(MouseEvent e) 
	{
		// TODO Auto-generated method stub
		
	}

	public void mouseExited(MouseEvent e) 
	{
		// TODO Auto-generated method stub
		
	}

	public void mousePressed(MouseEvent e) 
	{
		// TODO Auto-generated method stub
	}

	public void mouseReleased(MouseEvent e) 
	{
		// TODO Auto-generated method stub
		
	}

	public void mouseDragged(MouseEvent e) 
	{
		// TODO Auto-generated method stub	
	}

	public void mouseMoved(MouseEvent e) 
	{
		actionQueue.addItem(e);
		repaint();
	}

	public void mouseWheelMoved(MouseWheelEvent arg0)
	{
		actionQueue.addItem(arg0);
		repaint();
	}

	public int print(Graphics g, PageFormat pf, int page) throws PrinterException
	{
		
		if (page > 0)
		{ /* We have only one page, and 'page' is zero-based */
			return NO_SUCH_PAGE;
		}
		
		/*
		 * User (0,0) is typically outside the imageable area, so we must
		 * translate by the X and Y values in the PageFormat to avoid clipping
		 */
		Graphics2D g2d = (Graphics2D) g;
		double scaledHeight=((double)getHeight())/getWidth()*pf.getImageableWidth();
		g2d.setClip((int)pf.getImageableX(), (int)pf.getImageableY(), (int)pf.getWidth(), (int)scaledHeight);
		g2d.translate(pf.getImageableX(), pf.getImageableY());
		double scale=((double)pf.getImageableWidth())/getWidth();
		g2d.scale(scale, scale);
		
		/* Now we perform our rendering */
		//g.drawString("Hello world!", 100, 100);
		realPainter(g,false);
		
		/* tell the caller that this page is part of the printed document */
		return PAGE_EXISTS;
	}

	public void doPrint()
	{
		System.out.println("Begining Print...");
/*		net.sf.epsgraphics.Drawable d=new net.sf.epsgraphics.Drawable()
		{
			 public void draw(java.awt.Graphics2D g2, java.awt.geom.Rectangle2D area) 
			 {
				 g2.setClip(0, 0, getWidth(), getHeight());
				 realPainter(g2,false);				 
			 }
		};
		net.sf.epsgraphics.EpsTools.createFromDrawable(d, "d:\\onur\\asdf.ps", getWidth(), getHeight(), net.sf.epsgraphics.ColorMode.COLOR_RGB);
*/
	    PrinterJob job = PrinterJob.getPrinterJob();
        job.setPrintable(this);
        boolean ok = job.printDialog();
        if (ok) {
            try {
                 job.print();
            } catch (PrinterException ex) {
             /* The job did not successfully complete */
            }
        }
    
		System.out.println("Printing Complete");
	}

}
