import java.awt.geom.*;
import java.lang.reflect.Field;
import java.util.*;

public class TrailKernel
{
    Vector<GeomNode> nodes;
	Vector<Field> setFields;
	Random rnd;
	/** whether this is applet or not 
	 * Warning!!! don't change this explicitly 
	 */
	boolean isApplet=false;
	/** Base station */
	Vector<BS> bss;
	/** Regions of interest */
	Vector<ROI> rois;
	/** width of the arena */
	int width=800;
	/** height of the arena */
	int height=600;
	/** number of node */
	int nNodes=300;
	/** random number seed */
	int seed=42;
	/** communication range */
	double range=90; 
	/** singleton for easy access */
	static TrailKernel krn;
	/** total area of Voronoi cells for sanity checks */
    double totalArea=0; 
    /** circumference */
    double totalCircumference=0;
    /** connection score (badarea*badCirc)/(totalArea*totalCirc) */
    double connectionScore=0;
    /** what happens with indirect handoffs */
    double indirectConnectionScore=0;
    /** ratio of bad edges */
    double badEdgeRatio=0;
    /** handoff count used in detailed simulations */
    int handoffcount=0;
    /** disconnection count */
    int disconnectcount=0;
    /** current Simulation time */
    double time=0;
    
    /** number of steps to run simulation (for batch mode) */
    int nSteps=0;
    /** batch mode or not */
    boolean hideDisplay=false;
    /** whether to use indirect handoffs */
    boolean useIndirectHandoff=true;
    /** whether to color cells in display */
    boolean colorCells=false;
    boolean showCellBorders=true;
    /** whether to show arrows or not */
    boolean showArrows=true;
    /** whether to highlight nodes in range */
    boolean showInRange=false;
    /** whether to show hop counts or not */
    boolean showHopCount=false;
    /** whether to show show cell stats (number of crosses / disconnects per cell)*/
    boolean showCellStats=false;
    /** whether to show handoff shade */
    boolean showHoffShade=false;
    /** whether to show energy use shade */
    boolean showEnergyShade=false;
    /** whether to show prediction shade */
    boolean showPredictionShade=false;
    /** whether to show buffers on nodes */
    boolean showBuffers=false;
    /** whether to show region of interest */
    boolean showRoi=true;
    /** whether to show approximate positions of nodes */
    boolean showApproxPosition=false;
    /** whether to show package origins for prowler sim */
    boolean showPacketOrigins=false;
    /** whether to show package trails */
    boolean showPacketTrails=true;
    /** whether to show snooped packets for follow flow */
    boolean showSnoopedPackets=false;
    /** whether to show neighbors */
    boolean showNeighbors=false;
    /** whether to show common locations for Human like ROI */
    boolean showHumanLikeROIDetail=false;
    /** Basestation tries to follow the data generation */
    boolean useFollowData=true;
    /** Basestation tries to follow the data flow */
    boolean useFollowFlow=false;
    /** Basestation knows the com of roi */
    boolean useOracle=false;
    /** Basestation only updates one edge as in DataSalmon */
    boolean useUnicastRouteUpdate=false;
    /** whether to use random waypoint method */
    boolean useRandomWPMethod=false;
    /** whether to use edge weights */
    boolean useEdgeWeights=false;
    /** whether to use edge length correction (only meaningful when edge weights are not used */
    boolean useEdgeLengthCorrection=false;
    /** whether to use extremely simple ratio (disregarding handoff ratios) supersedes other methods*/
    boolean useSimpleRatioApproximation=false;
    /** whether to use position approximation */
    boolean usePositionApproximation=false;
    /** whether to use random anchors */
    boolean useRandomAnchors=false;
    /** whether to use Prowler */
    boolean useProwler=true;
    /** whether to use Acknowledgments */
    boolean useAcknowledgments=false;
    /** whether to use bs snooping to reduce anchor load */
    boolean useBSSnooping=true;
    /** whether to place bs in center */
    boolean useOptimizedBSPosition=false;
    /** bs randomwalk avoids crossing bad edges */
    boolean useSmartRandomWalk=true;
    /** roi uses humanlike mobility */
    boolean useHumanLikeRoi=true;
    /** roi teleports */
    boolean useTeleportingRoi=false;
    /** roi size is shared among multiple rois if present */
    boolean useShareROISize=true;
    /** whether to print all or just display */
    boolean printAll=true;
    /** do an initial flood to set links */
    boolean doInitialFlood=true;
    /** update probability of random anchors */
    double anchorUpdateProbability=0.05;
    /** ROI Teleport Period in seconds*/
    int roiTeleportPeriod=600;
    /** The ratio of data generated in ROI */
    double dataLocality=1.0;
    /** size of buffer on nodes */
    int bufferSize=100;
    /** single hop bandwidth in number of packets per timestep*/
    int bandwidth=100;
    /** Data Generation rate in packets / second */
    double dataGenerationRate=0.01; // 1 packet per 100 seconds
    /** Message Queue check rate in checks per second */
    double messageQueueCheckRate=10; // 1 check per 5 seconds
    /** size of the roi */
    double ROIsize=100;
    /** number of roi */
    int roiCount=1;
    /** number of bs */
    int bsCount=1;
    /** number of holes in the arena */
    int holeCount=0;
    /** bs speed */
    double bsSpeed=1;
    /** bs speed */
    double roiSpeed=0.2;
    /** Preset configuration */
    String presetConfiguration="trail";
    /** number of packets generated */
    int packetsGenerated=0;
    /** total number of messages sent */
    int messagesSent=0;
    /** total number of packets dropped */
    int packetsDropped=0;
    /** total number of packets delivered */
    int packetsDelivered=0;
    /** total of all delays in deliveries */
    double totalDelay=0;
    /** delivery ratio */
    double deliveryRatio=0;
    /** average Delay */
    double averageDelay=0;
    /** throughput in packets/sec */
    double throughput=0;
    /** maximum energy use in the network */
    double maxEnergyUse=0;
    /** expected time (days) till first node death */
    double expectedLifeTime=0;
    /** Prowler Simulation Wrapper */
    ProwlerDriver prowlerDriver;
    /** lines not to cross */
    Vector<Line2D.Double> linesNotToCross; 
    
    String [] getHelpText()
    {
    	Vector<String> strings=new Vector<String>();
    	strings.add("Usage:");
    	strings.add("-seed                : random number seed  [default="+seed+"]");
		strings.add("-nNodes              : number of nodes     [default="+nNodes+"]");
		strings.add("-range               : communication range [default="+range+"]");
		strings.add("-nSteps              : number of sim steps [default="+nSteps+"]");
		strings.add("-hideDisplay         : hide gui            [default="+hideDisplay+"]");
		strings.add("-useIndirectHandoff  : better handoff alg  [default="+useIndirectHandoff+"]");
		strings.add("WARNING: Below are expert options. Results may vary use at your own risk.");
		strings.add("NOTE: for booleans 'true' and 'false' should explicitly stated for expert options");
		Field[] fields=this.getClass().getDeclaredFields();
		for (Field f:fields)
		{
			String valueStr="";
			try 
			{
				if(f.getGenericType().toString().compareTo("int")==0)
				{
						valueStr=""+f.getInt(this);
				}
				else if(f.getGenericType().toString().compareTo("boolean")==0)
				{
						valueStr=""+f.getBoolean(this);
				}
				else if(f.getGenericType().toString().compareTo("double")==0)
				{
						valueStr=""+f.getDouble(this);
				}
			} 
			catch (Exception e) 
			{
			} 
			if (valueStr.length()>0)
				strings.add("-"+f.getName()+"   :<"+f.getGenericType().toString()+">  [default="+valueStr+"]");
		}
    	return strings.toArray(new String[0]);
    }
	
    void printHelp()
    {
    	String [] strings=getHelpText();
    	for (String str:strings)
    	{
    		System.out.println(strings);
    	}
		System.exit(0);
    }
    
    String [] getParameterText()
    {
    	LinkedList<String> strings=new LinkedList<String>();
		Field[] fields=this.getClass().getDeclaredFields();
		for (Field f:fields)
		{
			try
			{
				if (f.getType().isPrimitive())
					strings.add(f.getName()+"="+f.get(this));
			}catch(IllegalAccessException e)
			{
				// skip broken names
			}
		}
		return strings.toArray(new String[0]);
    }
    
    /**
     * This function dumps all kernel parameters
     */
    void dumpParameters()
    {
    	String [] strings=getParameterText();
    	for (String str:strings)
    		System.out.println(str);
    }
    /**
     * This function tries to fill other parameters not specified explicitly
     * @param args the commandline arguments
     * @param idx current item
     * @return whether found a parameter or not
     */
    boolean parseOther(String args[],int idx)
    {
		Field[] fields=this.getClass().getDeclaredFields();
		for (Field f:fields)
		{
			String cline="-"+f.getName();
			if (cline.compareTo(args[idx])==0)
			{
				try 
				{
					if(f.getGenericType().toString().compareTo("int")==0)
					{
						f.setInt(this, Integer.parseInt(args[idx+1]));
						setFields.add(f);
						return true;
					}
					else if(f.getGenericType().toString().compareTo("boolean")==0)
					{
						f.setBoolean(this, Boolean.parseBoolean(args[idx+1]));
						return true;
					}
					else if(f.getGenericType().toString().compareTo("double")==0)
					{
						f.setDouble(this, Double.parseDouble(args[idx+1]));
						return true;
					}
				}
				catch (Exception e) 
				{
				} 
			}
		}    	
		return false;
    }
    
    boolean isFieldSet(String fieldName)
    {
    	for (Field f:setFields)
    	{
    		if (f.getName().compareTo(fieldName)==0)
    		{
    			return true;
    		}
    	}
    	return false;
    }

    void parseArgs(String args[])
    {

	    setFields=new Vector<Field>();
	    
    	if (args!=null)
    	{
	    	for (int i=0;i<args.length;i++)
	    	{
	    		if (args[i].compareTo("-hideDisplay")==0)
	    			hideDisplay=true;
	    		else if (args[i].compareTo("-presetConfiguration")==0)
	    		{
	    			presetConfiguration=args[++i];
	        		loadPreset();
	    		}
	    		else if (parseOther(args,i)) // got an expert parameter
	    			i++;
	    		else
	    		{
	    			System.out.println("invalid argument '"+args[i]+"'");
	    			printHelp();
	    		}
	    	}
    	}
    	if (!isFieldSet("bufferSize"))
    	{
	    	if (nNodes>=200)
	    		bufferSize=100*nNodes/200;
	    	else
	    		bufferSize=100;
    	}
    	if (!isFieldSet("bandwidth"))
    		bandwidth=bufferSize;
    }
    
	private void loadPreset()
	{		
		if (presetConfiguration.compareTo("trailSource")==0)
		{
			useFollowData=true;
			useFollowFlow=false;
			useOracle=false;
			useUnicastRouteUpdate=false;
			useOptimizedBSPosition=true;
		}
		else if (presetConfiguration.compareTo("trailFlow")==0)
		{
			useFollowData=false;
			useFollowFlow=true;
			useOracle=false;
			useUnicastRouteUpdate=false;
			useOptimizedBSPosition=true;
		}
		else if (presetConfiguration.compareTo("static")==0)
		{
			useFollowData=false;
			useFollowFlow=false;
			useOracle=false;
			useUnicastRouteUpdate=false;
			bsSpeed=0;
			useOptimizedBSPosition=true;
		}
		else if (presetConfiguration.compareTo("salmon")==0)
		{
			useFollowData=false;
			useFollowFlow=true;
			useOracle=false;
			useUnicastRouteUpdate=true;
			useOptimizedBSPosition=true;
		}
		else if (presetConfiguration.compareTo("random")==0)
		{
			useFollowData=false;
			useFollowFlow=false;
			useOracle=false;
			useUnicastRouteUpdate=false;
			useOptimizedBSPosition=true;
		}
		else if (presetConfiguration.compareTo("oracle")==0)
		{
			useFollowData=false;
			useFollowFlow=false;
			useOracle=true;
			useUnicastRouteUpdate=false;
			useOptimizedBSPosition=true;			
		}
		
		else
		{
			System.out.println("invalid preset configuration '"+presetConfiguration+"'");
			printHelp();			
		}
	}

	TrailKernel(String args[])
	{
		parseArgs(args);
		TrailKernel.krn=this;
		rnd=new Random(seed);
		nodes=new Vector<GeomNode>();
		for (int i=0;i<nNodes;i++)
		{
			GeomNode n=new GeomNode();
			nodes.add(n);
		}
		rois=new Vector<ROI>();
		for (int i=0;i<roiCount;i++)
		{
			ROI roi;
			if (useTeleportingRoi)
				roi=new TeleportingROI(rnd.nextDouble()*width,rnd.nextDouble()*height);
			else if (useHumanLikeRoi)
				roi=new SmartROI(rnd.nextDouble()*width,rnd.nextDouble()*height);
			else
				roi=new ROI(rnd.nextDouble()*width,rnd.nextDouble()*height);
			rois.add(roi);
		}
		distributeBSs();
		distributeNodes();
		prowlerDriver=new ProwlerDriver(this);
		linesNotToCross=new Vector<Line2D.Double>();
		compute();
		fillStats();
		if (useSmartRandomWalk)
			fillLinesNotToCross();
		rnd=new Random(seed);/** NOTE: This is to ensure behavior is the same after node distribution*/
	}

	private void distributeBSs()
	{
		bss=new Vector<BS>();
		if (useOptimizedBSPosition)
		{
			if (bsCount==1)
			{
				bss.add(new BS(width/2,height/2));
			}
			else if (bsCount==2)
			{
				bss.add(new BS(width/4, height/2));
				bss.add(new BS(width*3.0/4,height/2));
			}
			else if (bsCount==3)
			{
				bss.add(new BS(width/3, height*2.0/3));
				bss.add(new BS(width/2, height/3));
				bss.add(new BS(width*2.0/3, height*2.0/3));
			}
			else if (bsCount==4)
			{
				bss.add(new BS(width/4, height/4));
				bss.add(new BS(width*3.0/4,height/4));
				bss.add(new BS(width/4, height*3.0/4));
				bss.add(new BS(width*3.0/4,height*3.0/4));
			}
			else
			{
				Point2D.Double bestLocation[]=new Point2D.Double[bsCount];
				Point2D.Double location[]=new Point2D.Double[bsCount];
				Point2D.Double corners[]=new Point2D.Double[4];
				double bestScore=Double.POSITIVE_INFINITY;
				corners[0]=new Point2D.Double(0,0);
				corners[1]=new Point2D.Double(0,width);
				corners[2]=new Point2D.Double(height,width);
				corners[3]=new Point2D.Double(height,0);
				for (int i=0;i<bsCount;i++)
				{
					location[i]=new Point2D.Double();
					bestLocation[i]=new Point2D.Double();
				}
				for (int tries=0;tries<100000;tries++)
				{
					for (int i=0;i<bsCount;i++)
					{
						location[i].x=rnd.nextDouble()*width;
						location[i].y=rnd.nextDouble()*height;
					}
					double score=0;
					for (int j=0;j<4;j++)
					{
						double dist=Double.POSITIVE_INFINITY;
						for (int i=0;i<bsCount;i++)
						{
							double d=location[i].distance(corners[j]);
							if (d<dist)
								dist=d;
						}
						score+=dist;
					}
					for (int i=0;i<bsCount;i++)
					{
						double dist=Double.NEGATIVE_INFINITY;
						for (int j=0;j<bsCount;j++)
						{
							if (i!=j)
							{
								double d=location[i].distance(location[j]);
								if (d>dist)
									dist=d;
							}
						}
						score+=dist;
					}
					if (bestScore>score)
					{
						for (int i=0;i<bsCount;i++)
						{
							bestLocation[i].x=location[i].x;
							bestLocation[i].y=location[i].y;
						}
						bestScore=score;
					}
				}
				for (int i=0;i<bsCount;i++)
				{
					BS bs;
					bs=new BS(bestLocation[i].x,bestLocation[i].y);
					bss.add(bs);
				}
			}
		}
		else
		{
			for (int i=0;i<bsCount;i++)
			{
				BS bs;
				bs=new BS(rnd.nextDouble()*width,rnd.nextDouble()*height);
				bss.add(bs);
			}			
		}
	}
	
	private void fillLinesNotToCross()
	{
		for (GeomNode n:nodes)
		{
			for (DelunayNeighbor dn:n.neighs)
			{
				
				if (dn.node!=null && dn.node.id<n.id)
				{
					if (dn.type==DelunayNeighbor.DelunayNeighborType.BadEdge || dn.type==DelunayNeighbor.DelunayNeighborType.IndirectEdge)
					{
						Line2D.Double l=new Line2D.Double(dn.p, dn.q);
						linesNotToCross.add(l);
					}
				}
			}
		}
	}

	void distributeNodes()
	{
		Point2D.Double holes[]=new Point2D.Double[holeCount];
		double holeSizes[]=new double[holeCount];
		for (int i=0;i<holeCount;i++)
		{
			holes[i]=new Point2D.Double(rnd.nextDouble()*width, rnd.nextDouble()*height);
			holeSizes[i]=rnd.nextDouble()*200;
		}
		
		for (GeomNode n:nodes)
		{
			boolean isInAHole;
			do
			{				
				isInAHole=false;
				n.setPos(rnd.nextDouble()*width, rnd.nextDouble()*height);
				for (int i=0;i<holeCount;i++)
				{
					if (n.pos.distance(holes[i])<holeSizes[i])
					{
						isInAHole=true;
						break;
					}
				}
			}while(isInAHole);
		}
		if (krn.usePositionApproximation)
		{
			Random rand=new Random(seed+123); // use a separate random number generator
			for (GeomNode n:nodes)
			{
				n.approxPos.x=rand.nextDouble()*width;
				n.approxPos.y=rand.nextDouble()*height;
			}
		}
	}
	
	/**
	 * find intersection of line segment between p and q
	 * @param p first point of line segment (this is assumed to be in box)
	 * @param q second point of line segment
	 * @param qprime intersection point value set
	 * @return true if there is an intersection. false otherwise
	 */
	final boolean getBoxIntersection(Point2D.Double p,Point2D.Double q,Point2D.Double qprime)
	{
		Point2D.Double v=new Point2D.Double();
		double lambda;
    	qprime.x=q.x;
    	qprime.y=q.y;
		v.x=qprime.x-p.x;
		v.y=qprime.y-p.y;
    	if (qprime.x<0 || qprime.x>width) // make x inside
    	{
    		if (qprime.x<0)
    			lambda=(0-p.x)/v.x;
    		else
    			lambda=(width-p.x)/v.x;
    		qprime.x=v.x*lambda+p.x;
    		qprime.y=v.y*lambda+p.y;
    	}
    	if (qprime.y<0 || qprime.y>height) // if y is still not fitting fix it
    	{
    		if (qprime.y<0)
    			lambda=(0-p.y)/v.y;
    		else
    			lambda=(height-p.y)/v.y;
    		qprime.x=v.x*lambda+p.x;
    		qprime.y=v.y*lambda+p.y;
    	}
    	return true;
	}
	
	/** This is seriously flawed
	 * it is only different order
	 * @see getBoxIntersection 
	 * @param p
	 * @param q
	 * @param qprime
	 * @return
	 */
	final boolean getBoxIntersectionInverseOrder(Point2D.Double p,Point2D.Double q,Point2D.Double qprime)
	{
		Point2D.Double v=new Point2D.Double();
		double lambda;
    	qprime.x=q.x;
    	qprime.y=q.y;
		v.x=qprime.x-p.x;
		v.y=qprime.y-p.y;
    	if (qprime.x<0 || qprime.x>width) // make x inside
    	{
    		if (qprime.x>width)
    			lambda=(width-p.x)/v.x;
    		else
    			lambda=(0-p.x)/v.x;
    		qprime.x=v.x*lambda+p.x;
    		qprime.y=v.y*lambda+p.y;
    	}
    	if (qprime.y<0 || qprime.y>height) // if y is still not fitting fix it
    	{
    		if (qprime.y>height)
    			lambda=(height-p.y)/v.y;
    		else
    			lambda=(0-p.y)/v.y;
    		qprime.x=v.x*lambda+p.x;
    		qprime.y=v.y*lambda+p.y;
    	}
    	return true;
	}
	
	final double signumstar(double x)
	{
		if (x<0)
			return -1;
		else
			return 1;
	}
	
	final boolean getLineSegmentCircleIntersection(Point2D.Double c, double r, Point2D.Double p, Point2D.Double q, Point2D.Double qprime)
	{
		double x1=p.x-c.x;
		double x2=q.x-c.x;
		double y1=p.y-c.y;
		double y2=p.y-c.x;
		double dx=x2-x1;
		double dy=y2-y1;
		double dr=Math.sqrt(dx*dx+dy*dy);
		double D=x1*y2-y1*x2;
		double delta=r*r*dr*dr-D*D;
		if (delta<=0)
			return false;
		double deltaroot=Math.sqrt(delta);
		double resx1=(D*dy+signumstar(dy)*dx*deltaroot)/(dr*dr);
		double resx2=(D*dy-signumstar(dy)*dx*deltaroot)/(dr*dr);
		double resy1=(-D*dx+Math.abs(dy)*deltaroot)/(dr*dr);
		double resy2=(-D*dx-Math.abs(dy)*deltaroot)/(dr*dr);
		double d1=(x2-resx1)*(x2-resx1)+(y2-resy1)*(y2-resy1);
		double d2=(x2-resx2)*(x2-resx2)+(y2-resy2)*(y2-resy2);
		if (d1<d2)
		{
			qprime.x=resx1+c.x;
			qprime.y=resy1+c.y;
		}
		else
		{
			qprime.x=resx2+c.x;
			qprime.y=resy2+c.y;
		}
		return true;
	}
	
	boolean doesLinesSegmentsIntersect(Point2D.Double p1,Point2D.Double q1,Point2D.Double p2,Point2D.Double q2)
	{
		Line2D l1=new Line2D.Double(p1,q1);
		Line2D l2=new Line2D.Double(p2,q2);
		return l1.intersectsLine(l2);
	}
	
	void computeIndirects()
	{
		for (GeomNode n:krn.nodes)
		{
			for (DelunayNeighbor dn:n.neighs)
			{
				if (dn.node!=null && dn.type!=DelunayNeighbor.DelunayNeighborType.ProperEdge)
				{
					// there might be indirect neighbors
					Point2D.Double p=new Point2D.Double(dn.p.x,dn.p.y);					
					Point2D.Double q=new Point2D.Double(dn.q.x,dn.q.y);
					boolean allcovered=false;
					for (GeomNode n2:nodes)
						if (n2!=n && n2!=dn.node &&
							n2.pos.distance(n.pos)<=krn.range &&
							n2.pos.distance(dn.node.pos)<=krn.range)
						{ // wow factor.
							double dist2p=n2.pos.distance(p);
							double dist2q=n2.pos.distance(q);
							if (dist2p<=krn.range && dist2q<=krn.range)
							{
								allcovered=true;
								break;
							}
							else if (dist2p<=krn.range)
							{
								getLineSegmentCircleIntersection(n2.pos, krn.range, p, q, q);
								if (p.distance(q)<1e-20)
								{
									allcovered=true;
									break;
								}
							}
							else if (dist2q<=krn.range)
							{
								getLineSegmentCircleIntersection(n2.pos, krn.range, q, p, p);
								if (p.distance(q)<1e-20)
								{
									allcovered=true;
									break;
								}
							}
						}
					if (allcovered)
						dn.type=DelunayNeighbor.DelunayNeighborType.IndirectEdge;
					else
					{
						double d=1-p.distance(q)/dn.p.distance(dn.q);
						dn.indirectCoverage=d;
						//System.out.println(d);
					}
				}
			}
		}
	}
	
	void compute()
	{
	    int initialSize = 10000;      // Controls size of initial triangle
	    Simplex<Pnt> initialTriangle; // The large initial triangle
		DelaunayTriangulation dt;     // The Delaunay triangulation
		initialTriangle = new Simplex<Pnt>(new Pnt(-initialSize, -initialSize),
                new Pnt( initialSize, -initialSize),
                new Pnt(           0,  initialSize));
        dt = new DelaunayTriangulation(initialTriangle);

        for (GeomNode n:nodes) // do actual computation
		{
        	Pnt point = new Pnt(n);
            dt.delaunayPlace(point);
		}
        
        // Loop through all the edges of the DT (each is done twice)
        Point2D.Double pd=new Point2D.Double(),qd=new Point2D.Double();
        Point2D.Double pprime=new Point2D.Double();
        Point2D.Double pprime2=new Point2D.Double();
        Point2D.Double qprime=new Point2D.Double();
        Point2D.Double qprime2=new Point2D.Double();
		Point2D.Double corners[]=new Point2D.Double[4];
		corners[0]=new Point2D.Double(0,0); 
		corners[1]=new Point2D.Double(width,0);
		corners[2]=new Point2D.Double(width,height);
		corners[3]=new Point2D.Double(0,height); 

		for (Simplex<Pnt> triangle: dt)
        {
        	for (Simplex<Pnt> other: dt.neighbors(triangle))
        	{
        		if (other.hashCode()>triangle.hashCode())
        			continue;// only process once
	            Pnt p = Pnt.circumcenter(triangle.toArray(new Pnt[0]));
	            Pnt q = Pnt.circumcenter(other.toArray(new Pnt[0]));
	            Set<Pnt> common=triangle.intersect(other);
	            if (common.size()!=2)
	            {
	            	System.out.println("darn you lemmings");
	            }
	            Pnt ends[]=common.toArray(new Pnt[2]);

	            // NOTE 
	        	// This stuff should be handled as follows let n be node pos
	        	// if p is outside but q is in
	        	// add another vertex p' at boundary (where [pq] intersects)
	        	// create a triangle with this (p',q,n)
	        	// add another vertex p'' at boundary (where [pn] intersects)
	        	// create a triangle (p',p'',n)
	        	// if both outside
	        	// add two vertices p' and q' at boundary (where [pn] and [qn] intersects respectively)
	        	// create (p',q',n)
	            
	            if (p.inside(0, 0, width, height) &&
	            	q.inside(0, 0, width, height)) // both of them inside
	            {
	            	boolean isBad=ends[0].distance(ends[1])>range;
	            	DelunayNeighbor dn1,dn2;
		            dn1=new DelunayNeighbor(ends[1].n,p.coord(0),p.coord(1),q.coord(0),q.coord(1));
		            if (isBad)
		            	dn1.type=DelunayNeighbor.DelunayNeighborType.BadEdge;
		            dn2=new DelunayNeighbor(ends[0].n,p.coord(0),p.coord(1),q.coord(0),q.coord(1));
		            if (isBad)
		            	dn2.type=DelunayNeighbor.DelunayNeighborType.BadEdge;

		            ends[0].n.neighs.add(dn1);
	            	ends[1].n.neighs.add(dn2);
	            }
	            
	            else if (p.inside(0, 0, width, height) ||
		            	q.inside(0, 0, width, height)) // one of them inside
	            {
	            	if (p.inside(0, 0, width, height)) // only p inside pd/qd as is
	            	{
		            	qd.x=q.coord(0);
		            	qd.y=q.coord(1);
		            	pd.x=p.coord(0);
		            	pd.y=p.coord(1);
	            	}
	            	else // only q inside (switch them to reuse code)
	            	{
		            	qd.x=p.coord(0);
		            	qd.y=p.coord(1);
		            	pd.x=q.coord(0);
		            	pd.y=q.coord(1);	            		
	            	}
	            	getBoxIntersection(pd, qd, qprime);
	            	
	            	// add (n,p,q') to both
	            	boolean isBad=ends[0].distance(ends[1])>range;
	            	DelunayNeighbor dn1,dn2;
		            dn1=new DelunayNeighbor(ends[1].n,pd.x,pd.y,qprime.x,qprime.y);
		            if (isBad)
		            	dn1.type=DelunayNeighbor.DelunayNeighborType.BadEdge;
		            dn2=new DelunayNeighbor(ends[0].n,pd.x,pd.y,qprime.x,qprime.y);
		            if (isBad)
		            	dn2.type=DelunayNeighbor.DelunayNeighborType.BadEdge;

		            ends[0].n.neighs.add(dn1);
	            	ends[1].n.neighs.add(dn2);
	            	
	            	// add (n,q',q'') to first. Don't add the neighbor (fake neigbor) (facets were needed here anyways)
	            	getBoxIntersection(ends[0].n.pos,qd,qprime2);
		            dn1=new DelunayNeighbor(null,qprime,qprime2);
		            if (isBad)
		            	dn1.type=DelunayNeighbor.DelunayNeighborType.BadEdge;           	

	            	// add (n,q',q'') to second. Don't add the neighbor (fake neigbor) (facets were needed here anyways)
		            getBoxIntersection(ends[1].n.pos,qd,qprime2);
		            dn2=new DelunayNeighbor(null,qprime,qprime2);
		            if (isBad)
		            	dn1.type=DelunayNeighbor.DelunayNeighborType.BadEdge;           	

		            ends[0].n.neighs.add(dn1);
	            	ends[1].n.neighs.add(dn2);
	            }
	            else // none of them inside
	            {
	            	DelunayNeighbor dn;

	            	pd.x=p.coord(0);
	            	pd.y=p.coord(1);	            	
	            	qd.x=q.coord(0);
	            	qd.y=q.coord(1);

	            	if (doesLinesSegmentsIntersect(pd, qd, corners[0], corners[1]) ||
	            		doesLinesSegmentsIntersect(pd, qd, corners[1], corners[2]) ||
	            		doesLinesSegmentsIntersect(pd, qd, corners[2], corners[3]) ||
	            		doesLinesSegmentsIntersect(pd, qd, corners[3], corners[0]))
	            	{
	            		// stupid special case where both ends are outside but there is still intersection
//	            		System.out.println("hell is around the corner");
            			getBoxIntersection(qd,pd,pprime);
            			getBoxIntersectionInverseOrder(pprime,qd,qprime);
            			
	            		dn=new DelunayNeighbor(ends[1].n,pprime,qprime);
	            		ends[0].n.neighs.add(dn);
	            		dn=new DelunayNeighbor(ends[0].n,pprime,qprime);
	            		ends[1].n.neighs.add(dn);

		            	for (Pnt end:common)
		            	{
		            		getBoxIntersection(end.n.pos,pd,pprime2);
		            		getBoxIntersection(end.n.pos,qd,qprime2);
		            		dn=new DelunayNeighbor(null,pprime,pprime2);
		            		end.n.neighs.add(dn);
		            		dn=new DelunayNeighbor(null,qprime,qprime2);
		            		end.n.neighs.add(dn);
		            	}
	            	}
	            	else
	            	{		            	
		            	for (Pnt end:common)
		            	{
			            	if (end.n!=null)
			            	{ 
				            	getBoxIntersection(end.n.pos,qd,qprime);
				            	getBoxIntersection(end.n.pos,pd,pprime);
			            		dn=new DelunayNeighbor(null,pprime,qprime);
			            		end.n.neighs.add(dn);
			            	}
		            	}
	            	}
	            }
        	}
        }
        
    	// handle corner cases by further subdiving if needed
		// |--+---       *--+---
		// | /|          |. |
		// |/ |          | .|
		// +  |          + .|
		// |\ |          |\.|
		// | \|          | \|
		// |  *          |  *
		
		GeomNode cornerHandler=null;
		double bestdist=Double.MAX_VALUE;
		double dist;
		
		for (int i=0;i<corners.length;i++)
		{
			bestdist=Double.MAX_VALUE;
			for (GeomNode n:nodes)
			{
				dist=n.pos.distance(corners[i]);
				if (bestdist>dist)
				{
					bestdist=dist;
					cornerHandler=n;
				}
			}
			boolean isHandled=false;
			for (Iterator<DelunayNeighbor> j=cornerHandler.neighs.iterator();j.hasNext();)
			{
				DelunayNeighbor dn=j.next();
				if (doesLinesSegmentsIntersect(cornerHandler.pos,corners[i],dn.p,dn.q))
				{
					j.remove(); // there was a triangle that could cover this one breka it into two
					DelunayNeighbor dn2;
					dn2=new DelunayNeighbor(null,dn.p,corners[i]);
					cornerHandler.neighs.add(dn2);
					dn2=new DelunayNeighbor(null,corners[i],dn.q);
					cornerHandler.neighs.add(dn2);
					isHandled=true;
					break;
				}
			}
			if (!isHandled) 
			{
				System.out.println("This means the corner handling is not enough");
				// man this is tough now we need to find two good enough neighbors to create sub division
			}
		}
		computeIndirects();
		for (GeomNode n:nodes)
		{
			for (GeomNode m:nodes)
				if (n!=m && n.id<m.id && n.pos.distance(m.pos)<krn.range)
				{
					n.oneHopNodes.add(m);
					m.oneHopNodes.add(n);
				}
		}
		
		for (final GeomNode n:nodes)
		{
			Collections.sort(n.oneHopNodes,new Comparator<GeomNode>(){
				public int compare(GeomNode arg0, GeomNode arg1)
				{
					double d0=n.pos.distance(arg0.pos);
					double d1=n.pos.distance(arg1.pos);
					if (d0>d1)
						return 1;
					else if (d1>d0)
						return -1;
					return 0;
				}
			});
		}
		
		if (doInitialFlood && !useProwler)
		{
			for (BS bs:bss)
			{
				GeomNode nearest=nodes.get(0);
				for (GeomNode n:nodes)
				{
					if (nearest.pos.distance(bs.pos)>n.pos.distance(bs.pos))
					{
						nearest=n;
					}
				}
				nearest.next=bs;
			}
			boolean change;
			do
			{
				change=false;
				for (GeomNode n:nodes)
				{
					if (n.next==null)
					{
						for (GeomNode m:n.oneHopNodes)
						{
							if (m.next!=null)
							{
								change=true;
								n.next=m;
								break;
							}
						}
					}
				}
			}while(change);
		}
		if (krn.useEdgeWeights)
			calculateEdgeWeights();
	}
	
	void fillStats()
	{
        // do further stuff like voronoi area
		connectionScore=0;
		indirectConnectionScore=0;
		totalArea=0;
		totalCircumference=0;
		handoffcount=0;
		disconnectcount=0;
		int nedges=0;
		int badedges=0;
		double[] correction=new double[krn.nNodes];
        for (GeomNode n:nodes)
        {
        	n.voronoiCellArea=0;
        	n.voronoiCellCircum=0;
        	n.badEdgeLength=0;
        	n.badIndirectEdgeLength=0;
        	for (DelunayNeighbor dn:n.neighs)
        	{
        		// use heron's formula
        		// A=sqrt(s*(s-a)*(s-b)*(s-c)
        		// where s=(a+b+c)/2
        		double a=n.pos.distance(dn.p);
        		double b=n.pos.distance(dn.q);
        		double c=dn.p.distance(dn.q);
        		double s=(a+b+c)/2;
        		double A=Math.sqrt(s*(s-a)*(s-b)*(s-c));
        		double e;
        		if (useEdgeWeights)
        			e=dn.edgeWeight;
        		else
        			e=dn.p.distance(dn.q);
        		n.voronoiCellArea+=A;
        		if (dn.node!=null) /** NOTE: ignores boundaries in calculation*/
        		{
	        		n.voronoiCellCircum+=e;
	        		if (dn.type!=DelunayNeighbor.DelunayNeighborType.ProperEdge)
	        		{
	        			n.badEdgeLength+=e;
	        			if (dn.type!=DelunayNeighbor.DelunayNeighborType.IndirectEdge)
	        			{
	        				n.badIndirectEdgeLength+=e*(1-dn.indirectCoverage);
	        			}
	        		}
	        		if (dn.node!=null)
	        		{
	        			nedges++;
	        			if (dn.type!=DelunayNeighbor.DelunayNeighborType.ProperEdge)
	        				badedges++;
	        		}
        		}
        		else if (useEdgeLengthCorrection)
        			correction[(int)n.id]+=e;
        	}
        	totalArea+=n.voronoiCellArea;
        	totalCircumference+=n.voronoiCellCircum+correction[(int)n.id];
        }
        
        double totalBadEdgeLength=0;
        double totalIndirectBadEdgeLength=0;
        double totalEdgeLength=0;
        for (GeomNode n:krn.nodes)
        {
        	totalEdgeLength+=n.voronoiCellCircum;
        	totalBadEdgeLength+=n.badEdgeLength;
        	totalIndirectBadEdgeLength+=n.badIndirectEdgeLength;
        	n.handoffRatio=(n.voronoiCellCircum+correction[(int)n.id])/totalCircumference;
        	n.connRatio=1-n.badEdgeLength/n.voronoiCellCircum;
        	n.connIndirectRatio=1-n.badIndirectEdgeLength/n.voronoiCellCircum;
        	connectionScore+=n.handoffRatio*n.connRatio;
        	indirectConnectionScore+=n.handoffRatio*n.connIndirectRatio;
        }
        badEdgeRatio=1-((double)badedges)/nedges;
        if (useSimpleRatioApproximation)
        {
        	connectionScore=1-(totalBadEdgeLength/totalEdgeLength);
        	indirectConnectionScore=1-(totalIndirectBadEdgeLength/totalEdgeLength);
        }
        //System.out.println("Total Area= "+totalArea+" Total Circumference="+totalCircumference+" Connection Score="+connectionScore);
	}
	
	void printStats()
	{
		double disconnectRatio=0;
		double dropRate=0;
		double averageHops=0;
		double averageDelay=0;
		if (handoffcount>0)
			disconnectRatio=1-(((double)disconnectcount)/handoffcount);
		if (packetsGenerated>0)
			dropRate=((double)packetsDropped)/packetsGenerated;
		if (packetsGenerated>0)
			averageHops=((double)messagesSent)/packetsGenerated;
		if (krn.packetsDelivered>0)
			averageDelay=krn.totalDelay/krn.packetsDelivered;
		System.out.print(seed+",");
		System.out.print(nNodes+",");
		System.out.print(range+",");
		System.out.print((useIndirectHandoff?1:0)+",");
		System.out.print(nSteps+",");
		System.out.print(totalArea+",");
		System.out.print(totalCircumference+",");
		System.out.print(connectionScore+",");
		System.out.print(indirectConnectionScore+",");
		System.out.print(handoffcount+",");
		System.out.print(disconnectcount+",");
		System.out.print(disconnectRatio+",");
		System.out.print(dropRate+",");
		System.out.print(averageHops+",");
		System.out.print(averageDelay+",");
///		System.out.print(badEdgeRatio);
		
	}
	
	void doSillyTriangleComputation()
	{
		Random r=new Random(42);
        Point2D.Double p1,p2,p3,myp,mynextp;
        p1=new Point2D.Double(r.nextDouble(),r.nextDouble());
        p2=new Point2D.Double(r.nextDouble(),r.nextDouble());
        p3=new Point2D.Double(r.nextDouble(),r.nextDouble());
        myp=new Point2D.Double();
        mynextp=new Point2D.Double();
        double count1=0,count2=0,count3=0;
        
        Line2D.Double l1,l2,l3,myl;
        l1=new Line2D.Double(p1,p2);
        l2=new Line2D.Double(p2,p3);
        l3=new Line2D.Double(p3,p1);
        myl=new Line2D.Double();
        for (int i=0;i<1000;i++) // choose random points
        {
        	double r1,r2,r3;
        	r1=r.nextDouble();
        	r2=r.nextDouble();
        	r3=r.nextDouble();
        	double t=(r1+r2+r3);
        	r1/=t;
        	r2/=t;
        	r3/=t; // a random point 
        	myp.x=r1*p1.x+r2*p2.x+r3*p3.x;
        	myp.y=r1*p1.y+r2*p2.y+r3*p3.y;
        	for (int j=0;j<1000;j++)
        	{
//        		double theta=r.nextDouble()*2*Math.PI; // any direction
//        		mynextp.x=myp.x+Math.cos(theta)*100;
//        		mynextp.y=myp.y+Math.sin(theta)*100;
        		mynextp.x=(r.nextDouble()-0.5)*1000;
        		mynextp.y=(r.nextDouble()-0.5)*1000;
        		myl.setLine(myp, mynextp);
        		if (myl.intersectsLine(l1))
        			count1++;
        		else if (myl.intersectsLine(l2))
        			count2++;
        		else 
        			count3++;
        	}
        }
        double a=l1.getP1().distance(l1.getP2());
        double b=l2.getP1().distance(l2.getP2());
        double c=l3.getP1().distance(l3.getP2());
        double tmp=a+b+c;
        a/=tmp;
        b/=tmp;
        c/=tmp;
        System.out.print("a="+a);
        System.out.print(" b="+b);
        System.out.println(" c="+c);
        System.out.print("c(a)="+count1/1000000.0);
        System.out.print(" c(b)="+count2/1000000.0);
        System.out.println(" c(c)="+count3/1000000.0);
	}
	
	boolean getLineSegmentIntersection(Line2D.Double l1, Line2D.Double l2, Point2D.Double p)
	{
		double x1=l1.x1;
		double x2=l1.x2;
		double x3=l2.x1;
		double x4=l2.x2;
		double y1=l1.y1;
		double y2=l1.y2;
		double y3=l2.y1;
		double y4=l2.y2;
		double denom=(y4-y3)*(x2-x1)-(x4-x3)*(y2-y1);
		if (Math.abs(denom)<1e-20)
			return false;
		double ua=((x4-x3)*(y1-y3)-(y4-y3)*(x1-x3))/denom;
		double ub=((x2-x1)*(y1-y3)-(y2-y1)*(x1-x3))/denom;
		if (ua<0 || ua>1 || ub<0 || ub>1)
			return false; // not in line segments
		p.x=x1+ua*(x2-x1);
		p.y=y1+ua*(y2-y1);
		return true;
	}
	
	void boxEdgeCounts()
	{
		Random r=new Random(42);
		// simulates a random walk and calculates the probability of hitting edges
		Line2D.Double [] edges={new Line2D.Double(0,0,width,0),
								new Line2D.Double(width,0,width,height),
								new Line2D.Double(width,height,0,height),
								new Line2D.Double(0,height,0,0)};
		double [][] edgecounts={new double[(int)width+1],
							new double[(int)height+1],
							new double[(int)width+1],
							new double[(int)height+1]};
		Point2D.Double c=new Point2D.Double(width/2,height/2);
		double theta=r.nextDouble()*Math.PI*2;
		Point2D.Double p=new Point2D.Double(c.x-Math.cos(theta)*(width+height),c.y-Math.sin(theta)*(width+height));
		Point2D.Double q=new Point2D.Double(c.x+Math.cos(theta)*(width+height),c.y+Math.sin(theta)*(width+height));
		Line2D.Double l=new Line2D.Double(c,q);
		int lastEdge=0;
		for (int i=0;i<edges.length;i++)
		{
			if (getLineSegmentIntersection(l, edges[i], p))
			{
				lastEdge=i;
				if (lastEdge%2==0)
					edgecounts[lastEdge][(int)p.x]++;
				else
					edgecounts[lastEdge][(int)p.y]++;
				break;
			}
		}
		// p now contains the intersection point
		for (int i=0;i<10000000;i++)
		{
			boolean intersectionFound=false;
			do
			{
				theta=r.nextDouble()*Math.PI*2;
				q.x=p.x+Math.cos(theta)*(width+height);
				q.y=p.y+Math.sin(theta)*(width+height);
				l.setLine(p, q);
				for (int j=0;j<edges.length;j++)
				{
					if (j==lastEdge)
						continue;
					if (getLineSegmentIntersection(l, edges[j], p))
					{
						//System.out.println(p.x+","+p.y);
						intersectionFound=true;
						lastEdge=j;
						if (lastEdge%2==0)
							edgecounts[lastEdge][(int)Math.round(p.x)]++;
						else
							edgecounts[lastEdge][(int)Math.round(p.y)]++;
						break;
					}
				}
			}while(!intersectionFound);
		}
		
		for (int i=0;i<4;i++)
		{
			System.out.println("****************************************");
			System.out.println("****************************************");
			for (int j=0;j<edgecounts[i].length;j++)
			{
				System.out.println(j+","+edgecounts[i][j]);
			}
		}
	}
	
	double getApproximateIntersections(Line2D.Double segment, double [][] edgecounts)
	{
		Line2D.Double [] edgedirs={new Line2D.Double(0,0,width,0),
				new Line2D.Double(width,0,width,height),
				new Line2D.Double(0,height,width,height),
				new Line2D.Double(0,0,0,height)};
		Point2D.Double pc=new Point2D.Double();
		double total=0;
		for (int i=0;i<edgecounts.length;i++)
		{
			for (int j=0;j<edgecounts[i].length;j++)
			{
				pc.x=edgedirs[i].x1+(edgedirs[i].x2-edgedirs[i].x1)/width*j+0.5;
				pc.y=edgedirs[i].y1+(edgedirs[i].y2-edgedirs[i].y1)/height*j+0.5;
				//if (segment.relativeCCW(pc)==1)
				{
	        		double a=segment.getP1().distance(segment.getP2());
	        		double b=segment.getP1().distance(pc);
	        		double c=segment.getP2().distance(pc);
	        		double s=(a+b+c)/2;
	        		double A=Math.sqrt(s*(s-a)*(s-b)*(s-c));
	        		double h=2*A/b;
					double alpha=Math.asin(h/c);
					total+=alpha*edgecounts[i][j];
				}
			}
		}
		return total;
	}
	
	void calculateEdgeWeights()
	{
		// simulates a random walk and calculates the probability of hitting edges
		double [][] approxEdgeCounts={new double[(int)width+1],
				new double[(int)height+1],
				new double[(int)width+1],
				new double[(int)height+1]};
		double totalEdgeWeight=0;
		for (int i=0;i<approxEdgeCounts.length;i++)
		{
			double len=approxEdgeCounts[i].length-1;
			for (int j=0;j<approxEdgeCounts[i].length;j++)
			{
				double d=Math.abs(j-len)/(len/2);
				approxEdgeCounts[i][j]=1.0/Math.sqrt(Math.sqrt(1+1105*d));
			}
		}
		for (GeomNode n:nodes)
		{
			for (DelunayNeighbor dn:n.neighs)
			{
				if (dn.node!=null)
				{
					dn.edgeWeight=getApproximateIntersections(new Line2D.Double(dn.p,dn.q), approxEdgeCounts);
					totalEdgeWeight+=dn.edgeWeight;
				}
			}
		}
		for (GeomNode n:nodes)
		{
			for (DelunayNeighbor dn:n.neighs)
			{
				dn.edgeWeight/=totalEdgeWeight;
			}
		}
	}
	
	void triangleSanityCheck()
	{
		Random r=new Random();
		// simulates a random walk and calculates the probability of hitting edges
		Line2D.Double [] edges={new Line2D.Double(0,0,width,0),
								new Line2D.Double(width,0,width,height),
								new Line2D.Double(width,height,0,height),
								new Line2D.Double(0,height,0,0)};
		double [][] edgecounts={new double[(int)width+1],
				new double[(int)height+1],
				new double[(int)width+1],
				new double[(int)height+1]};
		double [][] approxEdgeCounts={new double[(int)width+1],
				new double[(int)height+1],
				new double[(int)width+1],
				new double[(int)height+1]};
		for (int i=0;i<approxEdgeCounts.length;i++)
		{
			double len=approxEdgeCounts[i].length-1;
			for (int j=0;j<approxEdgeCounts[i].length;j++)
			{
				double d=Math.abs(j-len+0.5)/(len/2);
				approxEdgeCounts[i][j]=1.0/Math.sqrt(Math.sqrt(1+1105*d));
			}
		}
		Vector<Line2D.Double> segments=new Vector<Line2D.Double>();
		for (GeomNode n:nodes)
		{
			for (DelunayNeighbor dn:n.neighs)
			{
				if (dn.node==null || dn.node.id<n.id)
				{
					segments.add(new Line2D.Double(dn.p,dn.q));
				}
			}
		}
		double[] counts=new double[segments.size()];
		double[] approx=new double[segments.size()];
		double[] moreApprox=new double[segments.size()];
		double moreApproxTotal=0;
		double approxTotal=0;

		Point2D.Double c=new Point2D.Double(width/2,height/2);
		double theta=r.nextDouble()*Math.PI*2;
		Point2D.Double p=new Point2D.Double(c.x-Math.cos(theta)*(width+height),c.y-Math.sin(theta)*(width+height));
		Point2D.Double q=new Point2D.Double(c.x+Math.cos(theta)*(width+height),c.y+Math.sin(theta)*(width+height));
		Line2D.Double l=new Line2D.Double(c,q);
		int lastEdge=0;
		double countsTotal=0;
		for (int i=0;i<edges.length;i++)
		{
			if (getLineSegmentIntersection(l, edges[i], p))
			{
				lastEdge=i;
				if (lastEdge%2==0)
					edgecounts[lastEdge][(int)p.x]++;
				else
					edgecounts[lastEdge][(int)p.y]++;
				break;
			}
		}
		// p now contains the intersection point
		for (int i=0;i<500000;i++)
		{
			for (int j=0;j<segments.size();j++)
			{
				if (segments.get(j).intersectsLine(l))
					counts[j]++;
			}
			
			boolean intersectionFound=false;
			do
			{
				theta=r.nextDouble()*Math.PI*2;
				q.x=p.x+Math.cos(theta)*(width+height);
				q.y=p.y+Math.sin(theta)*(width+height);
				l.setLine(p, q);
				for (int j=0;j<edges.length;j++)
				{
					if (j==lastEdge)
						continue;
					if (getLineSegmentIntersection(l, edges[j], p))
					{
						//System.out.println(p.x+","+p.y);
						intersectionFound=true;
						lastEdge=j;
						if (lastEdge%2==0)
							edgecounts[lastEdge][(int)p.x]++;
						else
							edgecounts[lastEdge][(int)p.y]++;
						break;
					}
				}
			}while(!intersectionFound);
			if (i%10000==0)
			{
				System.out.print(".");
				System.out.flush();
			}
		}
		for (int i=0;i<counts.length;i++)
		{
			approx[i]=getApproximateIntersections(segments.get(i), edgecounts);
			moreApprox[i]=getApproximateIntersections(segments.get(i), approxEdgeCounts);
			moreApproxTotal+=moreApprox[i];
			approxTotal+=approx[i];
			countsTotal+=counts[i];
		}
		
		for (int i=0;i<segments.size();i++)
		{
			double d=segments.get(i).getP1().distance(segments.get(i).getP2());
			double dist=(segments.get(i).getP1().distance(width/2.0,height/2.0)+
						segments.get(i).getP2().distance(width/2.0,height/2.0))/2;
			System.out.println(d+" "+dist+" "+counts[i]/countsTotal+" "+approx[i]/approxTotal+" "+moreApprox[i]/moreApproxTotal);
		}
	}
	void computeCosmetics()
	{
		int initColors=7;
		Random r=new Random(42);
        for (GeomNode n:nodes)
        {
        	n.colorIdx=r.nextInt(initColors);
        }
        boolean change;
        do
        {
        	change=false;
	        for (GeomNode n:nodes)
	        {
	        	for (DelunayNeighbor dn:n.neighs)
	        	{
	        		if (dn.node!=null && n.colorIdx==dn.node.colorIdx)
	        		{
	        			n.colorIdx=(n.colorIdx+r.nextInt(initColors-1))%initColors; // definitely different and random
	        			change=true;
	        		}
	        	}
	        }
        }while(change);
        //boxEdgeCounts();
        //triangleSanityCheck();
    //    doSillyTriangleComputation();
        for (GeomNode n:nodes)
        {
        	n.createVoronoiCellShape();
        }
	}
	void stepPacketGeneration()
	{
		// deal with package generation
		for (GeomNode n:nodes)
		{
			if (n.active)
			{// data generation rate should be inserted here properly
				for (int i=0;i<1;i++)
				{
					packetsGenerated++;
					if ((n.queueFront+1)%bufferSize!=n.queueEnd) // there is space [note uses one less than there is ]
					{
						n.packetBuffer[n.queueFront]=time;
						n.queueFront=(n.queueFront+1)%bufferSize;
					}
					else
					{
						// data generation overflow
						packetsDropped++;
//						break;
					}
				}
			}
			n.energyUsed+=n.bwAllocated;
			n.bwAllocated=0;
		}		
	}
	
	void stepPackageForwarding()
	{
		// deal with package forwarding
		for (GeomNode n:nodes)
		{
			GeomNode m=n.next;
			while (m!=null && 
					n.bwAllocated<bandwidth && m.bwAllocated<bandwidth && // there is bandwidth 
					n.queueFront!=n.queueEnd && // there are packets 
					(m.queueFront+1)%bufferSize!=m.queueEnd) // there is space [note uses one less than there is ] 
			{
				double pack=n.packetBuffer[n.queueEnd];
				n.queueEnd=(n.queueEnd+1)%bufferSize; // remove packet from n
				
				m.packetBuffer[m.queueFront]=pack; // put it to m
				m.queueFront=(m.queueFront+1)%bufferSize;

				n.bwAllocated++;
				m.bwAllocated++;
				messagesSent++;
			}
		}
		for (BS bs:bss)
		{
			packetsDelivered+=bs.bwAllocated;
		}
	}

	void step()
	{
		for (BS bs:bss)
			bs.step();
		// deactivate all nodes 
		for (GeomNode n:nodes)
			n.active=false;
		// roi will make them active again
		for (ROI roi:rois)
		{
			roi.step();
		}
		if (useProwler)
		{
			prowlerDriver.runSim(1);
		}
		else
		{
			stepPacketGeneration();
			stepPackageForwarding();
			for (BS bs:bss)
			{
				for (int i=bs.queueEnd;i<bs.queueFront;i++)
				{
					totalDelay+=(time-bs.packetBuffer[i]);
				}
				bs.queueEnd=0; // clear bs buffer
				bs.queueFront=0;
				bs.bwAllocated=0; // clear bs bw usage
			}
		}
		
		time+=1;
	}
	
	void speedTest()
	{
		long vectorstart=System.currentTimeMillis();
		Vector<Integer> vi=new Vector<Integer>();
		for (int i=0;i<1000;i++)
			vi.add(i);
		for (int i=0;i<10000000;i++)
		{
			vi.add(i);
			vi.remove(0);
		}
		long vectorend=System.currentTimeMillis();
		System.out.println("vector time="+(vectorend-vectorstart)/1000.0);
		
		long llstart=System.currentTimeMillis();
		LinkedList<Integer> li=new LinkedList<Integer>();
		for (int i=0;i<1000;i++)
			li.add(i);
		for (int i=0;i<10000000;i++)
		{
			li.addLast(i);
			li.remove(0);
		}
		long llend=System.currentTimeMillis();
		System.out.println("linked list time="+(llend-llstart)/1000.0);		
	}

	void simulate()
	{
		//speedTest();
		for (int i=0;i<nSteps;i++)
		{
			step();
		}
		System.out.println("Packets="+krn.packetsDelivered);
		packetsDelivered=0;
		for (DataMessage d:prowlerDriver.generatedMessages)
		{
			if (d.isDelivered)
				packetsDelivered++;
		}
		deliveryRatio=((double)packetsDelivered)/packetsGenerated;
		dumpParameters();
		printStats();
	}
}