/**
 * @file allocRxBuffer.c
 * @provides allocRxBuffer
 *
 * $Id: allocRxBuffer.c 1577 2008-10-03 18:19:26Z mschul $
 */
/* Embedded Xinu, Copyright (C) 2008.  All rights reserved. */

#include <stddef.h>
#include <stdlib.h>
#include <device.h>
#include <ether.h>
#include <bufpool.h>
#include <mips.h>

/**
 * Allocate an ethernet packet buffer structure
 * @param peth ethernet table entry
 * @param destIndex destination index in ethernet reciever ring
 * @return bytes allocated
 */
int allocRxBuffer(struct ether *peth, int destIndex)
{
    struct ethPktBuffer *ppkt = NULL;
    struct dmaDescriptor *pdesc = NULL;

    destIndex %= peth->rxPending;
    ppkt = bufget(peth->inPool);
    if (SYSERR == (ulong)ppkt)
    {
#ifdef DETAIL
        kprintf("eth0 allocRxBuffer() error\r\n");
#endif                          /* DETAIL */
        return SYSERR;
    }
    ppkt->length = ETH_RX_BUF_SIZE;
    ppkt->buf = (uchar *)(ppkt + 1);

    /* Data region offset by size of rx header. */
    ppkt->data = ppkt->buf + peth->rxOffset;

    peth->rxBufs[destIndex] = ppkt;

    /* Fill in DMA descriptor fields */
    pdesc = peth->rxRing + destIndex;
    pdesc->control = ETH_DESC_CTRL_LEN & ppkt->length;
    if ((peth->rxPending - 1) == destIndex)
    {
        pdesc->control |= ETH_DESC_CTRL_EOT;
    }
    pdesc->address = (ulong)(ppkt->buf) & PMEM_MASK;

#ifdef DETAIL
    kprintf("eth0 rxBufs[%d] = 0x%08X\r\n",
            destIndex, peth->rxBufs[destIndex]);

    kprintf
        ("  control 0x%08X address 0x%08X buffer  0x%08X\r\n",
         pdesc->control, pdesc->address, ppkt);

    if (ppkt)
    {
        kprintf
            ("  epb buf 0x%08X epb dat 0x%08X epb len   %8d\r\n",
             ppkt->buf, ppkt->data, ppkt->length);
        struct rxHeader *rh;

        rh = (struct rxHeader *)ppkt->buf;
        kprintf
            ("  rxHead  0x%08X length  0x%08X flags   0x%08X\r\n",
             rh, rh ? rh->length : 0, rh ? rh->flags : 0);
    }
#endif                          /* DETAIL */

    return OK;
}
