/**
 * @file etherControl.c
 * @provides etherControl
 *
 * $Id: etherControl.c 1577 2008-10-03 18:19:26Z mschul $
 */
/* Embedded Xinu, Copyright (C) 2008.  All rights reserved. */

#include <stddef.h>
#include <device.h>
#include <ether.h>
#include <bcm4713.h>
#include <backplane.h>

/**
 * Control function for ethernet devices.
 * @param pdev ethernet device table entry
 * @param func control function to execute
 * @param arg1 first argument for the control function
 * @param arg2 second argument for the control function
 * @return the result of the control function
 */
devcall etherControl(device * pdev, int func, long arg1, long arg2)
{
    struct ether *peth;
    struct bcm4713 *pecsr;
    volatile uint *preg;
    uchar *pchar;
    ulong temp = 0;

    if (NULL == pdev || NULL == pdev->controlblk)
    {
        return SYSERR;
    }
    peth = (struct ether *)pdev->controlblk;
    if (NULL == peth->csr)
    {
        return SYSERR;
    }
    pecsr = peth->csr;

    switch (func)
    {
/* Clear stats counters on ethernet device. */
    case ETH_CTRL_CLEAR_STATS:
        pecsr->mibControl = MIB_CTRL_CLR_ON_READ;

        preg = &pecsr->txGoodOctets;
        while (preg <= &pecsr->txPause)
        {
            temp = *preg++;
        }

        preg = &pecsr->rxGoodOctets;
        while (preg <= &pecsr->rxNonPause)
        {
            temp = *preg++;
        }

        temp = pecsr->mibControl & ~MIB_CTRL_CLR_ON_READ;
        pecsr->mibControl = temp;
        break;

/* Program MAC address into card. */
    case ETH_CTRL_SET_MAC:
        pchar = (uchar *)arg1;
        /* b44 cam write */
        pecsr->camControl = 0;
        temp = ((uint)pchar[2]) << 24;
        temp |= ((uint)pchar[3]) << 16;
        temp |= ((uint)pchar[4]) << 8;
        temp |= ((uint)pchar[5]) << 0;
        pecsr->camDataLo = temp;

        temp = (CAM_DATA_HI_VALID
                | (((uint)pchar[0]) << 8) | (((uint)pchar[1]) << 0));
        pecsr->camDataHi = temp;
        pecsr->camControl =
            (pdev->minor << CAM_CTRL_INDEX_SHIFT) | CAM_CTRL_WRITE;

        waitOnBit(&pecsr->camControl, CAM_CTRL_BUSY, 0, 100);

        /* Re-enable CAM control */
        pecsr->camControl |= CAM_CTRL_ENABLE;
        break;

/* Get MAC address from card. */
    case ETH_CTRL_GET_MAC:
        pchar = (uchar *)arg1;
        pecsr->camControl =
            (pdev->minor << CAM_CTRL_INDEX_SHIFT) | CAM_CTRL_READ;
        waitOnBit(&pecsr->camControl, CAM_CTRL_BUSY, 0, 100);

        temp = pecsr->camDataLo;
        pchar[2] = temp >> 24;
        pchar[3] = temp >> 16;
        pchar[4] = temp >> 8;
        pchar[5] = temp >> 0;

        temp = pecsr->camDataHi;
        pchar[0] = temp >> 8;
        pchar[1] = temp >> 0;

        /* Re-enable CAM control */
        pecsr->camControl = CAM_CTRL_ENABLE;
        break;

/* Set receiver mode. */
    case ETH_CTRL_SET_RX_MODE:
        pecsr->rxConfig = (uint)arg1;
        break;

/* Reset Ethernet device. */
    case ETH_CTRL_RESET:
        /* Idle the network card if the backplane core is up */
        if (backplaneCoreUp(&(pecsr->bpConfig)))
        {
            etherControl(pdev, ETH_CTRL_DISABLE, 0, 0);
        }

        /* Reset the target core. */
        backplaneReset(&(pecsr->bpConfig));

        etherControl(pdev, ETH_CTRL_CLEAR_STATS, 0, 0);

        /* Make PHY accessible and set frequency */
        pecsr->mdioControl = (MDIO_CTRL_PREAMBLE | MDIO_CTRL_FREQ);

        /* Select external PHY, this model does not have internal. */
        pecsr->enetControl = ENET_CTRL_EPSEL;
        break;

/* Disable ethernet core */
    case ETH_CTRL_DISABLE:
        pecsr->rcvLazy = 0;     /* Turn off lazy reception. */
        pecsr->enetControl;     /* Read control register.   */

        /* Wait for controller */
        waitOnBit(&pecsr->enetControl, ENET_CTRL_DISABLE, 0, 100);
        /* Reset transmitter settings and transmitter queue. */
        pecsr->dmaTxControl = 0;
        peth->txHead = 0;
        peth->txTail = 0;
        if (pecsr->dmaRxStatus & DMARX_STAT_EMASK)
        {
            /* Wait for receiver to idle. */
            waitOnBit(&pecsr->dmaRxStatus, DMARX_STAT_SIDLE, 1, 100);
        }

        /* Reset receiver settings and receiver queue. */
        pecsr->dmaRxControl = 0;
        peth->rxHead = 0;
        peth->rxTail = 0;
        break;

    default:
        return SYSERR;
    }

    return OK;
}
