/**
 * @file etherInit.c
 * @provides etherInit
 *
 * $Id: etherInit.c 1577 2008-10-03 18:19:26Z mschul $
 */
/* Embedded Xinu, Copyright (C) 2008.  All rights reserved. */

#include <kernel.h>
#include <stdio.h>
#include <stdlib.h>
#include <mips.h>
#include <ether.h>
#include <memory.h>
#include <backplane.h>
#include <bcm4713.h>
#include <bcmswitch.h>
#include <nvram.h>

/* Global table of ethernet devices */
struct ether ethertab[NETHER];

/**
 * Initialize ethernet device structures.
 * @param pdev ETH device table entry
 * @return OK if device is intialized successfully, otherwise SYSERR
 */
devcall etherInit(device * pdev)
{
    struct ether *peth;

    /* Initialize structure pointers */
    peth = &ethertab[pdev->minor];
    bzero(peth, sizeof(struct ether));
    pdev->controlblk = peth;
    peth->dev = pdev;
    peth->csr = (struct bcm4713 *)pdev->csr;
    peth->phy = pdev;

    peth->state = ETH_STATE_DOWN;
    peth->rxPending = ETH_RX_RING_ENTRIES;
    peth->txPending = ETH_TX_RING_ENTRIES;
    peth->mtu = ETH_MTU;
    peth->interruptMask = IMASK_DEF;

    peth->errors = 0;
    peth->isema = semcreate(0);
    peth->istart = 0;
    peth->icount = 0;
    peth->ovrrun = 0;
    peth->rxOffset = sizeof(struct rxHeader);

    /* Lookup canonical MAC in NVRAM, and store in ether struct */
    colon2mac(nvramGet("et0macaddr"), peth->devAddress);

    /* NOTE: because device initialization runs early in the system, */
    /*  we are assured that this stkget() call will return          */
    /*  page-aligned (and cache-aligned) boundaries.                */
    peth->rxBufs = stkget(PAGE_SIZE);
    peth->txBufs = stkget(PAGE_SIZE);
    peth->rxRing = stkget(PAGE_SIZE);
    peth->txRing = stkget(PAGE_SIZE);

    if ((SYSERR == (int)peth->rxBufs)
        || (SYSERR == (int)peth->txBufs)
        || (SYSERR == (int)peth->rxRing) || (SYSERR == (int)peth->txRing))
    {
#ifdef DETAIL
        kprintf("eth%d ring buffer allocation error.\r\n", pdev->minor);
#endif                          /* DETAIL */
        return SYSERR;
    }

    /* bump buffers/rings to KSEG1 */
    peth->rxBufs =
        (struct ethPktBuffer
         **)(((ulong)peth->rxBufs - PAGE_SIZE +
              sizeof(int)) | KSEG1_BASE);
    peth->txBufs =
        (struct ethPktBuffer
         **)(((ulong)peth->txBufs - PAGE_SIZE +
              sizeof(int)) | KSEG1_BASE);
    peth->rxRing =
        (struct dmaDescriptor
         *)(((ulong)peth->rxRing - PAGE_SIZE + sizeof(int)) | KSEG1_BASE);
    peth->txRing =
        (struct dmaDescriptor
         *)(((ulong)peth->txRing - PAGE_SIZE + sizeof(int)) | KSEG1_BASE);

    /* Make sure nothing erroneous is in buffers/rings */
    bzero(peth->rxBufs, PAGE_SIZE);
    bzero(peth->txBufs, PAGE_SIZE);
    bzero(peth->rxRing, PAGE_SIZE);
    bzero(peth->txRing, PAGE_SIZE);

    /* Reset ethernet device to known state. */
    etherControl(pdev, ETH_CTRL_RESET, 0, 0);

    /* Initialize the network switch */
    switchInit(peth->csr);

    interruptVector[pdev->irq] = pdev->intr;
    enable_irq(pdev->irq);

    return OK;
}
