/**
 * @file etherInterrupt.c
 * @provides etherInterrupt
 *
 * $Id: etherInterrupt.c 1577 2008-10-03 18:19:26Z mschul $
 */
/* Embedded Xinu, Copyright (C) 2008.  All rights reserved. */

#include <thread.h>
#include <stdlib.h>
#include <mips.h>
#include <device.h>
#include <ether.h>
#include <vlan.h>
#include <bcm4713.h>
#include <platform.h>
#include <string.h>
#include <bufpool.h>
#include <network.h>

extern int resdefer;

void rxPackets(struct ether *peth, struct bcm4713 *pecsr)
{
    ulong head = 0, tail = 0;
    struct ethPktBuffer *ppkt = NULL;
    struct rxHeader *rh = NULL;
    struct vlanPkt *pvlan = NULL;
    struct ether *pveth = 0;

    /* 12-bit descriptor indicates where card is currently placing   */
    /*  received packets into the ring.                              */
    tail = (pecsr->dmaRxStatus & DMARX_STAT_CDMASK)
        / sizeof(struct dmaDescriptor);
    /* rxHead indicates where we last left off pulling received      */
    /*  packets off of the ring.                                     */
    head = peth->rxHead;
    while (head != tail)
    {
        ppkt = peth->rxBufs[head];
        rh = (struct rxHeader *)ppkt->buf;
        pvlan = (struct vlanPkt *)ppkt->data;

        if (ETH_TYPE_VLAN == net2hs(pvlan->tpi))
        {
            pveth = &ethertab[net2hs(pvlan->vlanId) & ETH_VLAN_IDMASK];
        }
        else
        {
            pveth = peth;
        }

        if ((rh->length >
             ETH_MAX_PKT_LEN + ETH_CRC_LEN)
            || (rh->flags & ETH_RX_FLAG_ERRORS))
        {
            pveth->rxErrors++;
            bzero(ppkt->buf, ppkt->length);
        }
        else
        {
            if (pveth->icount < ETH_IBLEN)
            {
                allocRxBuffer(peth, head);
                pveth->in[(pveth->istart + pveth->icount) % ETH_IBLEN] =
                    ppkt;
                pveth->icount++;
                signaln(pveth->isema, 1);
            }
            else
            {
                pveth->ovrrun++;
                bzero(ppkt->buf, ppkt->length);
            }
        }
        peth->rxTail = (peth->rxTail + 1) % peth->rxPending;
        pecsr->dmaRxLast = peth->rxTail * sizeof(struct dmaDescriptor);
        head = (head + 1) % peth->rxPending;
    }
    peth->rxHead = head;
}

/**
 * Describe function.
 */
void txPackets(struct ether *peth, struct bcm4713 *pecsr)
{
    struct ethPktBuffer **epb = NULL;
    struct ethPktBuffer *ppkt = NULL;
    ulong current = 0, head = 0;

    current = (pecsr->dmaTxStatus & DMATX_STAT_CDMASK)
        / sizeof(struct dmaDescriptor);

    for (head = peth->txHead; head != current;
         head = (head + 1) % peth->txPending)
    {
        epb = &peth->txBufs[head];
        ppkt = *epb;
        if (NULL == ppkt)
        {
            continue;
        }
        buffree((void *)((ulong)ppkt & (PMEM_MASK | KSEG0_BASE)));
        *epb = NULL;
    }
    peth->txHead = head;
    pecsr->gpTimer = 0;
}

/**
 * Decode and handle hardware interrupt request from ethernet device.
 */
interrupt etherInterrupt(void)
{
    int ethnum;
    struct ether *peth;
    struct bcm4713 *pecsr;
    uint status, mask;

    /* Initialize structure pointers */
    peth = &ethertab[0];        /* default physical ethernet for WRT54GL */
    if (!peth)
    {
        return;
    }
    pecsr = peth->csr;
    if (!pecsr)
    {
        return;
    }

    mask = pecsr->interruptMask;
    status = pecsr->interruptStatus & mask;

    /* remember interrupt status in ether struct */
    peth->interruptStatus = status;

    if (!status)
    {
        return;
    }

    resdefer = 1;               /* defer rescheduling */

    if (status & ISTAT_TX)
    {
        peth->txirq++;
        txPackets(peth, pecsr);
        /* Set Rx timeout to 2 seconds after last Tx */
        pecsr->gpTimer = 2 * platform.clkfreq;
    }

    if (status & ISTAT_RX)
    {
        peth->rxirq++;
        rxPackets(peth, pecsr);
        /* Set Rx timeout to 0 */
        pecsr->gpTimer = 0;
    }

    if (status & ISTAT_ERRORS)
    {
        /* Error closes physical NIC as well as vlans */
        for (ethnum = 0; ethnum < NETHER; ethnum++)
        {
            peth = &ethertab[ethnum];
            if (!peth)
            {
                continue;
            }
            peth->errors++;
            etherClose(peth->dev);
        }
    }

    /* signal the card with the interrupts we handled */
    pecsr->interruptStatus = status;

    if (--resdefer > 0)
    {
        resdefer = 0;
        resched();
    }

    return;
}
