/**
 * @file etherOpen.c
 * @provides etherOpen
 *
 * $Id: etherOpen.c 1577 2008-10-03 18:19:26Z mschul $
 */
/* Embedded Xinu, Copyright (C) 2008.  All rights reserved. */

#include <stddef.h>
#include <device.h>
#include <ether.h>
#include <memory.h>
#include <bufpool.h>
#include <stdlib.h>
#include <interrupt.h>
#include <mips.h>

/**
 * Open an ethernet device for use.
 * @param  pdev ethernet device table entry
 * @return OK if opened properly, otherwise SYSERR
 */
devcall etherOpen(device * pdev)
{
    int i;
    irqmask im;
    struct ether *peth;
    struct bcm4713 *pecsr;

    /* Initialize structure pointers */
    peth = &ethertab[pdev->minor];
    if (NULL == peth)
    {
        return SYSERR;
    }
    pecsr = peth->csr;
    if (NULL == pecsr)
    {
        return SYSERR;
    }

    /* Request memory buffer pool for Rx DMA and intermediate buffers of
       all ethernet devices--both real and virtual */
    peth->inPool =
        bfpalloc(ETH_RX_BUF_SIZE + sizeof(struct ethPktBuffer),
                 ETH_RX_RING_ENTRIES + ETH_IBLEN * NETHER);
    if (SYSERR == peth->inPool)
    {
#ifdef DETAIL
        kprintf("eth%d inPool buffer error.\r\n", pdev->minor);
#endif                          /* DETAIL */
        return SYSERR;
    }

    /* Request memory buffer pool for Tx DMA */
    peth->outPool =
        bfpalloc(ETH_TX_BUF_SIZE + sizeof(struct ethPktBuffer),
                 ETH_TX_RING_ENTRIES);
    if (SYSERR == peth->outPool)
    {
#ifdef DETAIL
        kprintf("eth%d outPool buffer error.\r\n", pdev->minor);
#endif                          /* DETAIL */
        return SYSERR;
    }

    im = disable();

    /* Initialize DMA descriptor processor */
    pecsr->dmaTxControl = DMATX_CTRL_ENABLE;
    pecsr->dmaTxAddress = (ulong)peth->txRing & PMEM_MASK;

    pecsr->dmaRxControl = DMARX_CTRL_ENABLE |
        (peth->rxOffset << DMARX_CTRL_ROSHIFT);
    pecsr->dmaRxAddress = (ulong)peth->rxRing & PMEM_MASK;

    for (i = 0; i < peth->rxPending; i++)
    {
        allocRxBuffer(peth, i);
    }

    pecsr->dmaRxLast = peth->rxPending * sizeof(struct dmaDescriptor);
    peth->rxTail = peth->rxPending;     // Producer goes to tail.

    /* Configure NIC */
    pecsr->macControl |= MAC_CTRL_CRC32_ENAB;   /* enable crc32 */
    pecsr->rcvLazy = (1 << RCV_LAZY_FC_SHIFT);  /* enable lazy Rx intr */

    etherControl(pdev, ETH_CTRL_SET_MAC, (long)peth->devAddress, 0);
    etherControl(pdev, ETH_CTRL_SET_RX_MODE, RXCONFIG_NONE, 0);

    /* Set maximum incoming and outgoing packet length */
    pecsr->rxMaxLength = ETH_HEADER_LEN + ETH_VLAN_LEN
        + peth->mtu + ETH_CRC_LEN;
    pecsr->txMaxLength = ETH_HEADER_LEN + ETH_VLAN_LEN
        + peth->mtu + ETH_CRC_LEN;

    pecsr->txWatermark = TX_WATERMARK;

    pecsr->enetControl |= ENET_CTRL_ENABLE;

    peth->state = ETH_STATE_UP;
    pecsr->interruptMask = peth->interruptMask;

    restore(im);

    return OK;
}
