/**
 * @file etherRead.c
 * @provides etherRead
 *
 * $Id: etherRead.c 1577 2008-10-03 18:19:26Z mschul $
 */
/* Embedded Xinu, Copyright (C) 2008.  All rights reserved. */

#include <stddef.h>
#include <device.h>
#include <ether.h>
#include <string.h>
#include <interrupt.h>
#include <bufpool.h>
#include <stdlib.h>
#include <network.h>
#include <vlan.h>

/**
 * Read a packet from the ethernet device.
 * @param pdev pointer to ethernet device
 * @param buf buffer for read
 * @param len size of the buffer
 * @return number of bytes read
 */
devcall etherRead(device * pdev, void *buf, uint len)
{
    irqmask im;
    struct ether *peth;
    struct ethPktBuffer *ppkt;
    struct rxHeader *rh;
    struct vlanPkt *pvlan;
    uint length;
    char *buffer = buf;

    /* Setup and error check pointers to structures */
    if (NULL == pdev)
    {
        return SYSERR;
    }
    peth = (struct ether *)pdev->controlblk;
    if (NULL == peth)
    {
        return SYSERR;
    }
    if (ETH_STATE_UP != peth->state)
    {
        return SYSERR;
    }

    /* make sure buffer is large enough to store packet */
    if (len < ETH_HEADER_LEN)
    {
        return SYSERR;
    }

    wait(peth->isema);

    im = disable();
    ppkt = peth->in[peth->istart];
    peth->in[peth->istart] = NULL;
    peth->istart = (peth->istart + 1) % ETH_IBLEN;
    peth->icount--;
    restore(im);

    if (NULL == ppkt)
    {
        return 0;
    }
    rh = (struct rxHeader *)ppkt->buf;
    pvlan = (struct vlanPkt *)ppkt->data;

    /* slice off CRC */
    rh->length -= ETH_CRC_LEN;

    if (ETH_TYPE_VLAN == net2hs(pvlan->tpi))
    {
        rh->length -= 4;        /* Account for vlan tag removal */
    }

    /* compute size of packet */
    length = (rh->length < len) ? rh->length : len;

    if (ETH_TYPE_VLAN == net2hs(pvlan->tpi))
    {
        /* strip vlan tag */
        memcpy(buffer, ppkt->data, 12);
        memcpy(buffer + 12, ppkt->data + 16, length - 12);
    }
    else
    {
        /* no vlan tagging */
        memcpy(buffer, ppkt->data, length);
    }

    buffree(ppkt);

    return length;
}
