/**
 * @file etherWrite.c
 * @provides etherWrite
 *
 * $Id: etherWrite.c 1577 2008-10-03 18:19:26Z mschul $
 */
/* Embedded Xinu, Copyright (C) 2008.  All rights reserved. */

#include <stddef.h>
#include <stdlib.h>
#include <device.h>
#include <bufpool.h>
#include <ether.h>
#include <vlan.h>
#include <bcm4713.h>
#include <interrupt.h>
#include <string.h>
#include <mips.h>
#include <network.h>

/**
 * Write a packet to the ethernet device
 * @param pdev device table entry
 * @param buf packet buffer
 * @param len size of the buffer
 * @return bytes output
 */
devcall etherWrite(device * pdev, void *buf, uint len)
{
    struct ether *peth;
    struct bcm4713 *pecsr;
    struct ethPktBuffer *ppkt = NULL;
    struct vlanPkt *pvlan;
    struct ether *pphy;
    irqmask im;
    ulong entry = 0, control = 0;
    uchar *buffer = buf;

    /* Setup and error check pointers to structures */
    if (NULL == pdev)
    {
        return SYSERR;
    }
    peth = (struct ether *)pdev->controlblk;
    if (NULL == peth)
    {
        return SYSERR;
    }
    if (ETH_STATE_UP != peth->state)
    {
        return SYSERR;
    }
    pecsr = peth->csr;
    if (NULL == pecsr)
    {
        return SYSERR;
    }
    pphy = (struct ether *)peth->phy->controlblk;
    if (ETH_STATE_UP != pphy->state)
    {
        return SYSERR;
    }

    /* make sure packet is not too small */
    if (len < ETH_HEADER_LEN)
    {
        return SYSERR;
    }

    /* make sure packet is not too big */
    if (len > (ETH_TX_BUF_SIZE - ETH_VLAN_LEN))
    {
        return SYSERR;
    }

    ppkt = (struct ethPktBuffer *)bufget(pphy->outPool);
    if (SYSERR == (ulong)ppkt)
    {
        return SYSERR;
    }
    ppkt = (struct ethPktBuffer *)((int)ppkt | KSEG1_BASE);
    ppkt->buf = (uchar *)(ppkt + 1);
    ppkt->data = ppkt->buf;
    pvlan = (struct vlanPkt *)ppkt->data;

    /* Copy packet to DMA buffer with added vlan tag */
    memcpy(ppkt->data, buffer, 12);
    pvlan->tpi = hs2net(ETH_TYPE_VLAN);
    pvlan->vlanId = hs2net(pdev->minor);
    memcpy(ppkt->data + 16, buffer + 12, len - 12);
    len += ETH_VLAN_LEN;        /* account for vlan tag addition */
    ppkt->length = len;

    /* Place filled buffer in outgoing queue */
    im = disable();
    entry = pphy->txTail;
    pphy->txBufs[entry] = ppkt;

    control = len & ETH_DESC_CTRL_LEN;
    /* Mark as start and end of frame, interrupt on completion. */
    control |= ETH_DESC_CTRL_IOC | ETH_DESC_CTRL_SOF | ETH_DESC_CTRL_EOF;
    if (pphy->txPending - 1 == entry)
    {
        control |= ETH_DESC_CTRL_EOT;
    }

    /* Add this buffer to the Tx ring. */
    pphy->txRing[entry].control = control;
    pphy->txRing[entry].address = (ulong)ppkt->data & PMEM_MASK;

    pphy->txTail = (entry + 1) % pphy->txPending;
    entry = pphy->txTail;

    pecsr->dmaTxLast = entry * sizeof(struct dmaDescriptor);

    restore(im);

    return len;
}
