/**
 * @file     nvram.c
 * @provides nvramInit, nvramGet
 * Functions to access the nvram settings from kernel space.
 *
 * $Id: nvram.c 1577 2008-10-03 18:19:26Z mschul $
 */
/* Embedded Xinu, Copyright (C) 2008.  All rights reserved. */

#include <kernel.h>
#include <memory.h>
#include <nvram.h>
#include <string.h>
#include <stdlib.h>

struct nvram_header *nvram_header = NULL;
struct nvram_tuple *nvram_tuples[NVRAM_NHASH];

static uint nvramHash(char *name);
static void nvramInsert(struct nvram_tuple *tuple);

/**
 * Initialize the nvram variable structures for editing
 * @return OK on success, SYSERR on failure
 */
devcall nvramInit()
{
    uint index, offset;
    uint pair_len, size;
    char *pair;

    struct nvram_tuple *tuple;

    /* check if we already have initialized nvram */
    if (nvram_header != NULL && NVRAM_MAGIC == nvram_header->magic)
    {
        return OK;
    }

    /* zero out nvram_tuples pointers */
    for (index = 0; index < NVRAM_NHASH; index++)
    {
        nvram_tuples[index] = NULL;
    }

    /* Hardcoded location of nvram (see nvram.h for more information) */
    nvram_header = (struct nvram_header *)NVRAM_LOCATION;

    /* initial offset of data at that location */
    offset = 0;

    if (NVRAM_MAGIC != nvram_header->magic)
    {
        return SYSERR;
    }

    /* march offset to next chunk of data */
    offset += sizeof(struct nvram_header);

    /* loop through nvram variables and add to array */
    while (offset < nvram_header->length)
    {
        /* get the length of a string (name=value\0) */
        pair = (char *)(NVRAM_LOCATION + offset);
        pair_len = strnlen(pair, nvram_header->length - offset);

        /* set offset to next string */
        offset += pair_len + 1;

        /* null pairs occur as padding at the end */
        if (pair_len <= 0)
        {
            continue;
        }

        /* allocate memory to store tuple */
        size = sizeof(struct nvram_tuple) + pair_len;
        tuple = memget(size);

        /* store tuple (+1 for the null terminator) */
        memcpy(tuple->pair, pair, pair_len + 1);
        nvramInsert(tuple);
    }

    return OK;
}

/**
 * Insert a tuple into the nvram_tuples table
 * @param *tuple pointer to tuple to add (should be memory address)
 */
static void nvramInsert(struct nvram_tuple *tuple)
{
    uint index;
    struct nvram_tuple *curr;

    /* hash into table */
    index = nvramHash(tuple->pair);
    curr = nvram_tuples[index];

    /* search for an open position */
    while (curr != NULL && curr->next != NULL)
    {
        curr = curr->next;
    }

    /* fill the open position */
    if (NULL == curr)
    {
        curr = tuple;
        curr->next = NULL;
        nvram_tuples[index] = curr;
    }
    else
    {
        curr->next = tuple;
        tuple->next = NULL;
    }
}

/**
 * Find the value of a variable.
 * @param *name name of variable to find
 * @return pointer to beginning of value
 */
char *nvramGet(char *name)
{
    struct nvram_tuple *tuple;
    uint hash;

    if (OK != nvramInit())
    {
        return NULL;
    }

    /* hash the name */
    hash = nvramHash(name);

    /* iterate until name == name */
    tuple = nvram_tuples[hash];
    while ((NULL != tuple) &&
           (strncmp(name, tuple->pair, strnlen(name, NVRAM_STRMAX)) != 0))
    {
        tuple = tuple->next;
    }

    /* make sure a valid string was found, if not return NULL */
    if (NULL == tuple)
    {
        return NULL;
    }

    /* return pointer to beginning of value in name=value pair */
    return tuple->pair + strnlen(name, NVRAM_STRMAX) + 1;
}

/**
 * Given an input string, this will calculate a simple hash between 0 and
 * NVRAM_NHASH-1 (inclusive).
 * @param *p null or '=' terminated string
 * @return value of hashed string
 */
static uint nvramHash(char *p)
{
    uint sum = 0;

    while ((*p) && (*p != '='))
    {
        sum = 13 * sum + *p++;
    }
    return sum % NVRAM_NHASH;
}
