/**
 * @file uartInterrupt.c
 * @provides uartInterrupt
 *
 * $Id: uartInterrupt.c 1577 2008-10-03 18:19:26Z mschul $
 */
/* Embedded Xinu, Copyright (C) 2008.  All rights reserved. */

#include <thread.h>
#include <device.h>
#include <uart.h>

extern int resdefer;

/**
 * Decode hardware interrupt request from UART device.
 */
interrupt uartInterrupt(void)
{
    int u = 0, iir = 0, lsr = 0, count = 0;
    char c;
    struct uart *puart = NULL;
    struct uart_csreg *pucsr = NULL;

    resdefer = 1;               /* deferral rescheduling. */

    for (u = 0; u < NUART; u++)
    {
        puart = &uarttab[u];
        if (NULL == puart)
        {
            continue;
        }
        pucsr = puart->csr;
        if (NULL == pucsr)
        {
            continue;
        }

        /* Check interrupt identification register */
        iir = pucsr->iir;
        if (iir & UART_IIR_IRQ)
        {
            continue;
        }

        /*
         * Decode interrupt cause based upon the value taken from the
         * UART interrupt identification register.  Clear interrupt source,
         * and perform the appropriate handling to coordinate properly
         * with the upper half of the driver.
         */

        /* Decode interrupt cause */
        iir &= UART_IIR_IDMASK;
        switch (iir)
        {
            /* Receiver line status interrupt */
        case UART_IIR_RLSI:
            lsr = pucsr->lsr;
            puart->lserr++;
            break;

            /* Receiver data available or timed out */
        case UART_IIR_RDA:
        case UART_IIR_RTO:
            puart->iirq++;
            count = 0;
            while (pucsr->lsr & UART_LSR_DR)
            {
                c = pucsr->buffer;
                if (puart->icount < UART_IBLEN)
                {
                    puart->in
                        [(puart->istart + puart->icount) % UART_IBLEN] =
                        c;
                    puart->icount++;
                    count++;
                }
                else
                {
                    puart->ovrrn++;
                }
            }
            puart->cin += count;
            signaln(puart->isema, count);
            break;

            /* Transmitter holding register empty */
        case UART_IIR_THRE:
            puart->oirq++;
            lsr = pucsr->lsr;   /* Read from LSR to clear interrupt */
            count = 0;
            if (puart->ocount > 0)
            {
                /* Write characters to the lower half of the UART. */
                do
                {
                    count++;
                    puart->ocount--;
                    pucsr->buffer = puart->out[puart->ostart];
                    puart->ostart = (puart->ostart + 1) % UART_OBLEN;
                }
                while ((count < UART_FIFO_LEN) && (puart->ocount > 0));
            }

            if (count)
            {
                puart->cout += count;
                signaln(puart->osema, count);
            }
            /* If no characters were written, set the output idle flag. */
            else
            {
                puart->oidle = TRUE;
            }
            break;

            /* Modem status change */
        case UART_IIR_MSC:
            break;
        }
    }

    if (--resdefer > 0)
    {
        resdefer = 0;
        resched();
    }
}
