/**
 * @file uartRead.c
 * @provides uartRead
 *
 * $Id: uartRead.c 1577 2008-10-03 18:19:26Z mschul $
 */
/* Embedded Xinu, Copyright (C) 2008.  All rights reserved. */

#include <stddef.h>
#include <uart.h>
#include <device.h>
#include <interrupt.h>

/**
 * Read into a buffer from the UART.
 * @param pdev UART device table entry
 * @param buf buffer to read bytes into
 * @param len size of the buffer
 * @return count of bytes read
 */
devcall uartRead(device * pdev, void *buf, uint len)
{
    irqmask im;
    int count = 0;
    char c;
    struct uart *puart;
    uchar *buffer = buf;

    /* Setup and error check pointers to structures */
    if (NULL == pdev)
    {
        return SYSERR;
    }

    puart = (struct uart *)pdev->controlblk;
    if (NULL == puart)
    {
        return SYSERR;
    }

    im = disable();

    /* If in non-blocking mode, ensure there is */
    /* enough input for the entire read request */
    if ((puart->iflags & UART_IFLAG_NOBLOCK)
        && (semcount(puart->isema) < len))
    {
        restore(im);
        return SYSERR;
    }

    /* Put each character into the buffer from the input buffer */
    while (count < len)
    {
        /* If in non-blocking mode, ensure there is another byte of input */
        if ((puart->iflags & UART_IFLAG_NOBLOCK)
            && (semcount(puart->isema) < 1))
        {
            break;
        }

        /* Wait for input and read character from the  */
        /* input buffer; Preserve the circular buffer  */
        wait(puart->isema);
        c = puart->in[puart->istart];
        *buffer++ = c;
        puart->icount--;
        puart->istart = (puart->istart + 1) % UART_IBLEN;
        count++;

        /* If echo is enabled, echo the character */
        if (puart->iflags & UART_IFLAG_ECHO)
        {
            uartPutc(puart->dev, c);
        }
    }

    restore(im);
    return count;
}
