/**
 * @file uartWrite.c
 * @provides uartWrite
 *
 * $Id: uartWrite.c 1577 2008-10-03 18:19:26Z mschul $
 */
/* Embedded Xinu, Copyright (C) 2008.  All rights reserved. */

#include <stddef.h>
#include <uart.h>
#include <device.h>
#include <interrupt.h>

/**
 * Write a buffer to the UART.
 *
 * @param pdev  pointer to UART device
 * @param buf   buffer of characters to write
 * @param len   number of characters to write from the buffer
 */
devcall uartWrite(device * pdev, void *buf, uint len)
{
    irqmask im;
    int count = 0;
    struct uart *puart;
    volatile struct uart_csreg *pucsr;
    uchar *buffer = buf;

    /* Setup and error check pointers to structures */
    if (NULL == pdev)
    {
        return SYSERR;
    }

    puart = (struct uart *)pdev->controlblk;
    if (NULL == puart)
    {
        return SYSERR;
    }

    pucsr = puart->csr;
    if (NULL == pucsr)
    {
        return SYSERR;
    }

    im = disable();

    /* If in non-blocking mode, ensure there is enough space for the entire
     * write request */
    if ((puart->oflags & UART_OFLAG_NOBLOCK)
        && (semcount(puart->osema) < len))
    {
        restore(im);
        return SYSERR;
    }

    /* Put each character from the buffer into the output buffer for the
     * lower half */
    while (count < len)
    {
        /* If in non-blocking mode, ensure there is another byte of space
         * for output */
        if ((puart->oflags & UART_OFLAG_NOBLOCK)
            && (semcount(puart->osema) < 1))
        {
            break;
        }

        /* If the UART transmitter hardware is idle, write directly to the
         * hardware */
        if (puart->oidle)
        {
            puart->oidle = FALSE;
            pucsr->thr = *buffer++;
            count++;
            puart->cout++;
        }
        /* Wait for space and place character in the output buffer for the
         * lower half; Preserve the circular buffer */
        else
        {
            wait(puart->osema);
            puart->out[(puart->ostart +
                        puart->ocount) % UART_OBLEN] = *buffer++;
            count++;
            puart->ocount++;
        }
    }

    restore(im);
    return count;
}
