/**
 * @file start.S
 * @provides _start, copyhandler, clearvector, getmaxaddr
 * This is where the common firmware environment begins to execute code.
 *
 * $Id: start.S 1577 2008-10-03 18:19:26Z mschul $
 */
/* Embedded Xinu, Copyright (C) 2008.  All rights reserved. */

#include <mips.h>
#include <interrupt.h>

.text
	.align	4
	.globl _start
	
/**
 * The Common Firmware Environment will load the image into temporary
 * memory via TFTP and will then jump to this (_start) label at address
 * 0x80001000.  The boot loader (_start) copies the kernel's interrupt 
 * handler to the proper location and clears the space for the 
 * Xinu-defined trap and interrupt vecotrs.  It also flushes the L1 
 * instruction and data caches.
 *
 * Note that CFE passes four parameters that Xinu never uses.
 */
	.ent _start
_start:
	/* Copy IRQ transfer handler to reserved memory location      */
	la    a0, IRQ_ADDR         /* Destination address             */
	la    a1, intdispatch      /* Starting address                */
	la    a2, (32*4)           /* Length of vector                */
	jal   copyhandler

	/* Clear Xinu-defined trap and interrupt vectors              */
	la    a0, TRAPVEC_ADDR
	la    a1, IRQVEC_END
	jal   clearvector
	
	/* Clear interrupt related registers in the coprocessor       */	
	mtc0  zero, CP0_STATUS     /* Clear interrupt masks           */
	mtc0  zero, CP0_CAUSE      /* Clear interrupt cause register  */

	/* Clear and invalidate the L1 instruction and data caches    */
	jal   flushcache
	
	/* Pass control to Xinu kernel.                               */
	j     _startup
	.end _start

/**
 * @fn void copyhandler(int *dst, int *src, uint bytes)
 * Copy text (code) from source to detination (in word-size chunks).
 * @param dst   location to store the source code
 * @param src   location holding the source code
 * @param bytes size of source code to copy
 */
	.ent copyhandler
copyhandler:
	lw    v0, 0(a1)
	sw    v0, 0(a0)
	addiu a1, a1, 4
	addiu a0, a0, 4
	addiu a2, a2, -4
	bgtz  a2, copyhandler
	jr    ra
	.end copyhandler

/**
 * @fn void clearvector(int *dstBegin, int *dstEnd)
 * Zero memory from dstBegin to dstEnd (non-inclusive).	
 * @param dstBegin start of the memory area of zero
 * @param dstEnd   end of the memory area to zero
 */
	.ent clearvector
clearvector:
	sw    zero, 0(a0)
	addiu a0, a0, 4
	blt   a0, a1, clearvector
	jr    ra
	.end clearvector

