/**
 * @file     xsh_ethstat.c
 * @provides xsh_ethstat
 *
 * $Id: xsh_ethstat.c 1577 2008-10-03 18:19:26Z mschul $
 */
/* Embedded Xinu, Copyright (C) 2008.  All rights reserved. */

#include <stddef.h>
#include <thread.h>
#include <device.h>
#include <ether.h>
#include <bcm4713.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <tty.h>

static void ethStat(ushort);
static void vethStat(ushort);

/**
 * Shell command (ethstat ETHNUM) provides information about the current
 * status of an Ethernet device.
 * @param nargs number of arguments in args array
 * @param args  array of arguments
 * @return non-zero value on error
 */
shellcmd xsh_ethstat(int nargs, char *args[])
{
    int dev;
    char n;

    /* Output help, if '--help' argument was supplied */
    if (nargs == 2 && strncmp(args[1], "--help", 6) == 0)
    {
        fprintf(stdout, "Usage: ethstat ETHNUM\n");
        fprintf(stdout, "Displays statistics for ethernet ETHNUM.\n");
        fprintf(stdout, "\t--help\t display this help and exit\n");
        return 1;
    }

    /* No arguments to ethstat gives statistics for all interfaces */
    if (nargs < 2)
    {
        for (n = 0; n < NETHER; n++)
        {
            ethStat(n);
        }

        return 0;
    }

    if (nargs > 2)
    {
        fprintf(stderr, "ethstat: too many arguments\n");
        fprintf(stderr, "Try 'ethstat --help' for more information\n");
        return 1;
    }

    /* Check for valid device */
    dev = atoi(args[1]);
    if (dev >= NETHER)
    {
        fprintf(stderr, "ethstat: (%d) No such ethernet device\n", dev);
        return 1;
    }

    ethStat(dev);

    return 0;
}

static void ethStat(ushort dev)
{
    struct ether *peth = NULL;  /* pointer to ether entry            */
    struct bcm4713 *pecsr = NULL;       /* pointer to ether device registers */
    ulong tmp;
    uchar mac[6];

    /* Initialize pointers */
    peth = &ethertab[dev];
    pecsr = peth->csr;

    fprintf(stdout, "eth%d:\n", dev);
    control(peth->dev->num, ETH_CTRL_GET_MAC, (long)mac, 0);
    fprintf(stdout,
            "  MAC Address  %02X:%02X:%02X:%02X:%02X:%02X",
            mac[0], mac[1], mac[2], mac[3], mac[4], mac[5]);

    if (peth->dev != peth->phy)
    {
        vethStat(dev);
        return;
    }

    fprintf(stdout, "  %8d MTU\n", peth->mtu);

    tmp = peth->state;
    fprintf(stdout, "  Device ");
    if (ETH_STATE_FREE == tmp)
    {
        fprintf(stdout, "FREE");
    }
    if (ETH_STATE_UP == tmp)
    {
        fprintf(stdout, "UP  ");
    }
    if (ETH_STATE_DOWN == tmp)
    {
        fprintf(stdout, "DOWN");
    }

    fprintf(stdout, "  RxErrors %d           ", peth->rxErrors);
    fprintf(stdout, "  Errors %d\n", peth->errors);

    /* Ether statistics */
    tmp = peth->interruptStatus;
    fprintf(stdout, "  IRQ Status   0x%08X [", tmp);
    if (tmp & ISTAT_PME)
    {
        fprintf(stdout, " PME");
    }
    if (tmp & ISTAT_TO)
    {
        fprintf(stdout, " TO");
    }
    if (tmp & ISTAT_DSCE)
    {
        fprintf(stdout, " DSCE");
    }
    if (tmp & ISTAT_DATAE)
    {
        fprintf(stdout, " DATAE");
    }
    if (tmp & ISTAT_DPE)
    {
        fprintf(stdout, " DPE");
    }
    if (tmp & ISTAT_RDU)
    {
        fprintf(stdout, " RDU");
    }
    if (tmp & ISTAT_RFO)
    {
        fprintf(stdout, " RFO");
    }
    if (tmp & ISTAT_TFU)
    {
        fprintf(stdout, " TFU");
    }
    if (tmp & ISTAT_RX)
    {
        fprintf(stdout, " RX");
    }
    if (tmp & ISTAT_TX)
    {
        fprintf(stdout, " TX");
    }
    if (tmp & ISTAT_EMAC)
    {
        fprintf(stdout, " EMAC");
    }
    if (tmp & ISTAT_MII_WRITE)
    {
        fprintf(stdout, " MII_WRITE");
    }
    if (tmp & ISTAT_MII_READ)
    {
        fprintf(stdout, " MII_READ");
    }
    fprintf(stdout, " ]\n");

    tmp = peth->interruptMask;
    fprintf(stdout, "  IRQ Mask     0x%08X [", tmp);
    if (tmp & ISTAT_PME)
    {
        fprintf(stdout, " PME");
    }
    if (tmp & ISTAT_TO)
    {
        fprintf(stdout, " TO");
    }
    if (tmp & ISTAT_DSCE)
    {
        fprintf(stdout, " DSCE");
    }
    if (tmp & ISTAT_DATAE)
    {
        fprintf(stdout, " DATAE");
    }
    if (tmp & ISTAT_DPE)
    {
        fprintf(stdout, " DPE");
    }
    if (tmp & ISTAT_RDU)
    {
        fprintf(stdout, " RDU");
    }
    if (tmp & ISTAT_RFO)
    {
        fprintf(stdout, " RFO");
    }
    if (tmp & ISTAT_TFU)
    {
        fprintf(stdout, " TFU");
    }
    if (tmp & ISTAT_RX)
    {
        fprintf(stdout, " RX");
    }
    if (tmp & ISTAT_TX)
    {
        fprintf(stdout, " TX");
    }
    if (tmp & ISTAT_EMAC)
    {
        fprintf(stdout, " EMAC");
    }
    if (tmp & ISTAT_MII_WRITE)
    {
        fprintf(stdout, " MII_WRITE");
    }
    if (tmp & ISTAT_MII_READ)
    {
        fprintf(stdout, " MII_READ");
    }
    fprintf(stdout, " ]\n");

    fprintf(stdout, "  Tx IRQ Count   %8d", peth->txirq);
    fprintf(stdout, "  Tx Octets      %8d", pecsr->txGoodOctets);
    fprintf(stdout, "  Tx Packets     %8d\n", pecsr->txGoodPackets);
    fprintf(stdout, "  Tx < 65 octets %8d", pecsr->tx_64);
    fprintf(stdout, "  Tx < 128       %8d", pecsr->tx_65_127);
    fprintf(stdout, "  Tx < 256       %8d\n", pecsr->tx_128_255);
    fprintf(stdout, "  Tx < 512       %8d", pecsr->tx_256_511);
    fprintf(stdout, "  Tx < 1024      %8d", pecsr->tx_512_1023);
    fprintf(stdout, "  Tx >= 1024     %8d\n", pecsr->tx_1024_max);
    fprintf(stdout, "  Tx Broadcast   %8d", pecsr->txBroadcast);
    fprintf(stdout, "  Tx Multicast   %8d", pecsr->txMulticast);
    fprintf(stdout, "  Tx Jabber      %8d\n", pecsr->txJabber);
    fprintf(stdout, "  Tx Oversize    %8d", pecsr->txOversize);
    fprintf(stdout, "  Tx Fragment    %8d", pecsr->txFragment);
    fprintf(stdout, "  Tx Underruns   %8d\n", pecsr->txUnderruns);
    fprintf(stdout, "  Tx Collisions  %8d", pecsr->txCollisions);
    fprintf(stdout, "  Tx Status    0x%08X\n", pecsr->dmaTxStatus);
    fprintf(stdout, "\n");
    fprintf(stdout, "  Rx IRQ Count   %8d", peth->rxirq);
    fprintf(stdout, "  Rx Octets      %8d", pecsr->rxGoodOctets);
    fprintf(stdout, "  Rx Packets     %8d\n", pecsr->rxGoodPackets);
    fprintf(stdout, "  Rx < 65 octets %8d", pecsr->rx_64);
    fprintf(stdout, "  Rx < 128       %8d", pecsr->rx_65_127);
    fprintf(stdout, "  Rx < 256       %8d\n", pecsr->rx_128_255);
    fprintf(stdout, "  Rx < 512       %8d", pecsr->rx_256_511);
    fprintf(stdout, "  Rx < 1024      %8d", pecsr->rx_512_1023);
    fprintf(stdout, "  Rx >= 1024     %8d\n", pecsr->rx_1024_max);
    fprintf(stdout, "  Rx Broadcast   %8d", pecsr->rxBroadcast);
    fprintf(stdout, "  Rx Multicast   %8d", pecsr->rxMulticast);
    fprintf(stdout, "  Rx Jabber      %8d\n", pecsr->rxJabber);
    fprintf(stdout, "  Rx Oversize    %8d", pecsr->rxOversize);
    fprintf(stdout, "  Rx Fragment    %8d", pecsr->rxFragment);
    fprintf(stdout, "  Rx Missed      %8d\n", pecsr->rxMissed);
    fprintf(stdout, "  Rx CRC Align   %8d", pecsr->rxCrcAlign);
    fprintf(stdout, "  Rx Undersize   %8d", pecsr->rxUndersize);
    fprintf(stdout, "  Rx CRC Error   %8d\n", pecsr->rxCrc);
    fprintf(stdout, "  Rx Alignment   %8d", pecsr->rxAlign);
    fprintf(stdout, "  Rx Symbol Err  %8d", pecsr->rxSymbol);
    fprintf(stdout, "  Rx Pause       %8d\n", pecsr->rxPause);
    fprintf(stdout, "  Rx Non-Pause   %8d", pecsr->rxNonPause);

    tmp = pecsr->dmaRxLast / sizeof(struct dmaDescriptor);
    fprintf(stdout, "  Rx DMA Last    %8d", tmp);

    tmp = (pecsr->dmaRxStatus & DMARX_STAT_CDMASK)
        / sizeof(struct dmaDescriptor);
    fprintf(stdout, "  Rx DMA Stat    %8d\n", tmp);

    fprintf(stdout, "\n");
}

static void vethStat(ushort dev)
{
    struct ether *peth = NULL;  /* pointer to ether entry    */
    ulong tmp;

    /* Initialize pointers */
    peth = &ethertab[dev];

    tmp = peth->state;
    fprintf(stdout, "\n  Device ");
    if (ETH_STATE_FREE == tmp)
    {
        fprintf(stdout, "FREE");
    }
    if (ETH_STATE_UP == tmp)
    {
        fprintf(stdout, "UP  ");
    }
    if (ETH_STATE_DOWN == tmp)
    {
        fprintf(stdout, "DOWN");
    }

    fprintf(stdout, "  RxErrors %d\n\n", peth->rxErrors);
}
