/** 
 * @file     xsh_help.c
 * @provides xsh_help
 *
 * $Id: xsh_help.c 1577 2008-10-03 18:19:26Z mschul $
 */
/* Embedded Xinu, Copyright (C) 2008.  All rights reserved. */

#include <stddef.h>
#include <shell.h>
#include <stdio.h>
#include <string.h>

/**
 * Shell command (help) provides a list of commands recognized by the
 * shell, or displays help information for a particular command.
 * @param nargs number of arguments in args array
 * @param args  array of arguments
 * @return non-zero value on error
 */
shellcmd xsh_help(int nargs, char *args[])
{
    uchar i;
    char *command_args[2];      /* temporary storage for [command] --help */

    /* Output help, if '--help' argument was supplied */
    if (nargs == 2 && strncmp(args[1], "--help", 6) == 0)
    {
        fprintf(stdout, "Usage: help [command]\n");
        fprintf(stdout, "Provides a list of commands for the shell.\n");
        fprintf(stdout, "If command is provided, help information will ");
        fprintf(stdout,
                "be provided for the specified command; equivalent ");
        fprintf(stdout, "to entering 'command --help' into the shell.\n");
        fprintf(stdout,
                "\tcommand\t command name to display for which to ");
        fprintf(stdout, "display help information\n");
        fprintf(stdout, "\t--help\t display this help and exit\n");
        return 0;
    }

    /* Check for correct number of arguments */
    if (nargs > 2)
    {
        fprintf(stderr, "help: too many arguments\n");
        fprintf(stderr, "Try 'help --help' for more information.\n");
        return 1;
    }

    /* Output help for specific command, if 'command' argument was supplied */
    if (nargs == 2)
    {
        for (i = 0; i < ncommand; i++)
        {
            if (strncmp(args[1], commandtab[i].name, 6) == 0)
            {
                command_args[0] = args[2];
                command_args[1] = "--help";
                (*commandtab[i].procedure) (2, command_args);
                return 0;
            }
        }
        fprintf(stdout, "help: no help topics match '%s'.", args[1]);
        fprintf(stdout, "  Try 'help --help' for more information.\n");
        return 1;
    }

    /* Output command list */
    fprintf(stdout, "Shell Commands:\n");
    for (i = 0; i < ncommand; i++)
    {
        fprintf(stdout, "\t%s\n", commandtab[i].name);
    }

    return 0;
}
