/**
 * @file     xsh_kill.c
 * @provides xsh_kill
 *
 * $Id: xsh_kill.c 1577 2008-10-03 18:19:26Z mschul $
 */
/* Embedded Xinu, Copyright (C) 2008.  All rights reserved. */

#include <stddef.h>
#include <thread.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

/**
 * Shell command (kill) terminates a thread.
 * @param nargs number of arguments in args array
 * @param args  array of arguments
 * @return non-zero value on error
 */
shellcmd xsh_kill(int nargs, char *args[])
{
    tid_typ tid;                /* tid for thread to kill */

    /* Output help, if '--help' argument was supplied */
    if (nargs == 2 && strncmp(args[1], "--help", 6) == 0)
    {
        fprintf(stdout, "Kills a thread number PID.\n");
        fprintf(stdout, "\t--help\t display this help and exit\n");
        return 0;
    }

    /* Check for correct number of arguments */
    if (nargs < 2)
    {
        fprintf(stderr, "kill: missing operand\n");
        fprintf(stderr, "Try 'kill --help' for more information\n");
        return 1;
    }
    if (nargs > 2)
    {
        fprintf(stderr, "kill: too many arguments\n");
        fprintf(stderr, "Try 'kill --help' for more information\n");
        return 1;
    }

    tid = atoi(args[1]);

    /* Block killing of the null thread */
    if (tid == NULLTHREAD)
    {
        fprintf(stderr, "kill: (%d) Operation not permitted\n", tid);
        return 1;
    }

    /* Notify of killing of the shell */
    if (tid == gettid())
    {
        fprintf(stderr, "kill: Shell killed\n");
        sleep(2000);
    }

    /* Kill thread, noting if kill failed */
    if (kill(tid) == SYSERR)
    {
        fprintf(stderr, "kill: (%d) No such thread\n", tid);
        return -1;
    }

    return 0;
}
