/**
 * @file     xsh_led.c
 * @provides xsh_led
 *
 * $Id: xsh_led.c 1577 2008-10-03 18:19:26Z mschul $
 */
/* Embedded Xinu, Copyright (C) 2008.  All rights reserved. */

#include <stddef.h>
#include <gpio.h>
#include <stdio.h>
#include <string.h>

/**
 * Shell command (led DESCRIPTOR STATE) turns a specified led on or off.
 * @param nargs number of arguments in args array
 * @param args  array of arguments
 * @return non-zero value on error
 */
shellcmd xsh_led(int nargs, char *args[])
{
    ulong led;                  /* led to control */

    /* Output help, if '--help' argument was supplied */
    if (nargs == 2 && strncmp(args[1], "--help", 6) == 0)
    {
        fprintf(stdout, "Usage:led DESCRIPTOR STATE\n");
        fprintf(stdout, "Turns a specified led on or off.\n");
        fprintf(stdout,
                "\tDESCRIPTOR: dmz, wlan, power, ciscow, ciscoo\n");
        fprintf(stdout, "\tSTATE: on, off\n");
        fprintf(stdout, "\t--help\t display this help and exit\n");
        return 0;
    }

    /* Check for correct number of arguments */
    if (nargs < 3)
    {
        fprintf(stderr, "led: missing operand\n");
        fprintf(stderr, "Try 'led --help' for more information\n");
        return 1;
    }
    if (nargs > 3)
    {
        fprintf(stderr, "led: too many arguments\n");
        fprintf(stderr, "Try 'led --help' for more information\n");
        return 1;
    }

    /* find led to change */
    if (strncmp(args[1], "dmz", 3) == 0)
    {
        led = GPIO_LED_DMZ;
    }
    else if (strncmp(args[1], "wlan", 4) == 0)
    {
        led = GPIO_LED_WLAN;
    }
    else if (strncmp(args[1], "power", 5) == 0)
    {
        led = GPIO1;
    }
    else if (strncmp(args[1], "ciscow", 6) == 0)
    {
        led = GPIO_LED_CISCOWHT;
    }
    else if (strncmp(args[1], "ciscoo", 6) == 0)
    {
        led = GPIO_LED_CISCOONG;
    }
    else
    {
        fprintf(stderr, "led: invalid DESCRIPTOR\n");
        fprintf(stderr, "Try 'led --help' for more information\n");
    }

    /* change led state */
    if (strncmp(args[2], "on", 2) == 0)
    {
        gpioLEDOn(led);
    }
    else if (strncmp(args[2], "off", 3) == 0)
    {
        gpioLEDOff(led);
    }
    else
    {
        fprintf(stderr, "led: invalid STATE\n");
        fprintf(stderr, "Try 'led --help' for more information\n");
    }

    return 0;
}
