/**
 * @file     xsh_memstat.c
 * @provides xsh_memstat
 *
 * $Id: xsh_memstat.c 1577 2008-10-03 18:19:26Z mschul $
 */
/* Embedded Xinu, Copyright (C) 2008.  All rights reserved. */

#include <stddef.h>
#include <platform.h>
#include <mips.h>
#include <memory.h>
#include <stdio.h>
#include <string.h>

extern char *maxaddr;
extern void _start(void);

/**
 * Shell command (gpiostat) provides memory use and free list information.
 * @param nargs number of arguments in args array
 * @param args  array of arguments
 * @return non-zero value on error
 */
shellcmd xsh_memstat(int nargs, char *args[])
{
    struct memblock *mptr;      /* pointer to memory block       */
    ulong phys = 0;             /* total physical memory         */
    ulong resrv = 0;            /* total reserved system memory  */
    ulong code = 0;             /* total Xinu code memory        */
    ulong stack = 0;            /* total stack memory            */
    ulong heap = 0;             /* total heap memory             */
    ulong free = 0;             /* total free memory             */
    long i;                     /* temp variable                 */

    /* Output help, if '--help' argument was supplied */
    if (nargs == 2 && strncmp(args[1], "--help", 6) == 0)
    {
        fprintf(stdout, "Usage: memstat\n");
        fprintf(stdout,
                "Displays the current memory usage and prints the");
        fprintf(stdout, "free list.\n");
        fprintf(stdout, "\t--help\t display this help and exit\n");
        return 0;
    }

    /* Check for correct number of arguments */
    if (nargs > 1)
    {
        fprintf(stderr, "memstat: too many arguments\n");
        fprintf(stderr, "Try 'memstat --help' for more information\n");
        return 1;
    }

    /* Calculate amount of physical memory */
    phys = (ulong)platform.maxaddr - (ulong)KSEG0_BASE;

    /* Calculate amount of reserved system memory */
    resrv = (ulong)_start - (ulong)KSEG0_BASE;

    /* Calculate amount of text memory */
    code = (ulong)&_end - (ulong)_start;

    /* Caculate amount of stack memory */
    for (i = 0; i < NTHREAD; i++)
    {
        if (thrtab[i].state != THRFREE)
        {
            stack += (ulong)thrtab[i].stklen;
        }
    }

    /* Calculate amount of free memory */
    for (mptr = memlist.next; mptr != NULL; mptr = mptr->next)
    {
        free += mptr->length;
    }

    /* Caculate amount of heap memory */
    heap = phys - resrv - code - stack - free;

    /* Ouput current memory usage */
    fprintf(stdout, "Current System Memory Usage:\n");
    fprintf(stdout, "----------------------------\n");
    fprintf(stdout, "%10d bytes system area\n", resrv);
    fprintf(stdout, "%10d bytes Xinu code\n", code);
    fprintf(stdout, "%10d bytes stack space\n", stack);
    fprintf(stdout, "%10d bytes heap space\n", heap);
    fprintf(stdout, "%10d bytes free space\n", free);
    fprintf(stdout, "----------------------------\n");
    fprintf(stdout, "%10d bytes physical memory\n", phys);

    /* check computed free size against maintained free size */
    if (memlist.length != free)
    {
        fprintf(stderr, "\nPotential memory leak detected:\n");
        fprintf(stderr, "\nComputed free memory: %10d bytes\n", free);
        fprintf(stderr,
                "Maintained free memory: %10d bytes\n\n", memlist.length);
    }

    /* Output free list */
    fprintf(stdout, "Free List:\n");
    fprintf(stdout, "BLOCK START  LENGTH  \n");
    fprintf(stdout, "-----------  --------\n");

    for (mptr = memlist.next; mptr != NULL; mptr = mptr->next)
    {
        fprintf(stdout, "0x%08X   %8d\n", mptr, mptr->length);
    }

    return 0;
}
