/**
 * @file     xsh_nvram.c
 * @provides xsh_nvram
 * Shell function wrapper to lower level nvram settings.
 *
 * $Id: xsh_nvram.c 1577 2008-10-03 18:19:26Z mschul $
 */
/* Embedded Xinu, Copyright (C) 2008.  All rights reserved. */

#include <kernel.h>
#include <nvram.h>
#include <stdio.h>
#include <string.h>

static void xsh_nvramHelp(void);
static void xsh_nvramList(void);

/**
 * Command 'nvram', allows shell access to nvram variables
 * @param nargs number of arguments
 * @param args  array of arguments
 * @return non-zero vaule on error
 */
shellcmd xsh_nvram(int nargs, char *args[])
{
    char *value;

    /* make sure nvram is initialized */
    if (SYSERR == nvramInit())
    {
        fprintf(stderr, "error: device does not appear to have nvram.\n");
    }

    if (2 == nargs && 0 == strncmp(args[1], "list", 4))
    {
        xsh_nvramList();
        return 0;
    }
    else if (3 == nargs && 0 == strncmp(args[1], "get", 3))
    {
        value = nvramGet(args[2]);
        if (value)
        {
            fprintf(stdout, "%s\n", nvramGet(args[2]));
            return 0;
        }
        else
        {
            fprintf(stderr, "no binding\n");
            return 1;
        }
    }

    xsh_nvramHelp();

    return 1;
}

/**
 * Display help information about nvram command
 */
static void xsh_nvramHelp(void)
{
    fprintf(stdout, "Usage: nvram [--help] [get name] [list]\n");
    fprintf(stdout, "\t--help\t\tdisplay this help and exit\n");
    fprintf(stdout,
            "\tlist\t\tshow a listing of all name=value tuples\n");
    fprintf(stdout,
            "\tget name\tdisplay tuple name=value, if name exists\n");
}

/**
 * Provide a listing of all the nvram variables known on Flash
 */
static void xsh_nvramList(void)
{
    ushort n, count;
    struct nvram_tuple *tuple;

    count = 0;
    for (n = 0; n < NVRAM_NHASH; n++)
    {
        tuple = nvram_tuples[n];
        if (NULL == tuple)
        {
            continue;
        }
        do
        {
            fprintf(stdout, "%s\n", tuple->pair);
            count++;
        }
        while ((tuple = tuple->next) != NULL);
    }

    fprintf(stdout,
            "%d pairs using %d bytes (%d bytes remaining)\n",
            count, nvram_header->length,
            NVRAM_SIZE - nvram_header->length);
}
