/**
 * @file     xsh_ps.c 
 * @provides xsh_ps
 *
 * $Id: xsh_ps.c 1577 2008-10-03 18:19:26Z mschul $
 */
/* Embedded Xinu, Copyright (C) 2008.  All rights reserved. */

#include <stddef.h>
#include <thread.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

/**
 * Shell command (ps) outputs thread table information.
 * @param nargs number of arguments in args array
 * @param args  array of arguments
 * @return non-zero value on error
 */
shellcmd xsh_ps(int nargs, char *args[])
{
    struct thrent *thrptr;      /* pointer to thread entry  */
    uchar i;                    /* temp variable            */

    /* readable names for PR* status in thread.h */
    char *pstnams[] = { "curr ", "free ", "ready", "recv ",
        "sleep", "susp ", "wait ", "rtim "
    };

    /* Output help, if '--help' argument was supplied */
    if (nargs == 2 && strncmp(args[1], "--help", 6) == 0)
    {
        fprintf(stdout, "Usage: ps\n");
        fprintf(stdout, "Displays a table of running threads.\n");
        fprintf(stdout, "\t--help\t display this help and exit\n");

        return 0;
    }

    /* Check for correct number of arguments */
    if (nargs > 1)
    {
        fprintf(stderr, "ps: too many arguments\n");
        fprintf(stderr, "Try 'ps --help' for more information\n");
        return 1;
    }

    /* Output thread table header */
    fprintf(stdout,
            "TID NAME         STATE PRIO PPID STACK BASE STACK PTR  STACK LEN \n");
    fprintf(stdout,
            "--- ------------ ----- ---- ---- ---------- ---------- ----------\n");

    /* Output information for each thread */
    for (i = 0; i < NTHREAD; i++)
    {
        thrptr = &thrtab[i];
        if (thrptr->state == THRFREE)
        {
            continue;
        }

        fprintf(stdout,
                "%3d %-12s %s %4d %4d 0x%08X 0x%08X %10d\n",
                i, thrptr->name,
                pstnams[(int)thrptr->state - 1],
                thrptr->prio, thrptr->parent,
                thrptr->stkbase, thrptr->stkptr, thrptr->stklen);
    }

    return 0;
}
