/**
 * @file     xsh_sleep.c
 * @provides xsh_sleep
 *
 * $Id: xsh_sleep.c 1577 2008-10-03 18:19:26Z mschul $
 */
/* Embedded Xinu, Copyright (C) 2008.  All rights reserved. */

#include <stddef.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <clock.h>

/**
 * Shell command (sleep DELAY) delay for a given number of seconds.
 * @param nargs number of arguments
 * @param args  array of arguments
 * @return non-zero value on error
 */
shellcmd xsh_sleep(int nargs, char *args[])
{
    long delay;                 /* time to sleep */

    /* Output help, if '--help' argument was supplied */
    if (nargs == 2 && strncmp(args[1], "--help", 6) == 0)
    {
        fprintf(stdout, "Usage:sleep DELAY\n");
        fprintf(stdout, "Sleep for DELAY seconds.\n");
        fprintf(stdout, "\t--help\t display this help and exit\n");
        return 0;
    }

    /* Check for correct number of arguments */
    if (nargs < 2)
    {
        fprintf(stderr, "sleep: missing operand\n");
        fprintf(stderr, "Try 'sleep --help' for more information\n");
        return 1;
    }
    if (nargs > 2)
    {
        fprintf(stderr, "sleep: too many arguments\n");
        fprintf(stderr, "Try 'sleep --help' for more information\n");
        return 1;
    }

    /* Calculate delay and sleep */
    delay = atol(args[1]);
    if (delay < 0)
    {
        fprintf(stderr, "sleep: invalid delay of %d seconds\n");
    }

    /* convert delay to milliseconds */
    if (sleep(delay * 1000) == SYSERR)
    {
        return -1;
    }

    return 0;
}
