/**
 * @file     xsh_uartstat.c
 * @provides xsh_uartstat
 *
 * $Id: xsh_uartstat.c 1577 2008-10-03 18:19:26Z mschul $
 */
/* Embedded Xinu, Copyright (C) 2008.  All rights reserved. */

#include <stddef.h>
#include <thread.h>
#include <device.h>
#include <uart.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

static void uartStat(ushort);

/**
 * Shell command (uartstat UARTNUM) provides information about the current 
 * status of a UART.
 * @param nargs number of arguments in args array
 * @param args  array of arguments
 * @return non-zero value on error
 */
shellcmd xsh_uartstat(int nargs, char *args[])
{
    ushort dev;
    char n;

    /* Output help, if '--help' argument was supplied */
    if (nargs == 2 && strncmp(args[1], "--help", 6) == 0)
    {
        fprintf(stdout, "Usage: uartstat [uartnum]\n");
        fprintf(stdout,
                "Displays statistics for uart uartnum.  If no uartnum is specified, statistics are displayed for all uarts.\n");
        fprintf(stdout,
                "\tuartnum\t the number of the uart to display statistics for\n");
        fprintf(stdout, "\t--help\t display this help and exit\n");
        return 1;
    }

    /* Check for correct number of arguments */
    if (nargs < 2)
    {
        for (n = 0; n < NUART; n++)
        {
            uartStat(n);
        }

        return 0;
    }

    if (nargs > 2)
    {
        fprintf(stderr, "uartstat: too many arguments\n");
        fprintf(stderr, "Try 'uartstat --help' for more information\n");
        return 1;
    }

    /* Check for valid device */
    dev = (ushort)atol(args[1]);
    if (dev >= NUART)
    {
        fprintf(stderr, "uartstat: (%d) No such uart\n", dev);
        return 1;
    }

    uartStat(dev);

    return 0;
}

static void uartStat(ushort dev)
{
    struct uart *puart;         /* pointer to uart entry     */
    struct uart_csreg *pucsr;   /* pointer to uart registers */
    char *label[2] = { "OFF", "ON " };

    /* Initialize pointers */
    puart = &uarttab[dev];
    pucsr = puart->csr;

    fprintf(stdout, "uart%d:\n", dev);

    /* UART statistics */
    fprintf(stdout, "\tSTATISTICS:\n");
    fprintf(stdout, "\t------------------------------------------\n");
    fprintf(stdout, "\t%8d Characters Output\n", puart->cout);
    fprintf(stdout, "\t%8d Characters Input\n", puart->cin);
    fprintf(stdout, "\t%8d Characters Overrun\n", puart->ovrrn);
    fprintf(stdout, "\t%8d Receiver Error Count\n", puart->lserr);
    fprintf(stdout, "\t%8d Output IRQ Count\n", puart->oirq);
    fprintf(stdout, "\t%8d Input IRQ Count\n", puart->iirq);

    /* Interrupt enable status */
    int enabled;

    fprintf(stdout, "\n\tINTERRUPT ENABLE:\n");
    fprintf(stdout, "\t------------------------------------------\n");

    enabled = (pucsr->ier & UART_IER_ERBFI) ? 1 : 0;
    fprintf(stdout,
            "\t%s  Receiver FIFO Reached Trigger Level\n",
            label[enabled]);

    enabled = (pucsr->ier & UART_IER_ETBEI) ? 1 : 0;
    fprintf(stdout, "\t%s  Transmitter FIFO Empty\n", label[enabled]);

    enabled = (pucsr->ier & UART_IER_ELSI) ? 1 : 0;
    fprintf(stdout, "\t%s  Receiver Error or BREAK\n", label[enabled]);

    enabled = (pucsr->ier & UART_IER_EMSI) ? 1 : 0;
    fprintf(stdout, "\t%s  Modem Status", label[enabled]);

    fprintf(stdout, "\n");
}
