/**
 * @file xsh_vlanstat.c
 * @provides xsh_vlanstat
 *
 * $Id: xsh_vlanstat.c 1577 2008-10-03 18:19:26Z mschul $
 */
/* Embedded Xinu, Copyright (C) 2008.  All rights reserved. */

#include <kernel.h>
#include <bcmswitch.h>
#include <bcm4713.h>
#include <stdio.h>
#include <string.h>

static int vlanStat(void);

/**
 * Provides info about the current status of VLAN switch.
 * @param nargs number of arguments in args array
 * @param args  array of arguments
 * @return non-zero value on error
 */
shellcmd xsh_vlanstat(int nargs, char *args[])
{
    /* Output help, if '--help' argument was supplied */
    if (nargs == 2 && strncmp(args[1], "--help", 6) == 0)
    {
        fprintf(stdout, "Usage: vlanstat\n");
        fprintf(stdout, "Displays status of VLAN switch.\n");
        fprintf(stdout, "\t--help\t display this help and exit\n");
        return 1;
    }

    /* Check for correct number of arguments */
    if (nargs == 1)
    {
        return vlanStat();
    }
    else
    {
        fprintf(stderr, "vlanstat: too many arguments\n");
        fprintf(stderr, "Try 'vlanstat --help' for more information\n");
        return 1;
    }
}

static int vlanStat(void)
{
    struct mdioregs *mdio = NULL;
    ulong result = 0;
    int table = 0, port = 0;
    ushort deftags[SWT_PORTS];

    if (NULL == swtregs.mdio)
    {
        fprintf(stderr, "Switch Core not found!\n");
        return 1;
    }
    mdio = swtregs.mdio;

    fprintf(stdout, "VLAN Switch Status:\n");

    switchReadReg(mdio, SWT_PAGE_VLAN, SWT_REG_VLAN_CONTROL0, &result);
    fprintf(stdout, "\tControl %d : %02X, ",
            SWT_REG_VLAN_CONTROL0, result);
    switchReadReg(mdio, SWT_PAGE_VLAN, SWT_REG_VLAN_CONTROL1, &result);
    fprintf(stdout, "%d : %02X, ", SWT_REG_VLAN_CONTROL1, result);
    switchReadReg(mdio, SWT_PAGE_VLAN, SWT_REG_VLAN_CONTROL2, &result);
    fprintf(stdout, "%d : %02X, ", SWT_REG_VLAN_CONTROL2, result);
    switchReadReg(mdio, SWT_PAGE_VLAN, SWT_REG_VLAN_CONTROL3, &result);
    fprintf(stdout, "%d : %02X, ", SWT_REG_VLAN_CONTROL3, result);
    switchReadReg(mdio, SWT_PAGE_VLAN, SWT_REG_VLAN_CONTROL4, &result);
    fprintf(stdout, "%d : %02X, ", SWT_REG_VLAN_CONTROL4, result);
    switchReadReg(mdio, SWT_PAGE_VLAN, SWT_REG_VLAN_CONTROL5, &result);
    fprintf(stdout, "%d : %02X\n", SWT_REG_VLAN_CONTROL5, result);

    fprintf(stdout, "\tVLAN Table\n");
    for (table = 0; table < SWT_VLAN_TABLEMAX; table++)
    {
        result = 0x2000 | table;
        switchWriteReg(mdio, SWT_PAGE_VLAN, SWT_REG_VLAN_TABACC, &result);
        switchReadReg(mdio, SWT_PAGE_VLAN, SWT_REG_VLAN_READ, &result);
        if (result & SWT_VLAN_GROUPMASK)
        {
            fprintf(stdout, "\t    VID %2d:", table);
            for (port = 0; port < SWT_PORTS; port++)
            {
                if (result & (1 << port))
                {
                    fprintf(stdout, " %d", port);
                    if (!(result & (1 << (port + SWT_PORTS))))
                    {
                        if (port == SWT_CPUPORT)
                        {
                            fprintf(stdout, "*");
                        }
                        else
                        {
                            fprintf(stdout, "t");
                        }
                    }
                }
            }
            fprintf(stdout, "\n");
        }
    }

    for (port = 0; port < SWT_PORTS; port++)
    {
        switchReadReg(mdio,
                      SWT_PAGE_VLAN,
                      SWT_REG_VLAN_TAGPORT0 + 2 * port, &result);
        deftags[port] = result;
        fprintf(stdout,
                "\tDefault Port %d tag: 0x%04X\r\n", port, deftags[port]);
    }

    fprintf(stdout, "\n");

    return 0;
}
