/**
 * @file initialize.c
 * @provides nulluser, sysinit
 * The system begins intializing after the C environment has been
 * established.  After intialization, the null thread remains always in
 * a ready (THRREADY) or running (THRCURR) state.
 *
 * $Id: initialize.c 1577 2008-10-03 18:19:26Z mschul $
 */
/* Embedded Xinu, Copyright (C) 2008.  All rights reserved. */

#include <kernel.h>
#include <backplane.h>
#include <clock.h>
#include <device.h>
#include <gpio.h>
#include <memory.h>
#include <bufpool.h>
#include <mips.h>
#include <platform.h>
#include <thread.h>
#include <queue.h>
#include <semaphore.h>
#include <mailbox.h>
#include <stddef.h>
#include <stdio.h>
#include <string.h>

/* Linker provides start and end of image */
extern void _start(void);       /* start of Xinu code                  */

/* Function prototypes */
extern void main(void);         /* main is the first thread created    */
extern void xdone(void);        /* system "shutdown" procedure         */
static int platforminit(void);  /* determines platform settings        */
static int sysinit(void);       /* intializes system structures        */

/* Declarations of major kernel variables */
struct thrent thrtab[NTHREAD];  /* Thread table                   */
struct sement semtab[NSEM];     /* Semaphore table                */
qid_typ readylist;              /* List of READY threads          */
struct memblock memlist;        /* List of free memory blocks     */
struct bfpentry bfptab[NPOOL];  /* List of memory buffer pools    */

/* Active system status */
int thrcount;                   /* Number of live user threads         */
tid_typ thrcurrent;             /* Id of currently running thread      */

/* Params set by startup.S */
void *memheap;                  /* Bottom of heap (top of O/S stack)   */
ulong cpuid;                    /* Processor id                        */

struct platform platform;       /* Platform specific configuration     */

/**
 * Intializes the system and becomes the null thread.
 * This is where the system begins after the C environment has been 
 * established.  Interrupts are initially DISABLED, and must eventually 
 * be enabled explicitly.  This routine turns itself into the null thread 
 * after initialization.  Because the null thread must always remain ready 
 * to run, it cannot execute code that might cause it to be suspended, wait 
 * for a semaphore, or put to sleep, or exit.  In particular, it must not 
 * do I/O unless it uses kprintf for synchronous output.
 */
int nulluser(void)
{
    kprintf(VERSION);
    kprintf("\r\n\r\n");

    platforminit();

#ifdef DETAIL
    /* Output detected platform. */
    kprintf("Processor identification: 0x%08X\r\n", cpuid);
    kprintf("Detected platform as: %s\r\n\r\n", platform.name);
#endif

#ifdef SB_BUS
    backplaneInit(NULL);
#endif                          /* SB_BUS */

    sysinit();

    /* Output Xinu memory layout */
    kprintf("%10d bytes physical memory.\r\n",
            (ulong)platform.maxaddr & 0x7FFFFFFF);
#ifdef DETAIL
    kprintf("           [0x%08X to 0x%08X]\r\n",
            (ulong)KSEG0_BASE, (ulong)(platform.maxaddr - 1));
#endif
    kprintf("%10d bytes reserved system area.\r\n",
            (ulong)_start - KSEG0_BASE);
#ifdef DETAIL
    kprintf("           [0x%08X to 0x%08X]\r\n",
            (ulong)KSEG0_BASE, (ulong)_start - 1);
#endif

    kprintf("%10d bytes Xinu code.\r\n", (ulong)&_end - (ulong)_start);
#ifdef DETAIL
    kprintf("           [0x%08X to 0x%08X]\r\n",
            (ulong)_start, (ulong)&_end - 1);
#endif

    kprintf("%10d bytes stack space.\r\n", (ulong)memheap - (ulong)&_end);
#ifdef DETAIL
    kprintf("           [0x%08X to 0x%08X]\r\n",
            (ulong)&_end, (ulong)memheap - 1);
#endif

    kprintf("%10d bytes heap space.\r\n",
            (ulong)platform.maxaddr - (ulong)memheap);
#ifdef DETAIL
    kprintf("           [0x%08X to 0x%08X]\r\n\r\n",
            (ulong)memheap, (ulong)platform.maxaddr - 1);
#endif
    kprintf("\r\n");

    open(CONSOLE, SERIAL0);

    /* enable interrupts here */
    enable();

    ready(create
          ((void *)main, INITSTK, INITPRIO, "MAIN", 2, 0,
           NULL), RESCHED_YES);

    while (TRUE)
    {
        pause();
    }
}

/**
 * Intializes all Xinu data structures and devices.
 * @return OK if everything is initialized successfully
 */
static int sysinit(void)
{
    int i;
    struct thrent *thrptr;      /* thread control block pointer  */
    device *pdev;               /* device entry pointer          */
    struct sement *psem;        /* semaphore entry pointer       */
    struct memblock *pmblock;   /* memory block pointer          */
    struct bfpentry *bfpptr;

    /* Initialize system variables */
    /* Count this NULLTHREAD as the first thread in the system. */
    thrcount = 1;

    /* Initialize free memory list */
    memheap = roundmb(memheap);
    platform.maxaddr = truncmb(platform.maxaddr);
    memlist.next = pmblock = (struct memblock *)memheap;
    memlist.length = (uint)(platform.maxaddr - memheap);
    pmblock->next = NULL;
    pmblock->length = (uint)(platform.maxaddr - memheap);

    /* Initialize thread table */
    for (i = 0; i < NTHREAD; i++)
    {
        thrtab[i].state = THRFREE;
    }

    /* initialize null thread entry */
    thrptr = &thrtab[NULLTHREAD];
    thrptr->state = THRCURR;
    thrptr->prio = 0;
    strncpy(thrptr->name, "prnull", 7);
    thrptr->stkbase = (void *)&_end;
    thrptr->stklen = (ulong)memheap - (ulong)&_end;
    thrptr->stkptr = 0;
    thrcurrent = NULLTHREAD;

    /* Initialize semaphores */
    for (i = 0; i < NSEM; i++)
    {
        psem = &semtab[i];
        psem->state = SFREE;
        psem->count = 0;
        psem->queue = queinit();
    }

    /* Initialize buffer pools */
    for (i = 0; i < NPOOL; i++)
    {
        bfpptr = &bfptab[i];
        bfpptr->state = BFPFREE;
    }

    /* initialize thread ready list */
    readylist = queinit();

#ifdef RTCLOCK
    /* initialize real time clock */
    clkinit();
#endif

#ifdef NMAILBOX
    /* intialize mailboxes */
    mailboxInit();
#endif

#ifdef NDEVS
    for (i = 0; i < NDEVS; i++)
    {
        if (!isbaddev(i))
        {
            pdev = &devtab[i];
            (pdev->init) (pdev);
        }
    }
#endif

#ifdef NVRAM_LOCATION
    nvramInit();
#endif

    gpioLEDOn(GPIO_LED_CISCOWHT);

    return OK;
}

/**
 * Determines and stores all platform specific information.
 * @return OK if everything is determined successfully
 */
static int platforminit(void)
{
    switch (cpuid & PRID_REV)
    {
    case PRID_REV_WRT54G:
        strncpy(platform.name, "Linksys WRT54G", 16);
        platform.maxaddr = (void *)(KSEG0_BASE | MAXADDR_WRT54G);
        platform.clkfreq = CLKFREQ_WRT54G;
        platform.uart_dll = UART_DLL_WRT54G;
        break;
    case PRID_REV_WRT54GL:
        strncpy(platform.name, "Linksys WRT54GL", 16);
        platform.maxaddr = (void *)(KSEG0_BASE | MAXADDR_WRT54GL);
        platform.clkfreq = CLKFREQ_WRT54GL;
        platform.uart_dll = UART_DLL_WRT54GL;
        break;
    case PRID_REV_WRT350N:
        strncpy(platform.name, "Linksys WRT350N", 16);
        platform.maxaddr = (void *)(KSEG0_BASE | MAXADDR_WRT350N);
        platform.clkfreq = CLKFREQ_WRT350N;
        platform.uart_dll = UART_DLL_WRT350N;
        break;
    default:
        strncpy(platform.name, "Unknown Platform", 16);
        platform.maxaddr = (void *)(KSEG0_BASE | MAXADDR_DEFAULT);
        platform.clkfreq = CLKFREQ_DEFAULT;
        platform.uart_dll = UART_DLL_DEFAULT;
        return SYSERR;
    }

    return OK;
}
