/**
 * @file kprintf.c
 * @provides kputc, kprintf
 *
 * $Id: kprintf.c 1577 2008-10-03 18:19:26Z mschul $
 */
/* Embedded Xinu, Copyright (C) 2008.  All rights reserved. */

#include <stddef.h>
#include <stdarg.h>
#include <device.h>
#include <stdio.h>
#include <uart.h>

/**
 * perform a synchronous character write to a serial port
 * @param *pdev pointer to device on which to write character
 * @param c character to write
 * @return c on success, SYSERR on failure
 */
syscall kputc(device * pdev, uchar c)
{
    int status, irmask;
    struct uart *puart;
    volatile struct uart_csreg *pucsr;

    if (NULL == pdev)
    {
        return SYSERR;
    }
    puart = (struct uart *)pdev->controlblk;
    if (NULL == puart)
    {
        puart = &uarttab[0];
    }
    pucsr = puart->csr;
    if (NULL == pucsr)
    {
        pucsr = (struct uart_csreg *)pdev->csr;
    }

    irmask = pucsr->ier;        /* Save UART interrupt state.   */
    pucsr->ier = 0;             /* Disable UART interrupts.     */

    do                          /* Wait for transmitter         */
    {
        status = pucsr->lsr;
    }
    while ((status & UART_LSR_TEMT) != UART_LSR_TEMT);

    /* Send character. */
    pucsr->thr = c;
    puart->cout++;

    pucsr->ier = irmask;        /* Restore UART interrupts.     */
    return c;
}

/**
 * perform a synchronous kernel read from a serial device
 * @param *pdev pointer to device on which to write character
 * @return character read on success, SYSERR on failure
 */
syscall kgetc(device * pdev)
{
    int irmask;
    struct uart *puart;
    volatile struct uart_csreg *pucsr;
    uchar c;

    if (NULL == pdev)
    {
        return SYSERR;
    }
    puart = (struct uart *)pdev->controlblk;
    if (NULL == puart)
    {
        puart = &uarttab[0];
    }
    pucsr = puart->csr;
    if (NULL == pucsr)
    {
        pucsr = (struct uart_csreg *)pdev->csr;
    }

    irmask = pucsr->ier;        /* Save UART interrupt state.   */
    pucsr->ier = 0;             /* Disable UART interrupts.     */

    while (0 == (pucsr->lsr & UART_LSR_DR))
    {                           /* Do Nothing */
    }

    /* read character from Receive Holding Register */
    c = pucsr->rbr;

    pucsr->ier = irmask;        /* Restore UART interrupts.     */
    return c;
}

/**
 * kernel printf: formatted, unbuffered output to SERIAL0
 * @param *fmt pointer to string being printed
 * @return OK on success
 */
syscall kprintf(char *fmt, ...)
{
    va_list ap;

    va_start(ap, fmt);
    _doprnt(fmt, ap, (int (*)(int, int))kputc, (int)&devtab[SERIAL0]);
    va_end(ap);
    return OK;
}
