/**
 * @file     startup.S
 * @provides _startup
 * This is where kernel execution begins once loader hands off.  The boot
 * loader flushes caches, copies this kernel's interrupt handler to proper
 * location in memory and hands off control.
 *
 * $Id: startup.S 1577 2008-10-03 18:19:26Z mschul $
 */
/* Embedded Xinu, Copyright (C) 2008.  All rights reserved. */

#include <mips.h>
#include <uart.h>

#define NULLSTK      8192       /* Safe size for NULLSTK           */
#define UART_ADDR    0xB8000300 /* Known address for first UART    */
#define UART_LCR     3          /* Offset of line control register */
#define UART_DLL     0          /* Offset of divisor latch low     */
#define UART_DLM     1          /* Offset of divisor latch hight   */
#define UART_DLL_VAL 11         /* divisor is 11, for a device     */
#define UART_DLM_VAL 0          /*  ... with a 20MHz clock rate    */

.text
	.align 4
	.globl _startup

/**
 * _startup sets up the stack pointer and clears the stack for the
 * null process.  Additionally, the BSS (uninitialized data) section
 * is cleared prior to calling the null process.
 */
 	.ent _startup
_startup:
	/* Set up Stack segment (see function summary)                    */
 	li    s1, NULLSTK     /* Stack is NULLSTK bytes                   */
	la	  a0, _end
 	addu  s1, s1, a0      /* Find top of stack (_end + NULLSTK)       */

	/* Word align the top of the stack                                */
	subu  s1, s1, 1
	srl   s1, 4
	sll   s1, 4
	
	/* Initialize the stack and frame pointers                        */
	move  sp, s1
	move  fp, s1
	
	/* Zero out NULLSTK space below new stack pointer                 */
	subu  a1, s1, a0  
	jal   bzero 
	/* Note: bzero routine must be optimized leaf function without    */
	/* activation record, or this will zero its own return address.   */

	/* Clear the BSS section.                                         */
	la    a0, _bss
	la    a1, _end
	sub   a1, a1, a0
	jal   bzero

	/* Store processor ID                                             */
	mfc0  v0, CP0_PRID
	la    t0, cpuid
	sw    v0, 0(t0)

	/* Store bottom of the heap                                       */
	la    t0, memheap
	sw    s1, 0(t0)

	/* Place serial port in sane state                                */
	li    v0, UART_ADDR     /* address of serial0                     */

	li    a0, UART_LCR_DLAB /* divisor latch access bit               */
	sb    a0, UART_LCR(v0)

	li    a0, UART_DLL_VAL  /* divisor latch low                      */
	sb    a0, UART_DLL(v0)

	li    a0, UART_DLM_VAL  /* divisor latch high                     */
	sb    a0, UART_DLM(v0)

	li    a0, UART_LCR_8N1  /* line control register                  */
	sb    a0, UART_LCR(v0)
	
	j     nulluser          /* start the null user process            */
 	.end _startup

