package edu.ucla.ccb.graphshifts.image;

import java.io.*;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.channels.FileChannel;

/*      An ANALYZE File (Header and Volume) read/write Java object
 * 
 *  Original Source taken from LOVE package by
 *       Ivo D. Dinov, Ph.D. / Roger Woods, M.D. May 08, 2001
 *  Updated for reading and writing by Jason Corso 2006  
 *    Analyze reading and writing.
 *    Only support 8,16 and 32 bit images (16 bit are converted to 32 bit on read)
 *    Also support for floating point pixels is included.
 *  Updated Jason Corso, Jan 8 2006
 *    Reading can now detect Big Endian or Little Endian Image.
 *    Reading now uses the Java NIO classes for faster reading.
 *    
 *    This class links directly to the image classes in my image package
 *      (edu.ucla.ccb.muleseg.image)
 */
public class AnalyzeHeader 
{
	short type=0;
    short bits=0;
    short x_dim=0;
    short y_dim=0;
    short z_dim=0;
    short t_dim=1;
    float x_step=(float)1.0;
    float y_step=(float)1.0;
    float z_step=(float)1.0;
    int glmax=0;
    int glmin=0;
    private byte fullheader[]=new byte[348];
    String HDR_file = null;
    String IMG_file = null;
    Image3PixelAccess image;
    boolean flipOrderOnRead = false;
    
    public AnalyzeHeader(String filename)
    {
    	assignFilenames(filename);
    }
    
    public AnalyzeHeader(String filename, Image3PixelAccess image) 
    {
    	assignFilenames(filename);
    	assignImage(image);
    }

	private void assignFilenames(String filename)
	{
		if ( !filename.endsWith(".hdr") &&
    			!filename.endsWith(".img") &&
    			!filename.endsWith(".LHD") &&
    			!filename.endsWith(".lhd") &&
    			!filename.endsWith(".header")
    	)
    	{   // just the name string
    		HDR_file = filename + ".hdr";
    		IMG_file = filename + ".img";
    	}
    	else if ( filename.endsWith(".img") )
    	{
    		IMG_file = filename;
    		HDR_file = filename.substring(0,filename.length()-4) + ".hdr";
    	}
    	else
    	{
    		HDR_file = filename;
    		IMG_file = filename.substring(0,filename.length()-4) + ".img";
    	}
	}
	
	
	private void assignImage(Image3PixelAccess image)
	{
		this.image = image;
		setXdim((short)image.getWidth());
		setYdim((short)image.getHeight());
		setZdim((short)image.getDepth());
		setGLmax((int)image.getMaxValue());
		setGLmin((int)image.getMinValue());
		setBits(image.getBitsPerPixel());
		
		if (image instanceof ScalarImageF3)
			type = 16;
		else if (bits == 8)
			type = 2;
		else if (bits == 16)
			type = 4;
		else if (bits == 32)
			type = 8;
		else
			type = 0;
		
	}
	
	
	/**
	 * This method will force the format of the image and is mostly useful for image writing.
	 * This coudl cause problems, however, and should be used with care...
	 * @param type
	 * @param bits
	 */
	public void forceType(short type, short bits)
	{
		this.type = type;
		this.bits = bits;
	}
	
	// GET methods
	public short getBits()
	{
		return bits;
	}

	public int getGLmax()
	{
		return glmax;
	}

	public int getGLmin()
	{
		return glmin;
	}

	public Image3PixelAccess getImage()
	{
		return image;
	}

	public short getTdim()
	{
		return t_dim;
	}

	public short getType()
	{
		return type;
	}

	public short getXdim()
	{
		return x_dim;
	}

	public float getXstep()
	{
		return x_step;
	}

	public short getYdim()
	{
		return y_dim;
	}

	public float getYstep()
	{
		return y_step;
	}

	public short getZdim()
	{
		return z_dim;
	}
    
    
    public float getZstep()
	{
		return z_step;
	}
    public Image3PixelAccess loadFromDisk()
	{
		if ( (HDR_file == null) || (IMG_file == null) )
			return null;
		
		readHeader();
		
		try
		{
			if (type == 16)
				readImageFloat();
			else if (bits == 8)
				readImage8Bit();
			else if (bits == 16)
				readImage16Bit();
			else
				readImage32Bit();
		}
		catch (IOException e)
		{
			System.err.println("AnalyzeHeader: error while reading from image data");
		}
		
		return image;
	}
    private void readHeader()
    {	
    	readHeader(HDR_file);   
    }
    private void readHeader(String filename) 
    {
        try {
            FileInputStream fr=new FileInputStream(filename);
            fr.read(fullheader);
            fr.close();
            ByteBuffer hdrbuf = ByteBuffer.wrap(fullheader);
            
            int hdrsize = hdrbuf.getInt();
            if (hdrsize != 348) 
            {
            	flipOrderOnRead = true;
            	hdrbuf.order( (hdrbuf.order() == ByteOrder.BIG_ENDIAN) ? 
            			                       ByteOrder.LITTLE_ENDIAN : ByteOrder.BIG_ENDIAN );
            }
            hdrbuf.position(hdrbuf.position()+38);
            x_dim=hdrbuf.getShort();
            y_dim=hdrbuf.getShort();
            z_dim=hdrbuf.getShort();
            t_dim=hdrbuf.getShort();
            hdrbuf.position(hdrbuf.position()+20);
            type=hdrbuf.getShort();
            bits=hdrbuf.getShort();
            hdrbuf.position(hdrbuf.position()+6);
            x_step=hdrbuf.getFloat();
            y_step=hdrbuf.getFloat();
            z_step=hdrbuf.getFloat();
            hdrbuf.position(hdrbuf.position()+48);
            glmax=hdrbuf.getInt();
            glmin=hdrbuf.getInt();
            
            /*
            DataInputStream fr2=new DataInputStream(
            new ByteArrayInputStream(fullheader));
            
            fr2.skipBytes(42);
            x_dim=fr2.readShort();
            y_dim=fr2.readShort();
            z_dim=fr2.readShort();
            t_dim=fr2.readShort();
            fr2.skipBytes(20);
            type=fr2.readShort();
            bits=fr2.readShort();
            fr2.skipBytes(6);
            x_step=fr2.readFloat();
            y_step=fr2.readFloat();
            z_step=fr2.readFloat();
            fr2.skipBytes(48);
            glmax=fr2.readInt();
            glmin=fr2.readInt();
            fr2.close();
            */
        }
        catch (IOException e) {
            System.out.println("Can't open for "+ "reading header LHD file:" + filename);
            e.printStackTrace();
        }
    }
    private void readImage16Bit() throws IOException
	{
		ByteBuffer buffer = readImageBuffer();
		
		ScalarImageI3 I = new ScalarImageI3(x_dim,y_dim,z_dim);
		I.setRange(glmin,glmax);
		
		for (int z=0;z<z_dim;z++)
			for (int y=0;y<y_dim;y++)
				for (int x=0;x<x_dim;x++)
					I.setPixelAt(x, y, z, 0xFFFF & buffer.getShort());
		            // apply the 0xFFFF here because we need to assume the signed short. 
		            //  and we need to get rid of it.  If there was a ScalarImageS3 class, this
		            //  would not be necessary because that class would take care of the signed-ness
		
		image = I;
	}
    
    private void readImage32Bit() throws IOException
	{
		ByteBuffer buffer = readImageBuffer();
		
		ScalarImageI3 I = new ScalarImageI3(x_dim,y_dim,z_dim);
		I.setRange(glmin,glmax);
		
		for (int z=0;z<z_dim;z++)
			for (int y=0;y<y_dim;y++)
				for (int x=0;x<x_dim;x++)
					I.setPixelAt(x, y, z, buffer.getInt());
		
		image = I;
	}
    
	
    /**
	 * This is the easy case -- just read the buffer in and create the image
	 * @throws IOException
	 */
	private void readImage8Bit() throws IOException
	{
		ByteBuffer bb = readImageBuffer();
		
		int size = x_dim * y_dim * z_dim * bits/8;
		byte[] buffer = new byte[size];
		bb.get(buffer);
		
		ScalarImageB3 I = new ScalarImageB3(x_dim,y_dim,z_dim,buffer);
		I.setMaxValue(glmax);
		I.setMinValue(glmin);
		
		image = I;
	}
    
    
    private ByteBuffer readImageBuffer() throws IOException
	{
		File file = new File(IMG_file);
		FileInputStream fis = new FileInputStream(IMG_file);
		FileChannel chan = fis.getChannel();
		ByteBuffer  buf = chan.map(FileChannel.MapMode.READ_ONLY, 0, (int) file.length());
		
		if (flipOrderOnRead)
		{
			buf.order( (buf.order() == ByteOrder.BIG_ENDIAN) ? 
                                     ByteOrder.LITTLE_ENDIAN : ByteOrder.BIG_ENDIAN );
		}
		
	    return buf;
	}
    private void readImageFloat() throws IOException
	{
		ByteBuffer buffer = readImageBuffer();
		
		ScalarImageF3 I = new ScalarImageF3(x_dim,y_dim,z_dim);
		
		for (int z=0;z<z_dim;z++)
			for (int y=0;y<y_dim;y++)
				for (int x=0;x<x_dim;x++)
					I.setPixelAt(x, y, z, buffer.getFloat());
		
		// sometimes for float images, the range is specified as 0:0
		if (glmin == glmax)
			I.computeMinMax();
		else
			I.setRange(glmin,glmax);
		
		image = I;
	}
    private void saveHeader()
    {	
    	saveHeader(HDR_file);   
   	}
    
    
    private void saveHeader(String filename) {	
        try {
            FileOutputStream fw=new FileOutputStream(filename);
            DataOutputStream fw2=new DataOutputStream(fw);
            
            byte [] skip_bytes = new byte[200];
            for (int i = 0; i < 200; i++)   skip_bytes[i] = 0;
            
            fw2.writeInt(348);
            fw2.write(skip_bytes, 0, 28);
            fw2.writeInt(16384);
            fw2.write(skip_bytes, 0, 2);
            fw2.writeByte('r');
            fw2.write(skip_bytes, 0, 1);
            
            fw2.writeShort(4);
            fw2.writeShort(x_dim);
            fw2.writeShort(y_dim);
            fw2.writeShort(z_dim);
            fw2.writeShort(t_dim);
            
            fw2.write(skip_bytes, 0, 20);
            
            fw2.writeShort(type);
            fw2.writeShort(bits);
            
            fw2.write(skip_bytes, 0, 6);
            
            fw2.writeFloat(x_step);
            fw2.writeFloat(y_step);
            fw2.writeFloat(z_step);
            
            fw2.write(skip_bytes, 0, 48);
            
            fw2.writeInt(glmax);
            fw2.writeInt(glmin);
            
            fw2.write(skip_bytes, 0, 200);
            
            fw2.close();
            fw.close();
        }
        catch (IOException e) {
            System.out.println("(AnalyzeHeader)::Can't open for "+
            "writing ANALYZE *.hdr file:" + filename);
        }
    }
    
    
    private void saveImage16Bit() throws IOException
    {
    	DataOutputStream dos = new DataOutputStream(
    			               new BufferedOutputStream(
    			               new FileOutputStream(IMG_file)));
    	
    	for (int z=0;z<z_dim;z++)
    		for (int y=0;y<y_dim;y++)
    			for (int x=0;x<x_dim;x++)
    				dos.writeShort(image.getPixelInt(x, y, z));
    	
    	dos.close();
    }
    
    private void saveImage32Bit() throws IOException
    {
    	DataOutputStream dos = new DataOutputStream(
    			               new BufferedOutputStream(
    			               new FileOutputStream(IMG_file)));
    	
   		for (int z=0;z<z_dim;z++)
    		for (int y=0;y<y_dim;y++)
    			for (int x=0;x<x_dim;x++)
    				dos.writeInt(image.getPixelInt(x, y, z));
    	
    	dos.close();
    }
    
    private void saveImage8Bit() throws IOException
    {
    	DataOutputStream dos = new DataOutputStream(
    			               new BufferedOutputStream(
    			               new FileOutputStream(IMG_file)));
    	
    	if (image instanceof ScalarImageB3)
    	{
    		ScalarImageB3 B = (ScalarImageB3)image;
    		dos.write(B.getBuffer().getData());
    	}
    	else
    	{
    		for (int z=0;z<z_dim;z++)
    			for (int y=0;y<y_dim;y++)
    				for (int x=0;x<x_dim;x++)
    					dos.writeByte(image.getPixelInt(x, y, z));
    	}
    	
    	dos.close();
    }
    
    private void saveImageFloat() throws IOException
    {
    	DataOutputStream dos = new DataOutputStream(
    			               new BufferedOutputStream(
    			               new FileOutputStream(IMG_file)));
    	
   		for (int z=0;z<z_dim;z++)
    		for (int y=0;y<y_dim;y++)
    			for (int x=0;x<x_dim;x++)
    				dos.writeFloat(image.getPixelFloat(x, y, z));
    	
    	dos.close();
    }
    
    
    public void saveToDisk()
    {
    	if ((image == null) || (HDR_file == null) || (IMG_file == null))
    	{
    		System.err.println("AnalyzeHeader:: not enough data to write");
    		return;
    	}
    	
		
    	saveHeader();
    	
    	try
    	{
    		if (type == 16)
    			saveImageFloat();
    		else if (bits == 8)
    			saveImage8Bit();
    		else if (bits == 16)
    			saveImage16Bit();
    		else if (bits == 32)
    			saveImage32Bit();
    	}
    	catch (IOException e)
    	{
    		System.err.println("AnalyzeHeader:: error while writing image voxel data");
    	}
    	
    }

	// SET methods
	public void setBits(short b)
	{
		bits = b;
	}

	public void setGLmax(int max)
	{
		glmax = max;
	}

	public void setGLmin(int min)
	{
		glmin = min;
	}

	public void setTdim(short t)
	{
		t_dim = t;
	}

	public void setType(short type)
	{
		this.type = type;
	}

	public void setXdim(short x)
	{
		x_dim = x;
	}

	public void setXstep(float x)
	{
		x_step = x;
	}

	public void setYdim(short y)
	{
		y_dim = y;
	}

	public void setYstep(float y)
	{
		y_step = y;
	}

	public void setZdim(short z)
	{
		z_dim = z;
	}

	public void setZstep(float z)
	{
		z_step = z;
	}
    
    
    
    /*
	 * public void writeVolume(byte[][][] voi_volume) // Writes out a BYTE { //
	 * raw MASK volume in Analyze format String IMG_file = new String(
	 * HDR_file.substring(0, HDR_file.length()-4)+ ".img");
	 * System.err.println("(writeVolume):: writing file: "+ IMG_file);
	 * 
	 * try { //FilterOutputStream write_vol; DataOutputStream output_stream;
	 * 
	 * output_stream = new DataOutputStream( new FileOutputStream(IMG_file));
	 * System.err.println("Writing out MASK volume: "); for (int z = 0; z <
	 * voi_volume.length; z++) { System.err.print("."); for (int y = 0; y <
	 * voi_volume[0].length; y++) { output_stream.write(voi_volume[z][y], 0,
	 * voi_volume[0][0].length); } } System.err.println(". Done!");
	 * output_stream.close(); } catch ( FileNotFoundException e) {
	 * System.out.println("(AnalyzeHeader)Can't Read data file: " + IMG_file +
	 * "\n\t Exiting ..."); } catch (IOException e) {
	 * System.out.println("(AnalyzeHeader)Can't READ 4 your BYTE data
	 * entries!!!"); } catch (Exception e) {
	 * System.out.println("(AnalyzeHeader)Can't READ 5 your BYTE data
	 * entries!!!"); System.out.println("(AnalyzeHeader)Exception: " +e ); } }
	 */


}

